#' Calculate Principal Component Scores from Count Tables
#'
#' This function computes selected principal component (PC) scores
#' for each sample in the given count table.
#'
#' By default, the first principal component (PC1) is returned.
#' Users can optionally stabilize the sign of each PC.
#'
#' @param count_table A data frame of gene count data (genes as rows, samples as columns). All columns must be numeric.
#' @param pc Integer vector specifying which principal components to extract (default = 1). Use "all" to return all PCs.
#' @param stabilize_sign Logical. Should the sign of each PC be stabilized? (default = FALSE)
#' @return A data frame with PC scores per sample and the sample IDs.
#' @examples
#'
#' count_table <- data.frame(
#'   sample1 = c(5, 10, 15, 20),
#'   sample2 = c(6, 11, 16, 21),
#'   sample3 = c(7, 12, 17, 22)
#' )
#' rownames(count_table) <- c("gene1", "gene2", "gene3", "gene4")
#'
#' pcscore(count_table)
#'
#' pcscore(count_table, pc = 2)
#'
#' pcscore(count_table, pc = c(1, 2))
#'
#' pcscore(count_table, pc = "all")
#'
#' pcscore(count_table, pc = 1, stabilize_sign = TRUE)
#'
#' @importFrom stats prcomp
#' @export
pcscore <- function(count_table, pc = 1, stabilize_sign = FALSE) {

  # Validation
  if (!is.data.frame(count_table)) {
    stop("count_table should be a data frame.")
  }

  if (!all(sapply(count_table, is.numeric))) {
    stop("All columns in count_table must be numeric.")
  }

  if (!(is.numeric(pc) || identical(pc, "all"))) {
    stop("pc must be a numeric vector or 'all'.")
  }

  if (!is.logical(stabilize_sign) || length(stabilize_sign) != 1) {
    stop("stabilize_sign must be TRUE or FALSE.")
  }

  # Transpose so samples are rows
  transposed <- t(count_table)

  # Remove zero-variance genes
  var_filter <- apply(transposed, 2, stats::var)
  transposed <- transposed[, var_filter > 0, drop = FALSE]

  # Run PCA
  pca_result <- stats::prcomp(transposed, center = TRUE, scale. = TRUE)

  # Determine PCs to extract
  if (identical(pc, "all")) {
    selected_scores <- pca_result$x
  } else {
    max_pc <- ncol(pca_result$x)

    if (any(pc < 1) || any(pc > max_pc)) {
      stop(paste("Requested PCs out of range. Available PCs: 1 to", max_pc))
    }

    selected_scores <- pca_result$x[, pc, drop = FALSE]
  }

  # Stabilize sign if requested
  if (stabilize_sign) {
    for (i in seq_len(ncol(selected_scores))) {
      if (sum(selected_scores[, i]) < 0) {
        selected_scores[, i] <- -selected_scores[, i]
      }
    }
  }

  # Convert to data frame
  score_frame <- as.data.frame(selected_scores)
  score_frame$ID <- rownames(score_frame)

  return(score_frame)
}
