% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim.fossils.R
\name{sim.fossils.intervals}
\alias{sim.fossils.intervals}
\title{Simulate fossils under a non-uniform model of preservation for a given set of consecutive time intervals}
\usage{
sim.fossils.intervals(
  tree = NULL,
  taxonomy = NULL,
  fossils = NULL,
  interval.ages = NULL,
  max.age = NULL,
  strata = NULL,
  probabilities = NULL,
  rates = NULL,
  ignore.taxonomy = FALSE,
  root.edge = TRUE,
  use.exact.times = TRUE
)
}
\arguments{
\item{tree}{Phylo object.}

\item{taxonomy}{Taxonomy object.}

\item{fossils}{Append fossils to to an existing fossils object.}

\item{interval.ages}{Vector of stratigraphic interval ages, starting with the minimum age of the youngest interval and ending with the maximum age of the oldest interval.}

\item{max.age}{Maximum age of the oldest stratigraphic interval.}

\item{strata}{Number of stratigraphic intervals.}

\item{probabilities}{Probability of sampling/preservation in each interval. The number of probabilities should match the number of intervals and the first entry should correspond to youngest interval.}

\item{rates}{Poisson sampling rate for each interval. The number of rates should match the number of intervals and the first entry should correspond to youngest interval.}

\item{ignore.taxonomy}{Ignore species taxonomy (returns sp = NA). Default = FALSE.}

\item{root.edge}{If TRUE include the root edge. Default = TRUE.}

\item{use.exact.times}{If TRUE use exact sampling times. If FALSE \code{hmin} and \code{hmax} will equal the start and end times of the corresponding interval. Default = TRUE.}
}
\value{
An object of class fossils.
}
\description{
Intervals can be specified by specifying the interval boundaries using \code{interval.ages} or specifying both \code{max.age} and \code{strata}.
In the second scenario all intervals will be of equal length.
Preservation can be specified using \code{rates}, which represent the rates of a Poisson process in each interval,
or \code{probabilities}, which represent the probabilities of sampling per interval.
When using \code{probabilities}, at most one fossil per species will be sampled per interval. \cr \cr
Fossils can be simulated for a phylo (\code{tree}) or taxonomy (\code{taxonomy}) object.
If both are specified, the function uses taxonomy.
If no taxonomic information is provided, the function assumes all speciation is symmetric (i.e. bifurcating, \code{beta = 1}).
}
\examples{
# simulate tree
t = ape::rtree(6)

# assign a max age based on tree height
max.age = tree.max(t)

# simulate fossils using max.age and strata & probabilities
strata = 4
probability = rep(0.7, 4)
f = sim.fossils.intervals(t, max.age = max.age, strata = strata, probabilities = probability)
plot(f, t, strata = strata, show.strata = TRUE)

# simulate fossils using interval.ages & rates
times = c(0, sort(runif(3, min = 0, max = max.age)), max.age)
rates = c(5, 0, 5, 0)
f = sim.fossils.intervals(t, interval.ages = times, rates = rates)
plot(f, t, interval.ages = times, show.strata = TRUE)

# simulate fossils using taxonomy
s = sim.taxonomy(t, 0.1, 0.1, 1)
f = sim.fossils.intervals(taxonomy = s, interval.ages = times, rates = rates)
plot(f, t, interval.ages = times, show.strata = TRUE)

# append fossils to an existing fossils object
new.rates = rates * 2
f2 = sim.fossils.intervals(taxonomy = s, fossils = f, interval.ages = times, rates = new.rates)

}
\seealso{
\code{\link{sim.fossils.poisson}}, \code{\link{sim.fossils.environment}}
}
\keyword{fossil}
\keyword{non-uniform}
\keyword{preservation}
\keyword{uniform}
