#' Find outliers by three methods
#'
#' Returns an index into outlying points, if any, identified by one of
#' three outlier detection rules: the three-sigma edit rule, the Hampel
#' identifier, or the boxplot rule
#'
#' @inheritParams ComputeOutlierLimits
#'
#' @return index into elements of `x` identified as outliers
#' @export
#'
#' @examples
#' x <- seq(-1, 1, length = 100)
#' x[1:10] <- 10
#' Tindex <- FindOutliers(x, "T")
#' x[Tindex]  # Example where the three-sigma rule fails
#' Hindex <- FindOutliers(x, "H")
#' x[Hindex]
#' Bindex <- FindOutliers(x, "B")
#' x[Bindex]
FindOutliers <- function(x, method, t = NULL){
  #
  stopifnot("x must be numeric"= is.numeric(x))
  validMethods <- c("T", "H", "B")
  stopifnot("Invalid method argument"= method %in% validMethods)
  stopifnot("t must be NULL or non-negative"= (is.null(t) | t >= 0))
  #
  outLims <- ComputeOutlierLimits(x, method, t)
  #
  if (outLims["nonMiss"] == 0){
    indexOut <- integer(0)
  } else {
    indexUp <- which(x > outLims["upLim"])
    indexLo <- which(x < outLims["loLim"])
    indexOut <- union(indexUp, indexLo)
  }
  return(indexOut)
}

