






use super::Dimension;
#[cfg(feature = "std")]
use std::error::Error;
use std::fmt;


#[derive(Clone)]
pub struct ShapeError {

    repr: ErrorKind,
}

impl ShapeError {

    #[inline]
    pub fn kind(&self) -> ErrorKind {
        self.repr
    }


    pub fn from_kind(error: ErrorKind) -> Self {
        from_kind(error)
    }
}





#[non_exhaustive]
#[derive(Copy, Clone, Debug)]
pub enum ErrorKind {

    IncompatibleShape = 1,

    IncompatibleLayout,

    RangeLimited,

    OutOfBounds,

    Unsupported,

    Overflow,
}

#[inline(always)]
pub fn from_kind(k: ErrorKind) -> ShapeError {
    ShapeError { repr: k }
}

impl PartialEq for ErrorKind {
    #[inline(always)]
    fn eq(&self, rhs: &Self) -> bool {
        *self as u8 == *rhs as u8
    }
}

impl PartialEq for ShapeError {
    #[inline(always)]
    fn eq(&self, rhs: &Self) -> bool {
        self.repr == rhs.repr
    }
}

#[cfg(feature = "std")]
impl Error for ShapeError {}

impl fmt::Display for ShapeError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let description = match self.kind() {
            ErrorKind::IncompatibleShape => "incompatible shapes",
            ErrorKind::IncompatibleLayout => "incompatible memory layout",
            ErrorKind::RangeLimited => "the shape does not fit in type limits",
            ErrorKind::OutOfBounds => "out of bounds indexing",
            ErrorKind::Unsupported => "unsupported operation",
            ErrorKind::Overflow => "arithmetic overflow",
        };
        write!(f, "ShapeError/{:?}: {}", self.kind(), description)
    }
}

impl fmt::Debug for ShapeError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}", self)
    }
}

pub fn incompatible_shapes<D, E>(_a: &D, _b: &E) -> ShapeError
where
    D: Dimension,
    E: Dimension,
{
    from_kind(ErrorKind::IncompatibleShape)
}
