% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prob-gain_capture.R
\name{gain_capture}
\alias{gain_capture}
\alias{gain_capture.data.frame}
\alias{gain_capture_vec}
\title{Gain capture}
\usage{
gain_capture(data, ...)

\method{gain_capture}{data.frame}(
  data,
  truth,
  ...,
  estimator = NULL,
  na_rm = TRUE,
  event_level = yardstick_event_level(),
  case_weights = NULL
)

gain_capture_vec(
  truth,
  estimate,
  estimator = NULL,
  na_rm = TRUE,
  event_level = yardstick_event_level(),
  case_weights = NULL,
  ...
)
}
\arguments{
\item{data}{A \code{data.frame} containing the columns specified by \code{truth} and
\code{...}.}

\item{...}{A set of unquoted column names or one or more
\code{dplyr} selector functions to choose which variables contain the
class probabilities. If \code{truth} is binary, only 1 column should be selected,
and it should correspond to the value of \code{event_level}. Otherwise, there
should be as many columns as factor levels of \code{truth} and the ordering of
the columns should be the same as the factor levels of \code{truth}.}

\item{truth}{The column identifier for the true class results
(that is a \code{factor}). This should be an unquoted column name although
this argument is passed by expression and supports
\link[rlang:topic-inject]{quasiquotation} (you can unquote column
names). For \verb{_vec()} functions, a \code{factor} vector.}

\item{estimator}{One of \code{"binary"}, \code{"macro"}, or \code{"macro_weighted"} to
specify the type of averaging to be done. \code{"binary"} is only relevant for
the two class case. The other two are general methods for calculating
multiclass metrics. The default will automatically choose \code{"binary"} or
\code{"macro"} based on \code{truth}.}

\item{na_rm}{A \code{logical} value indicating whether \code{NA}
values should be stripped before the computation proceeds.}

\item{event_level}{A single string. Either \code{"first"} or \code{"second"} to specify
which level of \code{truth} to consider as the "event". This argument is only
applicable when \code{estimator = "binary"}. The default uses an
internal helper that generally defaults to \code{"first"}, however, if the
deprecated global option \code{yardstick.event_first} is set, that will be
used instead with a warning.}

\item{case_weights}{The optional column identifier for case weights.
This should be an unquoted column name that evaluates to a numeric column
in \code{data}. For \verb{_vec()} functions, a numeric vector.}

\item{estimate}{If \code{truth} is binary, a numeric vector of class probabilities
corresponding to the "relevant" class. Otherwise, a matrix with as many
columns as factor levels of \code{truth}. \emph{It is assumed that these are in the
same order as the levels of \code{truth}.}}
}
\value{
A \code{tibble} with columns \code{.metric}, \code{.estimator},
and \code{.estimate} and 1 row of values.

For grouped data frames, the number of rows returned will be the same as
the number of groups.

For \code{gain_capture_vec()}, a single \code{numeric} value (or \code{NA}).
}
\description{
\code{gain_capture()} is a measure of performance similar to an AUC calculation,
but applied to a gain curve.
}
\details{
\code{gain_capture()} calculates the area \emph{under} the gain curve, but \emph{above}
the baseline, and then divides that by the area \emph{under} a perfect gain curve,
but \emph{above} the baseline. It is meant to represent the amount of potential
gain "captured" by the model.

The \code{gain_capture()} metric is identical to the \emph{accuracy ratio (AR)}, which
is also sometimes called the \emph{gini coefficient}. These two are generally
calculated on a cumulative accuracy profile curve, but this is the same as
a gain curve. See the Engelmann reference for more information.
}
\section{Relevant Level}{


There is no common convention on which factor level should
automatically be considered the "event" or "positive" result
when computing binary classification metrics. In \code{yardstick}, the default
is to use the \emph{first} level. To alter this, change the argument
\code{event_level} to \code{"second"} to consider the \emph{last} level of the factor the
level of interest. For multiclass extensions involving one-vs-all
comparisons (such as macro averaging), this option is ignored and
the "one" level is always the relevant result.
}

\section{Multiclass}{


Macro and macro-weighted averaging is available for this metric.
The default is to select macro averaging if a \code{truth} factor with more
than 2 levels is provided. Otherwise, a standard binary calculation is done.
See \code{vignette("multiclass", "yardstick")} for more information.
}

\examples{
# ---------------------------------------------------------------------------
# Two class example

# `truth` is a 2 level factor. The first level is `"Class1"`, which is the
# "event of interest" by default in yardstick. See the Relevant Level
# section above.
data(two_class_example)

# Binary metrics using class probabilities take a factor `truth` column,
# and a single class probability column containing the probabilities of
# the event of interest. Here, since `"Class1"` is the first level of
# `"truth"`, it is the event of interest and we pass in probabilities for it.
gain_capture(two_class_example, truth, Class1)

# ---------------------------------------------------------------------------
# Multiclass example

# `obs` is a 4 level factor. The first level is `"VF"`, which is the
# "event of interest" by default in yardstick. See the Relevant Level
# section above.
data(hpc_cv)

# You can use the col1:colN tidyselect syntax
library(dplyr)
hpc_cv \%>\%
  filter(Resample == "Fold01") \%>\%
  gain_capture(obs, VF:L)

# Change the first level of `obs` from `"VF"` to `"M"` to alter the
# event of interest. The class probability columns should be supplied
# in the same order as the levels.
hpc_cv \%>\%
  filter(Resample == "Fold01") \%>\%
  mutate(obs = relevel(obs, "M")) \%>\%
  gain_capture(obs, M, VF:L)

# Groups are respected
hpc_cv \%>\%
  group_by(Resample) \%>\%
  gain_capture(obs, VF:L)

# Weighted macro averaging
hpc_cv \%>\%
  group_by(Resample) \%>\%
  gain_capture(obs, VF:L, estimator = "macro_weighted")

# Vector version
# Supply a matrix of class probabilities
fold1 <- hpc_cv \%>\%
  filter(Resample == "Fold01")

gain_capture_vec(
   truth = fold1$obs,
   matrix(
     c(fold1$VF, fold1$F, fold1$M, fold1$L),
     ncol = 4
   )
)

# ---------------------------------------------------------------------------
# Visualize gain_capture()

# Visually, this represents the area under the black curve, but above the
# 45 degree line, divided by the area of the shaded triangle.
library(ggplot2)
autoplot(gain_curve(two_class_example, truth, Class1))

}
\references{
Engelmann, Bernd & Hayden, Evelyn & Tasche, Dirk (2003).
"Measuring the Discriminative Power of Rating Systems,"
Discussion Paper Series 2: Banking and Financial Studies 2003,01,
Deutsche Bundesbank.
}
\seealso{
\code{\link[=gain_curve]{gain_curve()}} to compute the full gain curve.

Other class probability metrics: 
\code{\link{average_precision}()},
\code{\link{brier_class}()},
\code{\link{classification_cost}()},
\code{\link{mn_log_loss}()},
\code{\link{pr_auc}()},
\code{\link{roc_auc}()},
\code{\link{roc_aunp}()},
\code{\link{roc_aunu}()}
}
\author{
Max Kuhn
}
\concept{class probability metrics}
