% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/module5_mst.R
\name{mst}
\alias{mst}
\alias{mst_assemble}
\alias{mst_constraint}
\alias{mst_get_items}
\alias{mst_module_mininfo}
\alias{mst_objective}
\alias{mst_route}
\alias{mst_set_rdp}
\alias{mst_stage_length}
\alias{plot.mst}
\alias{print.mst}
\title{Computerized Multistage Testing (MST)}
\usage{
mst(pool, design, npanel, method, len = NULL, maxselect = NULL)

mst_route(x, route, op)

mst_objective(x, theta, indices = NULL, target = NULL, flatten = NULL,
  theta.step = 2)

mst_constraint(x, coef, min = NA, max = NA, level = NULL,
  indices = NULL)

mst_stage_length(x, stages, min = NA, max = NA)

mst_set_rdp(x, theta, indices, tol)

mst_module_mininfo(x, theta, mininfo, indices)

mst_assemble(x, solver = "lpsolve", verbose = "none", ...)

\method{print}{mst}(x, ...)

\method{plot}{mst}(x, ...)

mst_get_items(x, panel, stage = NULL, module = NULL, route = NULL)
}
\arguments{
\item{pool}{a data frame of item parameters}

\item{design}{a numeric vector of the MST design (e.g., 1-2-3, or 1-2-2)}

\item{npanel}{the number of panels}

\item{method}{the design method (i.e., 'topdown' or 'bottomup')}

\item{len}{the module/route length}

\item{maxselect}{the maximum selection of items}

\item{x}{the mst object}

\item{route}{a vector of form indices to represent the route}

\item{op}{\code{"+"} for adding a route and \code{"-"} for removing a route}

\item{theta}{a theta point or interval for optimization}

\item{indices}{the index of the route (topdown) or the module (bottomup) for adding objectives}

\item{target}{the target valus of the TIF objectives. \code{NULL} for maximization}

\item{flatten}{the parameter for getting a flat TIF}

\item{theta.step}{the step parameters to process \code{theta} interval}

\item{coef}{the coefficients of the constraint}

\item{min}{the minimum value of the contraint}

\item{max}{the maximum value of the contraint}

\item{level}{the level value for a categorical constraint}

\item{stages}{the stage index}

\item{tol}{tolerance parameter}

\item{mininfo}{the minimum information treshold}

\item{solver}{the ata solver}

\item{verbose}{the verbose parameter}

\item{...}{further arguments}

\item{panel}{the panel index}

\item{stage}{the stage index}

\item{module}{the module index}
}
\description{
\code{mst} creates a multistage (MST) object

\code{mst_route} adds and removes routes in the mst

\code{mst_objective} adds objective functions to the mst

\code{mst_constraint} adds constraints to the mst

\code{mst_stage_length} sets the minimum and maximum length for a stage

\code{mst_set_rdp} anchors the routing decision point (rdp) between adjacent modules

\code{mst_module_mininfo} sets the minimum information for modules

\code{mst_assemble} assembles the mst

\code{mst_get_items} extracts items from results
}
\details{
The \code{mst} object contains an item pool (\code{pool}), a test assembler (\code{assembler}), 
a route map (\code{route}), a stage-module map (\code{module}), a design method (\code{method}),
and several constants such as \code{npanel}, \code{nstage}, \code{nmodule}, \code{nroute}.\cr
Two indices are used to index modules/testlets. \code{form} number is a unique identifier for all modules,
which is used for automated test assembly. \code{index} is a panel-wise unique identifier, which is used for
routing. \cr

There are two design methods for mst: \code{'bottomup'} and \code{'topdown'}. In the bottomup approach, 
constraitns are imposed on individual modules. Conversely, in the topdown approach, constraints are imposed on routes.\cr

\code{plot.mst} draws module information functions if \code{byroute=FALSE}
and route information functions if \code{byroute=TRUE}
}
\examples{
\dontrun{
# generate a 300-item pool
pool <- irt_model("3pl")$gendata(1,300)$items
pool$content <- sample(1:3, nrow(pool), replace=TRUE)
pool$time <- round(exp(rnorm(nrow(pool), log(60), .2)))

## ex. 1: 1-2-2 MST, 2 panels, topdown
## 20 items, content = c(10, 5, 5)
## maximize information at -1 and 1 for easy and hard routes
x <- mst(pool, design=c(1, 2, 2), npanel=2, method='topdown', len=20, maxselect=1)
x <- mst_objective(x, theta=-1, indices=1:2)
x <- mst_objective(x, theta= 1, indices=3:4)
x <- mst_constraint(x, coef="content", min=10, max=10, level=1)
x <- mst_constraint(x, coef="content", min=5, max=5, level=2)
x <- mst_constraint(x, coef="content", min=5, max=5, level=3)
x <- mst_stage_length(x, stages=c(1, 2, 3), min=1)
x <- mst_assemble(x, timeout=10)
x$items
plot(x)
plot(x, byroute=TRUE)
freq(mst_get_items(x, panel=1, route=1)$content, 1:3)$freq
freq(mst_get_items(x, panel=2, route=4)$content, 1:3)$freq

## ex. 2: 1-2-3 MST, 2 panels, bottomup, 
## 10 items in each module, content = c(4, 3, 3)
## maximize information at -1, 0 and 1 for easy, medium, and hard modules
x <- mst(pool, design=c(1, 2, 3), npanel=2, method='bottomup', len=10, maxselect=1)
x <- mst_route(x, c(1, 2, 6), "-")
x <- mst_route(x, c(1, 3, 4), "-")
x <- mst_objective(x, theta= 0, indices=1)
x <- mst_objective(x, theta=-1, indices=c(2,4))
x <- mst_objective(x, theta= 1, indices=c(3,5))
x <- mst_constraint(x, coef="content", min=4, max=4, level=1)
x <- mst_constraint(x, coef="content", min=3, max=3, level=2)
x <- mst_constraint(x, coef="content", min=3, max=3, level=3)
x <- mst_assemble(x, timeout=10)
plot(x)
plot(x, byroute=TRUE)
freq(mst_get_items(x, panel=1, route=1)$content, 1:3)$freq
freq(mst_get_items(x, panel=2, route=4)$content, 1:3)$freq

## ex. 3: 1-2-3 MST, 2 panels, topdown, 30 items, 
## 10 items in each content area
## target information at 18 at -1, 0, and 1 for easy, medium and hard routes
x <- mst(pool, design=c(1, 2, 3), npanel=2, method='topdown', len=30, maxselect=1)
x <- mst_route(x, c(1, 2, 6), "-")
x <- mst_route(x, c(1, 3, 4), "-")
x <- mst_objective(x, theta=-1, indices=1, target=16)
x <- mst_objective(x, theta= 0, indices=2:3, target=16)
x <- mst_objective(x, theta= 1, indices=4, target=16)
x <- mst_constraint(x, coef="content", min=10, max=10, level=1)
x <- mst_constraint(x, coef="content", min=10, max=10, level=2)
x <- mst_constraint(x, coef="content", min=10, max=10, level=3)
x <- mst_stage_length(x, stages=c(1, 2, 3), min=3)
x <- mst_assemble(x, timeout=20)
plot(x)
plot(x, byroute=TRUE)
freq(mst_get_items(x, panel=1, route=1)$content, 1:3)$freq
freq(mst_get_items(x, panel=2, route=4)$content, 1:3)$freq
}
}

