#' Get summary information on a particular bill
#'
#' Get legislative summary information for a particular bill,
#'     including bill ID, introduction date, bill title(s), and
#'     description
#'
#' @param biennium Character vector representing the biennium(s) to be
#'      searched. Each argument should take the form "XXXX-YY"
#' @param billNumber Character or numeric vector containing the bill number(s)
#'      to be retrieved.
#' @param paired If TRUE, will assume that equal length vectors represent
#'      paired data. Set to FALSE to generate an NxN grid of input
#'      arguments. Applies to equal length vector inputs only.
#' @param type One of "df", "list", or "xml". Specifies the format for
#'      the output.
#'
#' @return \code{getLegislation} returns an object of type equal to the
#'     \code{type} argument (defaults to dataframe)
#' @export
#'
#' @examples
#' getLegislation("2007-08", "1001")
#'
#' ## get XML data for the first 100 hundred bills of the 2007-08 session
#' \dontrun{getLegislation("2007-08", 1001:1100, type = "xml")}
getLegislation <- function(biennium, billNumber, paired = TRUE, type = c("df", "list", "xml")) {
  type <- rlang::arg_match(type)
  billNumber <- as.character(billNumber)

  if(!all(grepl(biennium_pattern, biennium))) {
    stop("Biennium formatted incorrectly. Use ?getLegislation for more information")
  } else if(!all(as.numeric(substr(biennium,1,4)) >= 1991)) {
    stop("Biennium out of range. Information is available going back to 1991-92")
  } else if(!all(grepl(billNum_pattern, billNumber))) {
    stop("Bill Number formatted incorrectly. Use ?getLegislation for more information")
  }

  if(length(biennium) == length(billNumber) & paired) {
    request <- data.frame(biennium = biennium, billId = billNumber)
  } else {
    request <- expand.grid(biennium, billNumber, KEEP.OUT.ATTRS = FALSE, stringsAsFactors = FALSE)
  }

  if(type == "df") {
    out <- data.frame()

    for(bill in 1:nrow(request)) {
      path <- paste(prefix, "legislationservice.asmx/GetLegislation?biennium=",request[bill,1],
                    "&billNumber=", gsub(" ", "%20", request[bill,2]), sep = "")

      tbl <- tryCatch(XML::xmlParse(path),
                      error = function(e){
                        e$message <- errMessage
                        stop(e)
                      })

      tbl <- XML::xmlToDataFrame(tbl,
                                 stringsAsFactors = FALSE)
      if(nrow(tbl) > 0) {
        tbl$Biennium = request[bill,1]
        tbl$BillNumber = request[bill,2]
        tbl <- tbl[c("Biennium", "BillNumber",
                     setdiff(names(tbl), c("Biennium", "BillNumber")))]
        out <- dplyr::bind_rows(out, tbl)
        out <- out[!duplicated(out),]
      }
    }
  } else if(type == "list") {
    out <- list()

    for(bill in 1:nrow(request)) {
      path <- paste(prefix, "legislationservice.asmx/GetLegislation?biennium=",request[bill,1],
                    "&billNumber=", gsub(" ", "%20", request[bill,2]), sep = "")

      tbl <- tryCatch(XML::xmlParse(path),
                      error = function(e){
                        e$message <- errMessage
                        stop(e)
                      })

      tbl <- XML::xmlToList(tbl)
      list <- list(tbl)
      names(list) <- request[bill,2]
      if(length(tbl) > 0) {
        out <- c(out, list)
      }
    }
  } else if(type == "xml") {
    out <- c()

    for(bill in 1:nrow(request)) {
      path <- paste(prefix, "legislationservice.asmx/GetLegislation?biennium=",request[bill,1],
                    "&billNumber=", gsub(" ", "%20", request[bill,2]), sep = "")

      tbl <- tryCatch(XML::xmlParse(path),
                      error = function(e){
                        e$message <- errMessage
                        stop(e)
                      })

      out <- c(out, tbl)
    }
    names(out) <- paste(request[,1], request[,2], sep = "//")
  }
  return(out)
}
