#' Visual Studio Project Release Definition Information
#'
#' @description
#' These functions will allow you to scrape release definition information from Visual Studio.
#'
#' @details
#' For more information about release definition API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/vsts/release/definitions}.
#'
#' @param domain the location of the visual studio server
#' @param project the name of the project in \code{domain} to look at
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param quiet logical whether want general running information from printing. Any
#' issue with the API call will still show up if set to \code{TRUE}
#'
#' @examples
#' #Add in own details to get a non-NULL output
#' auth_key <- vsts_auth_key('<username>', '<password>')
#' vsts_get_release_defs('domain', 'project', auth_key)
#'
#' @rdname vsts_release_def
#' @export
vsts_get_release_defs <- function(domain, project, auth_key, quiet = FALSE) {
  uri <- paste0('https://', domain, '.vsrm.visualstudio.com/', project, '/_apis/release/definitions')

  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key))
  if(response$status_code != 200) {
    cat(httr::http_condition(response, 'message', 'get release definition list')$message, '\n')
    return(invisible(NULL))
  }

  content <- httr::content(response, as = 'text', encoding = 'UTF-8') %>% jsonlite::fromJSON(., flatten = TRUE) %>% .$value
  if(!quiet) cat('Available release definitions:', paste(content$name, collapse = ', '), '\n')
  return(invisible(content))
}

#' Visual Studio Project Release Information
#'
#' @description
#' These functions will allow you to create releases from Visual Studio.
#'
#' @details
#' The \code{artifacts} object within the body contains two items:
#' \itemize{
#' \item{alias}{[character] Sets alias of artifact.}
#' \item{instanceReference}{[list] Sets instance reference of artifact. e.g. for build artifact it is build number.}
#' }
#'
#' For more information about release  API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/vsts/release/releases}.
#'
#' @param domain the location of the visual studio server
#' @param project the name of the project in \code{domain} to look at
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param body a list of extra parameters that can need to be sent to the API call (* mandatory):
#' \describe{
#' \item{\code{artifacts *}}{[list] Sets list of artifact to create a release. Check \code{Details} for more information.}
#' \item{\code{definitionId *}}{[integer] Sets definition Id to create a release.}
#' \item{\code{description *}}{[character] Sets description to create a release.}
#' \item{\code{isDraft}}{[logical] Sets 'true' to create release in draft mode, 'false' otherwise.}
#' \item{\code{manualEnvironments}}{[character] Sets list of environments to manual as condition.}
#' \item{\code{properties}}{[list] The class represents a property bag as a collection of key-value pairs.}
#' \item{\code{reason}}{[character] Sets reason to create a release.}
#' }
#'
#' @examples
#' #Add in own details to get a non-NULL output
#' auth_key <- vsts_auth_key('<username>', '<password>')
#' art_list <- list(list(alias = 'Art1', instanceReference = list(id = 1)),
#'                  list(alias = 'Art2', instanceReference = list(id = 2)))
#' body <- list(definitionId = 1, description = 'R API Release',
#'              artifacts = I(art_list))
#' vsts_create_release('domain', 'project', auth_key, body)
#'
#' @export
vsts_create_release <- function(domain, project, auth_key, body) {
  uri <- paste0('https://', domain, '.vsrm.visualstudio.com/', project, '/_apis/release/releases?api-version=4.1-preview.6')

  content_body <- jsonlite::toJSON(body, auto_unbox = TRUE)

  response <- httr::POST(uri, httr::add_headers(Authorization = auth_key),
                         httr::content_type('application/json'), body = content_body)
  if(httr::status_code(response) != 200) {
    cat(httr::http_condition(
      response, 'message', paste0('create release definition #', body$definitionId))$message, '\n')
    return(invisible(NULL))
  }

  content <- httr::content(response, as = 'text', encoding = 'UTF-8') %>% jsonlite::fromJSON(.)
  invisible(content)
}

#' Visual Studio Project Release Information
#'
#' @description
#' These functions will allow you to scrape releases from Visual Studio.
#'
#' @details
#' For more information about release API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/vsts/release/releases}.
#'
#' @param domain the location of the visual studio server
#' @param project the name of the project in \code{domain} to look at
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param query a list of extra parameters that can be sent to the API call:
#' \describe{
#' \item{\code{propertyFilters}}{[character] A comma-delimited list of extended properties to retrieve.}
#' \item{\code{tagFilter}}{[character] A comma-delimited list of tags. Only releases with these tags will be returned.}
#' \item{\code{isDeleted}}{[logical] Gets the soft deleted releases, if true.}
#' \item{\code{sourceBranchFilter}}{[character] Releases with given sourceBranchFilter will be returned.}
#' \item{\code{artifactVersionId}}{[character] Releases with given artifactVersionId will be returned.
#' E.g. in case of Build artifactType, it is buildId.}
#' \item{\code{sourceId}}{[character] Unique identifier of the artifact used. e.g. For build it would be
#' {projectGuid}:{BuildDefinitionId}, for Jenkins it would be {JenkinsConnectionId}:{JenkinsDefinitionId},
#' for TfsOnPrem it would be {TfsOnPremConnectionId}:{ProjectName}:{TfsOnPremDefinitionId}.
#' For third-party artifacts e.g. TeamCity, BitBucket you may refer 'uniqueSourceIdentifier'
#' inside vss-extension.json \url{https://github.com/Microsoft/vsts-rm-extensions/blob/master/Extensions}.}
#' \item{\code{artifactTypeId}}{[character] Releases with given artifactTypeId will be returned.
#' Values can be Build, Jenkins, GitHub, Nuget, Team Build (external), ExternalTFSBuild, Git, TFVC, ExternalTfsXamlBuild.}
#' \item{\code{$expand}}{[character] The property that should be expanded in the list of releases.}
#' \item{\code{continuationToken}}{[integer] Gets the releases after the continuation token provided.}
#' \item{\code{$top}}{[integer] Number of releases to get. Default is 50.}
#' \item{\code{queryOrder}}{[character] Gets the results in the defined order of created date for releases.
#' Default is descending.}
#' \item{\code{maxCreatedTime}}{[Date] Releases that were created before this time.}
#' \item{\code{minCreatedTime}}{[Date] Releases that were created after this time.}
#' \item{\code{statusFilter}}{[character] Releases that have this status.}
#' \item{\code{createdBy}}{[character] Releases created by this user.}
#' \item{\code{searchText}}{[character] Releases with names starting with searchText.}
#' \item{\code{definitionEnvironmentId}}{[integer] Releases from this release environment Id.}
#' \item{\code{definitionId}}{[integer] Releases from this release definition Id.}
#' \item{\code{releaseIdFilter}}{[character] A comma-delimited list of releases Ids. Only releases with these
#' Ids will be returned.}
#' }
#'
#' @examples
#' #Add in own details to get a non-NULL output
#' auth_key <- vsts_auth_key('<username>', '<password>')
#' vsts_get_releases('domain', 'project', auth_key)
#'
#' @rdname vsts_get_release
#' @export
vsts_get_releases <- function(domain, project, auth_key, query = NULL) {
  uri <- paste0('https://', domain, '.vsrm.visualstudio.com/', project, '/_apis/release/releases?api-version=4.1-preview.6')

  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key), query = query)
  if(httr::status_code(response) != 200) {
    cat(httr::http_condition(response, 'message', 'get release definitions')$message, '\n')
    return(invisible(NULL))
  }

  content <- httr::content(response, as = 'text', encoding = 'UTF-8') %>% jsonlite::fromJSON(., flatten = TRUE) %>% .$value
  invisible(content)
}

#' @param release Release Definition ID
#'
#' @rdname vsts_get_release
#' @export
vsts_get_release <- function(domain, project, release, auth_key) {
  uri <- paste0('https://', domain, '.vsrm.visualstudio.com/', project, '/_apis/release/releases/', release,
                '?api-version=4.1-preview.6')

  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key), httr::content_type('application/json'))
  if(httr::status_code(response) != 200) {
    cat(httr::http_condition(response, 'message', 'Get release definitions')$message, '\n')
    return(invisible(NULL))
  }

  content <- httr::content(response, as = 'text', encoding = 'UTF-8') %>% jsonlite::fromJSON(.)
  invisible(content)
}

#' Visual Studio Project Release Environment Information
#'
#' @description
#' These functions will allow you to run release environment tasks from Visual Studio.
#'
#' @details
#' For more information about release environment API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/vsts/release/releases/update\%20release\%20environment}.
#'
#' @param domain the location of the visual studio server
#' @param project the name of the project in \code{domain} to look at
#' @param release the release ID of the release
#' @param env the release environment ID to release on
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#'
#' @examples
#' #Add in own details to get a non-NULL output
#' auth_key <- vsts_auth_key('<username>', '<password>')
#' vsts_deploy_release('domain', 'project', auth_key, 1, 1)
#'
#' @rdname vsts_release_env
#' @export
vsts_deploy_release <- function(domain, project, release, env, auth_key) {
  uri <- paste0('https://', domain, '.vsrm.visualstudio.com/', project, '/_apis/Release/releases/', release,
                '/environments/', env, '?api-version=4.1-preview.5')

  content_body <- jsonlite::toJSON(list(status = 'inProgress'), auto_unbox = TRUE)

  response <- httr::PATCH(uri, httr::add_headers(Authorization = auth_key),
                          httr::content_type('application/json'), body = content_body)
  if(httr::status_code(response) != 200) {
    cat(httr::http_condition(response, 'message', httr::content(response)$message)$message, '\n')
    return(invisible(NULL))
  }

  cat('Deployment of release has started.\n')
  content <- httr::content(response, as = 'text', encoding = 'UTF-8') %>% jsonlite::fromJSON(., flatten = TRUE)
  invisible(content)
}
