% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vmgeom.R
\name{vmgeom}
\alias{vmgeom}
\alias{dvmgeom}
\alias{pvmgeom}
\alias{qvmgeom}
\alias{rvmgeom}
\title{Geometric Model of Visual Meteor Magnitudes}
\usage{
dvmgeom(m, lm, r, log = FALSE, perception.fun = NULL)

pvmgeom(m, lm, r, lower.tail = TRUE, log = FALSE, perception.fun = NULL)

qvmgeom(p, lm, r, lower.tail = TRUE, perception.fun = NULL)

rvmgeom(n, lm, r, perception.fun = NULL)
}
\arguments{
\item{m}{numeric; the meteor magnitude.}

\item{lm}{numeric; limiting magnitude.}

\item{r}{numeric; the population index.}

\item{log}{logical; if \code{TRUE}, probabilities p are given as \code{log(p)}.}

\item{perception.fun}{function; perception probability function (optional).
Default is \link{vmperception}.}

\item{lower.tail}{logical; if \code{TRUE} (default) probabilities are
\eqn{P[M < m]}, otherwise, \eqn{P[M \ge m]}.}

\item{p}{numeric; probability.}

\item{n}{numeric; count of meteor magnitudes.}
}
\value{
\itemize{
\item \code{dvmgeom}: density
\item \code{pvmgeom}: distribution function
\item \code{qvmgeom}: quantile function
\item \code{rvmgeom}: random generation
}

The length of the result is determined by \code{n} for \code{rvmgeom}, and by the maximum
of the lengths of the numeric vector arguments for the other functions.

Since the distribution is discrete, \code{qvmgeom} and \code{rvmgeom} always return integer values.
\code{qvmgeom} may return \code{NaN} with a warning.
}
\description{
Density, distribution function, quantile function, and random generation
for the geometric model of visual meteor magnitudes.
}
\details{
In visual meteor observations, magnitudes are usually estimated as integer values.
Hence, this distribution is discrete and its probability mass function is
\deqn{
    {\displaystyle P[X = x] \sim f(x) \, \mathrm r^{-x}} \,\mathrm{,}
}
where \eqn{x \ge -0.5} denotes the difference between the limiting magnitude \code{lm}
and the meteor magnitude \code{m}, and \eqn{f(x)} is the perception probability function.
Thus, the distribution is the product of the
\link[=vmperception]{perception probabilities} and the
underlying \link[stats:Geometric]{geometric distribution} of meteor magnitudes.
Therefore, the parameter \code{p} of the geometric distribution is given by \code{p = 1 - 1/r}.

The parameter \code{lm} specifies the reference for the meteor magnitude \code{m}.
\code{m} must be an integer meteor magnitude.
The length of the vector \code{lm} must either equal the length of the vector \code{m},
or \code{lm} must be a scalar value.
In the case of \code{rvmgeom}, the length of the vector \code{lm} must equal \code{n},
or \code{lm} must be a scalar value.

If a perception probability function \code{perception.fun} is provided,
it must have the signature \verb{function(x)} and return the perception probability of
the difference \code{x} between the limiting magnitude and the meteor magnitude.
If \code{x >= 15.0}, the function \code{perception.fun} should return a perception probability of \code{1.0}.
If \code{log = TRUE} is specified, the logarithm of the perception probabilities
must be returned.
The argument \code{perception.fun} is resolved using \link{match.fun}.
}
\examples{
N <- 100
r <- 2.0
limmag <- 6.5
(m <- seq(6, -7))

# discrete density of `N` meteor magnitudes
(freq <- round(N * dvmgeom(m, limmag, r)))

# log likelihood function
lld <- function(r) {
    -sum(freq * dvmgeom(m, limmag, r, log=TRUE))
}

# maximum likelihood estimation (MLE) of r
est <- optim(2, lld, method='Brent', lower=1.1, upper=4)

# estimations
est$par # mean of r

# generate random meteor magnitudes
m <- rvmgeom(N, r, lm=limmag)

# log likelihood function
llr <- function(r) {
    -sum(dvmgeom(m, limmag, r, log=TRUE))
}

# maximum likelihood estimation (MLE) of r
est <- optim(2, llr, method='Brent', lower=1.1, upper=4, hessian=TRUE)

# estimations
est$par # mean of r
sqrt(1/est$hessian[1][1]) # standard deviation of r

m <- seq(6, -4, -1)
p <- vismeteor::dvmgeom(m, limmag, r)
barplot(
    p,
    names.arg = m,
    main = paste0('Density (r = ', r, ', limmag = ', limmag, ')'),
    col = "blue",
    xlab = 'm',
    ylab = 'p',
    border = "blue",
    space = 0.5
)
axis(side = 2, at = pretty(p))
}
\seealso{
\link{vmperception}
\link[stats:Geometric]{stats::Geometric}
}
