% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/limitDistribution.R
\name{limitDistribution}
\alias{limitDistribution}
\title{Limit a virtual species distribution to a defined area}
\usage{
limitDistribution(x, geographical.limit = "extent", area = NULL,
  plot = TRUE)
}
\arguments{
\item{x}{a \code{rasterLayer} object composed of 0, 1 and NA, or the output list from 
\code{\link{generateSpFromFun}}, \code{\link{generateSpFromPCA}} 
or \code{\link{generateRandomSp}}}

\item{geographical.limit}{\code{"country"}, \code{"region"}, 
\code{"continent"}, \code{"polygon"}, \code{"raster"} or \code{"extent"}. The method used
to limit the distribution range: see details.}

\item{area}{\code{NULL}, a character string, a \code{polygon}, a \code{raster} or an \code{extent} object.
The area in which the distribution range will be limited: see details. If \code{NULL}
and \code{geographical.limit = "extent"}, then you will be asked to draw an
extent on the map.}

\item{plot}{\code{TRUE} or \code{FALSE}. If \code{TRUE}, the resulting limited
distribution will be plotted.}
}
\value{
a \code{list} containing 7 elements:
\itemize{
\item{\code{approach}: the approach used to generate the species, \emph{i.e.}, \code{"response"}}
\item{\code{details}: the details and parameters used to generate the species}
\item{\code{suitab.raster}: the virtual species distribution, as a Raster object containing the
environmental suitability)}
\item{\code{PA.conversion}: the parameters used to convert the suitability into presence-absence}
\item{\code{pa.raster}: the presence-absence map, as a Raster object containing 0 (absence) / 1 (presence) / NA}
\item{\code{geographical.limit}: the method used to
limit the distribution and the area in which the distribution is restricted}
\item{\code{occupied.area}: the area occupied by the virtual species as a
Raster of presence-absence}
}
The structure of the virtualspecies object can be seen using \code{str()}
}
\description{
This function is designed to limit species distributions to a subsample of
their total distribution range. It will thus generate a species which is not
at the equilibrium with its environment (i.e., which did not occupy the full
range of suitable environmental conditions).

This function basically takes any type of raster and will limit values above
0 to areas where the species is allowed to disperse.
}
\details{
\href{http://borisleroy.com/virtualspecies_tutorial/08-dispersallimitation.html}{Online tutorial for this function}


\bold{How the function works:}

The function will remove occurrences of the species outside the chosen area:
\itemize{
\item{NA are kept unchanged}
\item{0 are kept unchanged}
\item{values > 0 within the limits of \code{area} are kept unchanged}
\item{values > 0 outside the limits of \code{area} are set to 0}
}


\bold{How to define the area in which the range is limited:}

You can choose to limit the distribution range of the species to:
\enumerate{
\item{a particular country, region or continent (assuming your raster has
the WGS84 projection): 

Set the argument
\code{geographical.limit} to \code{"country"}, \code{"region"} or
\code{"continent"}, and provide the name(s) of the associated countries,
regions or continents to \code{area} (see examples). 

List of possible \code{area} names:
\itemize{
\item{Countries: type \code{levels(getMap()@data$SOVEREIGNT)} in the console}
\item{Regions: "Africa", "Antarctica", "Asia", "Australia", "Europe", 
"North America", "South America"}
\item{Continents: "Africa", "Antarctica", "Australia", "Eurasia", 
"North America", "South America"}}
}
\item{a polygon:

Set \code{geographical.limit} to \code{"polygon"}, and provide your
polygon to \code{area}.
}
\item{a raster:

Set \code{geographical.limit} to \code{"raster"}, and provide your
raster to \code{area}. Your raster values should be 1 (suitable area),
0 (unsuitable area) or NA (outside your mask).
}
\item{an extent object:

Set \code{geographical.limit} to \code{"extent"}, and either provide your
extent object to \code{area}, or leave it \code{NULL} to draw an extent on
the map.}
}
}
\examples{
# Create an example stack with six environmental variables
a <- matrix(rep(dnorm(1:100, 50, sd = 25)), 
            nrow = 100, ncol = 100, byrow = TRUE)
env <- stack(raster(a * dnorm(1:100, 50, sd = 25)),
             raster(a * 1:100),
             raster(a * logisticFun(1:100, alpha = 10, beta = 70)),
             raster(t(a)),
             raster(exp(a)),
             raster(log(a)))
names(env) <- paste("Var", 1:6, sep = "")   

# More than 6 variables: by default a PCA approach will be used
sp <- generateRandomSp(env)

# limiting the distribution to a specific extent
limit <- extent(0.5, 0.7, 0.6, 0.8)

limitDistribution(sp, area = limit)


# Example of a raster of habitat patches
habitat.raster <- setValues(sp$pa.raster, 
                          sample(c(0, 1), size = ncell(sp$pa.raster), replace = TRUE))

plot(habitat.raster) # 1 = suitable habitat; 0 = unsuitable habitat
sp <- limitDistribution(sp, geographical.limit = "raster", area = habitat.raster)
par(mfrow = c(2, 1))
plot(sp$pa.raster)
plot(sp$occupied.area) # Species could not occur in many cells because
# habitat patches were unsuitable
}
\author{
Boris Leroy \email{leroy.boris@gmail.com}

with help from C. N. Meynard, C. Bellard & F. Courchamp
}
