% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Denoise.R
\name{denoise_scene}
\alias{denoise_scene}
\title{Filters isolated points from a point cloud}
\usage{
denoise_scene(data, method, filter, k, store_noise)
}
\arguments{
\item{data}{LAS file of a 3D point cloud.}

\item{method}{character string. Defines the method to use for noise filtering.
Can be "quantile", "sd" or "voxel". See details. Default = "sd".}

\item{filter}{numeric. The intensity of the filter that depends on the
method. See details.}

\item{k}{numeric. The number of nearest neighbours to use to compute the mean
nearest neighbour distance. Required only if
method = "quantile" or "sd". Default = 5.}

\item{store_noise}{logical. If TRUE, the surveyed points considered as noise are not
removed from the data and a column "Noise" is added with a value of 1
indicating non-noisy points and a value of 2 indicating noisy points.
Default = FALSE.}
}
\value{
The filtered data (if \code{store_noise = FALSE}) or the classified
data (if \code{store_noise = TRUE}) with noisy points labeled as 2.
}
\description{
Filters isolated points from a point cloud
}
\details{
\subsection{\code{method = "quantile"}}{
the quantile-based method computes the distance of the k
nearest neighbours for each surveyed point and considers points that fall in
the last user defined quantile as noise. If quantile is used as the filtering
method, the default is set to = 0.999.}
\subsection{\code{method = "sd"}}{
the standard deviation-based method computes the average
distance of the k nearest neighbours of each surveyed point and considers
points as noise if they are more than the average distance plus a number of
times the standard deviation away from other surveyed points. The filter
parameter sets the standard deviation multiplier. Default = 4. This filter is
similar to the "SOR filter" available in
\href{https://www.cloudcompare.org/doc/wiki/index.php?title=SOR_filter}{CloudCompare}.}
\subsection{\code{method = "voxel"}}{
the voxel-based method considers surveyed points as noise if
they are the only surveyed point within a user defined voxel volume. The
\code{filter} parameter sets the voxel size (i.e., voxel side length).
Default = 0.5.}
}
\examples{
\donttest{
#- import the tree_line_plot dataset
file <- system.file("extdata", "tree_line_plot.laz", package="viewshed3d")
tls <- lidR::readLAS(file,select="xyz")

#- remove duplicated points
tls <- lidR::filter_duplicates(tls)

#- filter noise with the quantile base method
data <- viewshed3d::denoise_scene(tls,
                                  method="quantile",
                                  filter=0.999,
                                  k=5,
                                  store_noise = TRUE)

lidR::plot(data,color="Noise",colorPalette=c("white","red")) # plot

#- filter noise with the standard deviation based method
data <- viewshed3d::denoise_scene(tls,
                                  method="sd",
                                  filter=4,
                                  k=5,
                                  store_noise = TRUE)

lidR::plot(data,color="Noise",colorPalette=c("white","red")) # plot

#- filter noise with the voxel based method
data <- viewshed3d::denoise_scene(tls,
                                  method="voxel",
                                  filter=0.5,
                                  store_noise = TRUE)
lidR::plot(data,color="Noise",colorPalette=c("white","red")) # plot
}
}
