DEFAULT_RSPM_REPO_ID <-  "1" # cran
DEFAULT_RSPM <-  "https://packagemanager.rstudio.com"

#' Write a Dockerfile for a vetiver model
#'
#' After creating a Plumber file with [vetiver_write_plumber()], use
#' `vetiver_write_docker()` to create a Dockerfile plus an `renv.lock` file
#' for a pinned [vetiver_model()].
#'
#' @inheritParams vetiver_api
#' @param plumber_file A path for your Plumber file, created via
#' [vetiver_write_plumber()]. Defaults to `plumber.R` in the working directory.
#' @param path A path to write the Dockerfile and `renv.lock` lockfile,
#' capturing the model's package dependencies. Defaults to the working directory.
#'
#' @return The content of the Dockerfile, invisibly.
#' @export
#'
#' @examplesIf interactive() || identical(Sys.getenv("IN_PKGDOWN"), "true")
#'
#' library(pins)
#' tmp_plumber <- tempfile()
#' b <- board_temp(versioned = TRUE)
#' cars_lm <- lm(mpg ~ ., data = mtcars)
#' v <- vetiver_model(cars_lm, "cars_linear")
#' vetiver_pin_write(b, v)
#' vetiver_write_plumber(b, "cars_linear", file = tmp_plumber)
#' vetiver_write_docker(v, tmp_plumber, tempdir())
#'
vetiver_write_docker <- function(vetiver_model,
                                 plumber_file = "plumber.R",
                                 path = ".") {

    from_r_version <- glue::glue("FROM rocker/r-ver:{getRversion()}\n\n")

    pkgs <- unique(c(docker_pkgs, vetiver_model$metadata$required_pkgs))
    lockfile <- normalizePath(
        write_renv_lockfile(path = path, pkgs = pkgs),
        winslash = "/"
    )
    sys_reqs <- glue_sys_reqs(pkgs)
    copy_renv <- glue("COPY {lockfile} /opt/ml/renv.lock")
    copy_plumber <- glue("COPY {plumber_file} /opt/ml/plumber.R")


    ret <- compact(list(
        "# Generated by the vetiver package; edit with care\n",
        ## https://github.com/rstudio/plumber/blob/main/Dockerfile:
        from_r_version,
        sys_reqs,
        copy_renv,
        copy_plumber,
        '\nRUN Rscript -e "install.packages("renv")"',
        'RUN R -e "renv::restore()"\n',
        "EXPOSE 8000",
        'ENTRYPOINT ["R", "-e", "pr <- plumber::plumb(rev(commandArgs())[1]); pr$run(host = \'0.0.0.0\', port = 8000)"]',
        'CMD ["/opt/ml/plumber.R"]'
    ))

    readr::write_lines(ret, file = file.path(path, "Dockerfile"))
}

docker_pkgs <- c("pins", "plumber", "rapidoc", "vetiver", "renv")

write_renv_lockfile <- function(path, pkgs) {
    lockfile <- renv::paths$lockfile(project = path)
    renv::snapshot(
        project = path,
        lockfile = lockfile,
        packages = pkgs,
        prompt = FALSE,
        force = TRUE
    )
    lockfile
}

glue_sys_reqs <- function(pkgs) {
    rspm <- Sys.getenv("RSPM_ROOT", DEFAULT_RSPM)
    rspm_repo_id <- Sys.getenv("RSPM_REPO_ID", DEFAULT_RSPM_REPO_ID)
    rspm_repo_url <- glue("{rspm}/__api__/repos/{rspm_repo_id}")

    pkgnames <- glue_collapse(pkgs, sep = "&pkgname=")

    req_url <- glue(
        "{rspm_repo_url}/sysreqs?all=false",
        "&pkgname={pkgnames}&distribution=ubuntu&release=20.04"
    )
    res <- curl::curl_fetch_memory(req_url)
    sys_reqs <- jsonlite::fromJSON(rawToChar(res$content), simplifyVector = FALSE)
    if (!is.null(sys_reqs$error)) {
        rlang::abort(sys_reqs$error)
    }
    sys_reqs <- map(sys_reqs$requirements, pluck, "requirements", "packages")
    sys_reqs <- sort(unique(map_chr(sys_reqs, 1L)))
    sys_reqs <- glue_collapse(sys_reqs, sep = " \\\n  ")
    glue(
        "RUN apt-get update -qq && ",
        "apt-get install -y --no-install-recommends \\\n  ",
        sys_reqs,
        "\n",
        .trim = FALSE
    )
}

