ggConfidenceCurve <- function(metric = 'd', value = NULL, n = NULL,
                              conf.level=NULL,
                              wRange = c(.05, .8),
                              curveSize = 1,
                              curveColor = 'black',
                              confRange = c(.0001, .9999),
                              confLines = c(.50, .80, .95, .99),
                              confLineColor = 'grey',
                              confLineSize=1,
                              xlab = metric,
                              steps=1000,
                              theme=theme_bw()) {
  ### Check arguments
  if (is.null(conf.level) && is.null(n)) {
    stop("Arguments 'conf.level' and 'n' cannot both be NULL!");
  }
  if (is.null(value)) {
    stop("You must always specify a value for your metric.");
  }
  if (tolower(metric) == 'd') {
    if (is.null(n)) {
      ### So, sample sizes on the y axis, and values on the x axis.
      ### So we need the values (bounds) corresponding to different
      ### widths (which then correspond to sample sizes).
      minN <- pwr.cohensdCI(d=value, w=max(wRange));
      maxN <- pwr.cohensdCI(d=value, w=min(wRange));
      df1 <- df2 <- data.frame(n = seq(minN, maxN, length.out=steps));
      confInts <- cohensdCI(d=value,
                            n=seq(minN, maxN, length.out=steps),
                            conf.level=conf.level);
      df1 <- cbind(df1, confInts[, 1]);
      df2 <- cbind(df2, confInts[, 2]);
    } else {
      ### So, confidence is on the y axis, and values on the x axis.
      ### So we need to the confidence interval bounds for the given
      ### sample size, for different confidence levels.
      minConf <- min(confRange);
      maxConf <- max(confRange);
      df1 <- df2 <- data.frame(confidence = seq(minConf, maxConf, length.out=steps));
      confInts <- matrix(unlist(lapply(df1$confidence, cohensdCI, d=value,
                                       n=n)), ncol=2, byrow=TRUE);
      df1 <- cbind(df1, confInts[, 1]);
      df2 <- cbind(df2, confInts[, 2]);
      ### If need be, get metric values corresponding to confidence
      ### interval bounds
      if (!is.null(confLines) && (length(confLines) > 0)) {
        metricValues <- list();
        for (i in seq_along(confLines)) {
          metricValues[[i]] <- cohensdCI(d=value, n=n, conf.level=confLines[i]);
        }
      }      
    }
  } else if (tolower(metric)=='r') {
    if (is.null(n)) {
      ### So, sample sizes on the y axis, and values on the x axis.
      ### So we need the values (bounds) corresponding to different
      ### widths (which then correspond to sample sizes).
      minN <- pwr.confIntR(r=value, w=max(wRange));
      maxN <- pwr.confIntR(r=value, w=min(wRange));
      df1 <- df2 <- data.frame(n = seq(minN, maxN, length.out=steps));
      confInts <- matrix(unlist(lapply(df1$n, confIntR, r=value,
                                       conf.level=conf.level)),
                         ncol=2, byrow=TRUE);
      df1 <- cbind(df1, confInts[, 1]);
      df2 <- cbind(df2, confInts[, 2]);
    } else {
      ### So, confidence is on the y axis, and values on the x axis.
      ### So we need to the confidence interval bounds for the given
      ### sample size, for different confidence levels.
      minConf <- min(confRange);
      maxConf <- max(confRange);
      df1 <- df2 <- data.frame(confidence = seq(minConf, maxConf, length.out=steps));
      confInts <- matrix(unlist(lapply(df1$confidence, confIntR, r=value, N=n)),
                         ncol=2, byrow=TRUE);
      df1 <- cbind(df1, confInts[, 1]);
      df2 <- cbind(df2, confInts[, 2]);
      ### If need be, get metric values corresponding to confidence
      ### interval bounds
      if (!is.null(confLines) && (length(confLines) > 0)) {
        metricValues <- list();
        for (i in seq_along(confLines)) {
          metricValues[[i]] <- confIntR(r=value, N=n, conf.level=confLines[i]);
        }
      }
    }
  } else {
    stop("Sorry, metric '", metric, "' is not implemented.");
  }
  plot <- ggplot();
  if (is.null(n)) {
    names(df1) <- names(df2) <- c('n', metric);
    plot <- plot +
      geom_line(data=df1, aes_string(x=metric, y='n')) +
      geom_line(data=df2, aes_string(x=metric, y='n')) +
      ggtitle(paste0("Inverse Confidence Curve for ",
                     metric, " = ", value, " and conf.level = ", conf.level));
  } else  {
    ### Add lines to indicate confidence intervals, if we need to
    if (!is.null(confLines) && (length(confLines) > 0)) {
      for (i in seq_along(confLines)) {
        plot <- plot +
          geom_hline(yintercept = confLines[i],
                     color=confLineColor, size=confLineSize) +
          geom_vline(xintercept = metricValues[[i]][1],
                     color=confLineColor, size=confLineSize) +
          geom_vline(xintercept = metricValues[[i]][2],
                     color=confLineColor, size=confLineSize);
      }
    }
    names(df1) <- names(df2) <- c('confidence', metric);
    df1 <- rbind(c(1, -Inf), df1);
    df2 <- rbind(df2, c(1, Inf));
    plot <- plot +
      geom_line(data=df1, aes_string(x=metric, y='confidence'),
                color = curveColor, size = curveSize) +
      geom_line(data=df2, aes_string(x=metric, y='confidence'),
                color = curveColor, size = curveSize) +
      ggtitle(paste0("Confidence Curve for ",
                     metric, " = ", value, " and n = ", n));
    
  }
  if (!is.null(n) && !is.null(confLines) && (length(confLines) > 0)) {
    metricValues <- round(sort(unlist(metricValues)), 2);
    metricValueLabels <-
      ifelseObj((tolower(metric) == 'r'),
                noZero(metricValues),
                metricValues);
    plot <- plot +
      scale_x_continuous(sec.axis = dup_axis(breaks=metricValues,
                                             labels=metricValueLabels,
                                             name="")) +
      scale_y_continuous(sec.axis = dup_axis(breaks=round(sort(unlist(confLines)), 2),
                                             name=""));
  }
  plot <- plot + theme + xlab(xlab);
  return(plot);
}
