% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ipcw.R
\name{ipcw}
\alias{ipcw}
\title{Inverse Probability of Censoring Weights (IPCW) Method
for Treatment Switching}
\usage{
ipcw(
  data,
  id = "id",
  stratum = "",
  tstart = "tstart",
  tstop = "tstop",
  event = "event",
  treat = "treat",
  swtrt = "swtrt",
  swtrt_time = "swtrt_time",
  swtrt_time_lower = "",
  swtrt_time_upper = "",
  base_cov = "",
  numerator = "",
  denominator = "",
  logistic_switching_model = FALSE,
  strata_main_effect_only = TRUE,
  firth = FALSE,
  flic = FALSE,
  ns_df = 3,
  stabilized_weights = TRUE,
  trunc = 0,
  trunc_upper_only = TRUE,
  swtrt_control_only = TRUE,
  alpha = 0.05,
  ties = "efron",
  boot = TRUE,
  n_boot = 1000,
  seed = NA
)
}
\arguments{
\item{data}{The input data frame that contains the following variables:
\itemize{
\item \code{id}: The id to identify observations belonging to the same
subject for counting process data with time-dependent covariates.
\item \code{stratum}: The stratum.
\item \code{tstart}: The starting time of the time interval for
counting-process data with time-dependent covariates.
\item \code{tstop}: The stopping time of the time interval for
counting-process data with time-dependent covariates.
\item \code{event}: The event indicator, 1=event, 0=no event.
\item \code{treat}: The randomized treatment indicator, 1=treatment,
0=control.
\item \code{swtrt}: The treatment switch indicator, 1=switch, 0=no switch.
\item \code{swtrt_time}: The time from randomization to treatment switch.
\item \code{swtrt_time_lower}: The lower bound of treatment switching time.
\item \code{swtrt_time_upper}: The upper bound of treatment switching time.
\item \code{base_cov}: The baseline covariates (excluding treat) used in
the outcome model.
\item \code{numerator}: The baseline covariates used in the switching
model for the numerator for stabilized weights.
\item \code{denominator}: The baseline and time-dependent covariates
used in the switching model for the denominator.
}}

\item{id}{The name of the id variable in the input data.}

\item{stratum}{The name(s) of the stratum variable(s) in the input data.}

\item{tstart}{The name of the tstart variable in the input data.}

\item{tstop}{The name of the tstop variable in the input data.}

\item{event}{The name of the event variable in the input data.}

\item{treat}{The name of the treatment variable in the input data.}

\item{swtrt}{The name of the swtrt variable in the input data.}

\item{swtrt_time}{The name of the swtrt_time variable in the input data.}

\item{swtrt_time_lower}{The name of the swtrt_time_lower variable in
the input data.}

\item{swtrt_time_upper}{The name of the swtrt_time_upper variable in
the input data.}

\item{base_cov}{The vector of names of base_cov variables (excluding
treat) in the input data for the Cox model.}

\item{numerator}{The vector of names of variables in the input data
for the numerator switching model for stabilized weights.}

\item{denominator}{The vector of names of variables in the input data
for the denominator switching model.}

\item{logistic_switching_model}{Whether a pooled logistic regression
switching model is used.}

\item{strata_main_effect_only}{Whether to only include the strata main
effects in the logistic regression switching model. Defaults to
\code{TRUE}, otherwise all possible strata combinations will be
considered in the switching model.}

\item{firth}{Whether the firth's bias reducing penalized likelihood
should be used. The default is \code{FALSE}.}

\item{flic}{Whether to apply intercept correction to obtain more
accurate predicted probabilities. The default is \code{FALSE}.}

\item{ns_df}{Degrees of freedom for the natural cubic spline.
Defaults to 3 for two inner knots at the 33 and 67 percentiles
of the artificial censoring times due to treatment switching.}

\item{stabilized_weights}{Whether to use the stabilized weights.}

\item{trunc}{The pre-specified fraction of the weights. Defaults to 0
for no truncation in weights.}

\item{trunc_upper_only}{Whether to truncate the weights from the upper
end of the distribution only. Defaults to \code{TRUE}, otherwise
the weights will be truncated from both the lower and upper ends of
the distribution.}

\item{swtrt_control_only}{Whether treatment switching occurred only in
the control group.}

\item{alpha}{The significance level to calculate confidence intervals.}

\item{ties}{The method for handling ties in the Cox model, either
"breslow" or "efron" (default).}

\item{boot}{Whether to use bootstrap to obtain the confidence
interval for hazard ratio. Defaults to \code{TRUE}.}

\item{n_boot}{The number of bootstrap samples.}

\item{seed}{The seed to reproduce the bootstrap results.
The seed from the environment will be used if left unspecified.}
}
\value{
A list with the following components:
\itemize{
\item \code{logrank_pvalue}: The two-sided p-value of the log-rank test
based on the treatment policy strategy.
\item \code{cox_pvalue}: The two-sided p-value for treatment effect based on
the Cox model.
\item \code{hr}: The estimated hazard ratio from the Cox model.
\item \code{hr_CI}: The confidence interval for hazard ratio.
\item \code{hr_CI_type}: The type of confidence interval for hazard ratio,
either "Cox model" or "bootstrap".
\item \code{fit_switch}: A list of the fitted switching models for the
denominator and numerator by treatment group.
\item \code{df_outcome}: The input data frame for the outcome Cox model
including the inverse probability of censoring weights.
\item \code{fit_outcome}: The fitted outcome model.
\item \code{settings}: A list with the following components:
\itemize{
\item \code{logistic_switching_model}: Whether a pooled logistic
regression switching model is used.
\item \code{strata_main_effect_only}: Whether to only include the
strata main effects in the logistic regression switching model.
\item \code{firth}: Whether the firth's bias reducing penalized likelihood
should be used.
\item \code{flic}: Whether to apply intercept correction to obtain more
accurate predicted probabilities.
\item \code{stabilized_weights}: Whether to use the stabilized weights.
\item \code{trunc}: The truncation fraction of the weight distribution.
\item \code{trunc_upper_only}: Whether to truncate the weights from the
upper end of the distribution only.
\item \code{swtrt_control_only} Whether treatment switching occurred only
in the control group.
\item \code{alpa}: The significance level to calculate confidence
intervals.
\item \code{ties}: The method for handling ties in the Cox model.
\item \code{boot}: Whether to use bootstrap to obtain the confidence
interval for hazard ratio.
\item \code{n_boot}: The number of bootstrap samples.
\item \code{seed}: The seed to reproduce the simulation results.
}
\item \code{hr_boots}: The bootstrap hazard ratio estimates if \code{boot} is
\code{TRUE}.
}
}
\description{
Uses the IPCW method to obtain the hazard ratio estimate
of the Cox model to account for treatment switching.
}
\details{
We use the following steps to obtain the hazard ratio estimate
and confidence interval had there been no treatment switching:
\itemize{
\item Exclude observations after treatment switch and set up the crossover
and event indicators for the last time interval for each subject.
\item For time-dependent covariates Cox switching models, replicate unique
event times across treatment arms within each subject.
\item Fit the denominator switching model (and the numerator switching model
for stabilized weights) to obtain the inverse probability
of censoring weights. This can be a Cox model with time-dependent
covariates or a pooled logistic regression model.
\item Fit the weighted Cox model to the censored outcome survival times
to obtain the hazard ratio estimate.
\item Use bootstrap to construct the p-value and confidence interval for
hazard ratio.
}
}
\examples{

# Example 1: pooled logistic regression switching model

library(dplyr)

sim1 <- tsegestsim(
  n = 500, allocation1 = 2, allocation2 = 1, pbprog = 0.5, 
  trtlghr = -0.5, bprogsl = 0.3, shape1 = 1.8, 
  scale1 = 0.000025, shape2 = 1.7, scale2 = 0.000015, 
  pmix = 0.5, admin = 5000, pcatnotrtbprog = 0.5, 
  pcattrtbprog = 0.25, pcatnotrt = 0.2, pcattrt = 0.1, 
  catmult = 0.5, tdxo = 1, ppoor = 0.1, pgood = 0.04, 
  ppoormet = 0.4, pgoodmet = 0.2, xomult = 1.4188308, 
  milestone = 546, swtrt_control_only = TRUE,
  outputRawDataset = 1, seed = 2000)

fit1 <- ipcw(
  sim1$paneldata, id = "id", tstart = "tstart", 
  tstop = "tstop", event = "died", treat = "trtrand", 
  swtrt = "xo", swtrt_time = "xotime", 
  swtrt_time_lower = "timePFSobs",
  swtrt_time_upper = "xotime_upper", base_cov = "bprog", 
  numerator = "bprog", denominator = "bprog*catlag", 
  logistic_switching_model = TRUE, ns_df = 3,
  swtrt_control_only = TRUE, boot = FALSE)
  
c(fit1$hr, fit1$hr_CI) 

# Example 2: time-dependent covariates Cox switching model

fit2 <- ipcw(
  shilong, id = "id", tstart = "tstart", tstop = "tstop", 
  event = "event", treat = "bras.f", swtrt = "co", 
  swtrt_time = "dco", 
  base_cov = c("agerand", "sex.f", "tt_Lnum", "rmh_alea.c", 
               "pathway.f"),
  numerator = c("agerand", "sex.f", "tt_Lnum", "rmh_alea.c", 
                "pathway.f"),
  denominator = c("agerand", "sex.f", "tt_Lnum", "rmh_alea.c",
                  "pathway.f", "ps", "ttc", "tran"),
  swtrt_control_only = FALSE, boot = FALSE)

c(fit2$hr, fit2$hr_CI)

}
\references{
James M. Robins and Dianne M. Finkelstein.
Correcting for noncompliance and dependent censoring in an AIDS clinical
trial with inverse probability of censoring weighted (IPCW) log-rank tests.
Biometrics. 2000;56:779-788.
}
\author{
Kaifeng Lu, \email{kaifenglu@gmail.com}
}
