% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/approx_areas.R
\name{approx_areas}
\alias{approx_areas}
\title{Approximate area sizes of the shapes}
\usage{
approx_areas(shp, target = "metric", total.area = NULL)
}
\arguments{
\item{shp}{shape object, i.e., an \code{\link[sf:sf]{sf}} or \code{sp} object.}

\item{target}{target unit, one of
\describe{
    \item{\code{"prop"}:}{Proportional numbers. In other words, the sum of the area sizes equals one.}
    \item{\code{"norm"}:}{Normalized numbers. All area sizes are normalized to the largest area, of which the area size equals one.}
    \item{\code{"metric"} (default):}{Output area sizes will be either \code{"km"} (kilometer) or \code{"m"} (meter) depending on the map scale}
    \item{\code{"imperial"}:}{Output area sizes will be either \code{"mi"} (miles) or \code{"ft"} (feet) depending on the map scale}
    \item{other:}{Predefined values are "km^2", "m^2", "mi^2", and "ft^2". Other values can be specified as well, in which case \code{to} is required).}}
These units are the output units. See \code{orig} for the coordinate units used by the shape \code{shp}.}

\item{total.area}{total area size of \code{shp} in number of target units (defined by \code{target}). Useful if the total area of the \code{shp} differs from a reference total area value. For \code{"metric"} and \code{"imperial"} units, please provide the total area in squared kilometers respectively miles.}
}
\value{
Numeric vector of area sizes (class \code{\link[units:units]{units}}).
}
\description{
Approximate the area sizes of the polygons in real-world area units (such as sq km or sq mi), proportional numbers, or normalized numbers. Also, the areas can be calibrated to a prespecified area total. This function is a convenient wrapper around \code{\link[sf:geos_measures]{st_area}}.
}
\details{
Note that the method of determining areas is an approximation, since it depends on the used projection and the level of detail of the shape object. Projections with equal-area property are highly recommended. See \url{https://en.wikipedia.org/wiki/List_of_map_projections} for equal area world map projections.
}
\examples{
if (require(tmap) && packageVersion("tmap") >= "3.99") {
    data(NLD_muni)

    NLD_muni$area <- approx_areas(NLD_muni, total.area = 33893)

    tm_shape(NLD_muni) +
        tm_bubbles(size="area",
                   size.legend = tm_legend(title = expression("Area in " * km^2)))


    # function that returns min, max, mean and sum of area values
    summary_areas <- function(x) {
        list(min_area=min(x),
             max_area=max(x),
             mean_area=mean(x),
             sum_area=sum(x))
    }

    # area of the polygons
    approx_areas(NLD_muni) \%>\% summary_areas()

    # area of the polygons, adjusted corrected for a specified total area size
    approx_areas(NLD_muni, total.area=33893) \%>\% summary_areas()

    # proportional area of the polygons
    approx_areas(NLD_muni, target = "prop") \%>\% summary_areas()

    # area in squared miles
    approx_areas(NLD_muni, target = "mi mi") \%>\% summary_areas()

    # area of the polygons when unprojected
    approx_areas(NLD_muni \%>\% sf::st_transform(crs = 4326)) \%>\% summary_areas()
}
}
\seealso{
\code{\link{approx_distances}}
}
