#' Tipping Point Analysis (Model-Free)
#'
#' Performs a model-free tipping point analysis on time-to-event data by
#' repeatedly imputing censored observations under varying assumptions. The model-free
#' framework assumes that censored patients share similar survival
#' behavior with those from whom they are sampled, without fitting any parametric
#' survival model.
#'
#' @details
#' The **model-free tipping point analysis** provides a reproducible and intuitive
#' framework for exploring the robustness of treatment effects in time-to-event
#' (survival) endpoints when censoring may differ between study arms.
#'
#' Two sampling modes are supported:
#' \itemize{
#'   \item `method = "random sampling"` - performs re-sampling of event
#'         times from the best or worst percentile of observed patients ranked by their event or censoring time.
#'         The `tipping_range` specifies the percentiles of the observed data from which
#'         event times will be sampled to impute censored patients.
#'         For the treatment arm, use the worst percentiles (shortest survival times) from the
#'          observed data of both arms. For the control arm, use the best percentiles (longest survival times).
#'   \item `method = "deterministic sampling"` - imputes a fixed number of censored
#'         patients deterministically. The `tipping_range` specifies the number of patients to be imputed.
#'         For the treatment arm, it defines the number of patients that will be assumed to
#'         have an event at their time of censoring. For the control arm, it defines the
#'         number of patients that will be assumed to be event-free at data cut-off, their maximum potential follow-up time.
#' }
#'
#' This function iteratively applies the random- or deterministic-sampling
#' imputation procedure across a range of
#' tipping point parameters `tipping_range`. For each parameter value:
#' \enumerate{
#'   \item Multiple imputed datasets are generated (`J` replicates), where censored
#'         observations in the selected arm are replaced by sampled or reassigned
#'         event times according to the imputation method.
#'   \item A Cox proportional hazards model is fitted to each imputed dataset.
#'   \item Model estimates are pooled using **Rubin’s rules** to obtain a combined
#'         hazard ratio and confidence interval for that tipping point parameter.
#' }
#'
#' The process yields a series of results showing how the treatment effect changes
#' as increasingly conservative or optimistic assumptions are made about censored
#' observations. The *tipping point* is defined as the smallest value of the
#' sensitivity parameter (percentile or number of imputed patients) for which the upper
#' bound of the hazard ratio confidence interval crosses 1 - i.e., where the
#' apparent treatment benefit is lost.
#'
#' @param dat data.frame containing at least 5 columns: TRT01P (treatment arm as factor), AVAL (survival time), EVENT (event indicator), CNSRRS (censoring reason) and MAXAVAL (maximum potential survival time, duration between randomization to data cut-off)

#' @param reason Vector specifying censoring reasons to be imputed.
#' @param impute a string specifying the treatment arm(s) which require imputation. It must be one of the arms from variable TRT01P, the first level of TRT01P is considered as the control arm.
#' @param J numeric indicating number of imputations.
#' @param tipping_range Numeric vector. Percentiles to use when `method = "random sampling"`. Number of patients to impute when `method = "deterministic sampling"`.
#' @param verbose Logical. If `TRUE`, prints progress and analysis details.
#' @param method Character. Either `"random sampling"` or `"deterministic sampling"`.
#' @param seed Integer. Random seed for reproducibility.
#' @param cox_fit A Cox model that will be used to calculate HRs on imputed datasets.
#'   In case of inclusion of stratification factors or covariates, conditional HR will be used.
#'
#' @return A `tipse` object containing:
#' \describe{
#'   \item{original data}{Input argument from 'data'.}
#'   \item{imputation_results}{A data frame of combined pooled model results across tipping points}
#'   \item{original_HR}{The original hazard ratio.}
#'   \item{reason_to_impute}{Input argument from 'reason'.}
#'   \item{arm_to_impute}{Input argument from 'impute'.}
#'   \item{method_to_impute}{Input argument from 'method'.}
#'   \item{imputation_data}{A list of imputed datasets for each tipping point value.}
#' }
#' @import dplyr
#' @importFrom survival coxph
#' @importFrom utils head tail
#' @export
#'
#' @examples
#' cox1 <- survival::coxph(Surv(AVAL, EVENT) ~ TRT01P, data = codebreak200)
#' result <- tipping_point_model_free(
#'   dat = codebreak200,
#'   reason = "Early dropout",
#'   impute = "docetaxel",
#'   J = 10,
#'   tipping_range = seq(5, 95, by = 5),
#'   cox_fit = cox1,
#'   verbose = TRUE,
#'   method = "random sampling",
#'   seed = 12345
#' )
#'
#' result2 <- tipping_point_model_free(
#'   dat = codebreak200,
#'   reason = "Early dropout",
#'   impute = "docetaxel",
#'   J = 10,
#'   tipping_range = seq(1, 21, by = 2),
#'   cox_fit = cox1,
#'   verbose = TRUE,
#'   method = "deterministic sampling",
#'   seed = 12345
#' )
tipping_point_model_free <- function(dat,
                                     reason,
                                     impute,
                                     J = 10,
                                     tipping_range = seq(5, 95, by = 5),
                                     cox_fit = NULL,
                                     verbose = FALSE,
                                     method = c("random sampling", "deterministic sampling"),
                                     seed = 12345) {
  #----------------------------#
  # Setup and validation
  #----------------------------#
  method <- match.arg(method, c("random sampling", "deterministic sampling"))
  set.seed(seed)

  dat <- sanitize_dataframe(dat)

  control <- levels(dat[["TRT01P"]])[1]
  trt <- levels(dat[["TRT01P"]])[2]

  if (!inherits(cox_fit, "coxph")) {
    stop("Argument 'cox_fit' must be a valid cox model object, e.g. coxph(Surv(AVAL, EVENT) ~ TRT01P, data = codebreak200).")
  }
  if (!(impute %in% c(control, trt))) {
    stop("Argument 'impute' must be one of the arms in column TRT01P.")
  }

  if (length(reason) == 0) {
    stop("Argument 'reason' must specify at least one censoring reason to impute.")
  }

  if (method == "random sampling") {
    tipping_range <- sanitize_percentile_range(tipping_range)
  } else {
    tipping_range <- sanitize_npts_range(dat, reason, impute, tipping_range)
  }

  HR <- exp(cox_fit$coefficients[paste0("TRT01P", trt)])

  #----------------------------#
  # Print setup info
  #----------------------------#
  if (verbose) {
    cat("\u2192 Detected arms:\n")
    cat("   Control arm   :", control, "\n")
    cat("   Treatment arm :", trt, "\n")
    cat("   Imputing arm  :", impute, "\n\n")
    cat("Starting tipping point analysis using method:", method, "\n")
    cat("Replicates per tipping point parameter:", J, "\n\n")
  }

  #----------------------------#
  # Main computation function
  #----------------------------#
  run_imputation <- function(param, method, cox_fit, verbose, dat, reason, impute, J, seed) {
    if (verbose) {
      label <- if (method == "random sampling") paste0(param, "%") else param
      cat(" \u2192 Imputing for parameter:", label, "\n")
    }

    multiply_imputed_dfs <- switch(method,
      "random sampling" = impute_random(dat, reason, impute, percentile = param, J = J, seed = seed),
      "deterministic sampling" = impute_deterministic(dat, reason, impute, npts = param, J = J, seed = seed)
    )

    pooled <- pool_results(multiply_imputed_dfs, cox_fit)
    pooled$parameter <- param

    list(pooled = pooled, km_data = multiply_imputed_dfs)
  }

  #----------------------------#
  # Run all imputations
  #----------------------------#

  results <- lapply(tipping_range, run_imputation,
    method = method, cox_fit = cox_fit,
    verbose = verbose, dat = dat, reason = reason, impute = impute, J = J, seed = seed
  )

  summary_results <- dplyr::bind_rows(lapply(results, `[[`, "pooled"))
  km_data_list <- lapply(results, `[[`, "km_data")
  names(km_data_list) <- as.character(tipping_range)

  #----------------------------#
  # Check tipping point
  #----------------------------#

  summary_results <- summary_results %>% mutate(tipping_point = FALSE)

  if (any(summary_results$HR_upperCI < 1) & any(summary_results$HR_upperCI > 1)) {
    if (method == "random sampling") {
      tip <- tail(which(summary_results$HR_upperCI >= 1), 1)
    } else {
      tip <- head(which(summary_results$HR_upperCI >= 1), 1)
    }

    if (abs(summary_results$HR_upperCI[tip] - 1) > 0.1) {
      warning("Consider decreasing the step of 'tipping_range', the upper CL at tipping point was too far away from 1.0.")
    }

    summary_results$tipping_point[tip] <- TRUE
  } else {
    warning("Tipping point not found, please check 'tipping_range'.")
  }


  if (verbose) cat("\nTipping point analysis completed successfully \u2705 \n")

  #----------------------------#
  # Return structured output
  #----------------------------#

  tipse <- list(
    original_data = dat,
    original_HR = HR,
    reason_to_impute = reason,
    arm_to_impute = impute,
    method_to_impute = method,
    imputation_results = summary_results,
    imputation_data = km_data_list
  )
  class(tipse) <- "tipse"
  return(tipse)
}
