% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils-perm.R
\name{perm_tester}
\alias{perm_tester}
\title{Monte Carlo Permutation Tests for Model P-Values}
\usage{
perm_tester(
  data,
  model,
  perm_var = NULL,
  strat_var = NULL,
  statistic,
  perm_n = 1000,
  alternative = "all",
  alpha = 0.05,
  seed = NULL
)
}
\arguments{
\item{data}{The dataframe from which the model is estimated.}

\item{model}{The model which will be estimated and re-estimated.}

\item{perm_var}{The variable in the model that will be permuted. Default is \code{NULL}
which takes the first \code{Y}n term in the formula of the model}

\item{strat_var}{Categorical variable for within-stratum permutations.
Defaults to \code{NULL}.}

\item{statistic}{The name of the model statistic you want to "grab" after
re-running the model with each permutation to compare to
the original model statistic.}

\item{perm_n}{The total number of permutations. Defaults to 1000.}

\item{alternative}{The alternative hypothesis. One of \code{"two.sided"} (default), \code{"left"},
\code{"right"}, and \code{"all"}. Defaults to \code{"all"}, which reports
the p-value statistics for all three alternative hypotheses.}

\item{alpha}{Alpha level for the hypothesis test. Defaults to 0.05.}

\item{seed}{Optional seed for reproducibility of the p-value statistics. Defaults to null.}
}
\value{
Returns a data frame with the observed statistic (\code{stat}), the
p-values (\code{P_left}, for left-tailed, \code{P_right} for right-tailed, and/or
\code{P_two} for two-tailed), and the standard errors and confidence
intervals for those p-values, respectively.
}
\description{
\code{perm_tester()} carries out Monte Carlo permutation tests for model
p-values from two-tailed, left-tailed, and/or right-tailed hypothesis
testing.
}
\details{
\code{perm_tester()} can be used to derive p-values under the randomization
model of inference. There are various reasons one might want to do this---
with text data, and observational data more generally, this might be
because the corpus/sample is not a random sample from a target population.
In such cases, population model p-values might not make much sense since
the asymptotically-derived standard errors from which they are constructed
themselves do not make sense. We might therefore want to make inferences
on the basis of whether or not randomness, as a data-generating mechanism,
might reasonably account for a statistic at least as extreme as the one
we observed. \code{perm_tester()} works from this idea.

\code{perm_tester()} works like this. First, the model (supplied the \code{model}
parameter) is run on the observed data. Second, we take some statistic of
interest, which we indicate with the \code{statistic} parameter, and set it to
the side. Third, a variable, \code{perm_var}, is permuted---meaning the observed
values for the rows of \code{data} on \code{perm_var} are randomly reshuffled. Fourth,
we estimate the model again, this time with the permuted \code{perm_var}. Fifth,
we get grab that same \code{statistic}. We repeat steps two through
five a total of \code{perm_n} times, each time tallying the number of times the
\code{statistic} from the permutation-derived model is greater than or equal to
(for a right-tailed test), less-than or equal to (for a left-tailed test),
and/or has an absolute value greater than or equal to (for a two-tailed test)
the \code{statistic} from the "real" model.

If we divide those tallies by the total number of permutations, then we
get randomization-based p-values. This is what \code{perm_tester()} does. The
null hypothesis is that randomness could likely generate the statistic
that we observe. The alternative hypothesis is that randomness alone likely
can't account for the observed statistic.

We then reject the null hypothesis if the p-value is below a threshold indicated
with \code{alpha}, which, as in population-based inference, is the probability
below which we are willing to reject the null hypothesis when it is actually
true. So if the p-value is below, say, \code{alpha} = 0.05 and we're performing,
a right-tailed test, then fewer than 5\% of the statistics derived from the
permutation-based models are greater than or equal to our observed
statistic. We would then reject the null, as it is unlikely (based on our \code{alpha}
threshold), that randomness as a data-generating mechanism can account
for a test statistic at least as large the one we observed.

In most cases, analysts probably cannot expect to perform "exact" permutation
tests where every possible permutation is accounted for---i.e., where
\code{perm_n} equals the total number of possible permutations. Instead, we
can take random samples of the "population" of permutations. \code{perm_tester()}
does this, and reports the standard errors and (1 - \code{alpha}) confidence
intervals for the p-values.

\code{perm_tester()} can also perform stratified permutation tests, where the observed
\code{perm_var} variables within groups. This can be done by setting the \code{strat_var}
variable to be he grouping variable.
}
\examples{

data <- text2map::meta_shakespeare

model <- lm(body_count ~ boas_problem_plays + year + genre, data = data)

# without stratified permutations, two-sided test
out1 <- perm_tester(
  data = data,
  model = model,
  statistic = "coefficients",
  perm_n = 500,
  alternative = "two.sided",
  alpha = .01,
  seed = 8675309
)

# with stratified permutations, two-sided test
out2 <- perm_tester(
  data = data,
  model = model,
  strat_var = "boas_problem_plays",
  statistic = "coefficients",
  perm_n = 500,
  alternative = "two.sided",
  alpha = .01,
  seed = 8675309
)

}
\references{
Taylor, Marshall A. (2020)
'Visualization Strategies for Regression Estimates with Randomization
Inference' \emph{Stata Journal} 20(2):309-335.
\doi{110.1177/1536867X20930999}.\cr

#' Darlington, Richard B. and Andrew F. Hayes (2016)
\emph{Regression analysis and linear models: Concepts, applications, and implementation}.
Guilford Publications.\cr

Ernst, Michael D. (2004)
'permutation methods: a basis for exact inference' \emph{Statistical Scicence}
19(4):676-685.
\doi{10.1214/088342304000000396}.\cr

Manly, Bryan F. J. (2007)
\emph{Randomization, Bootstrap and Monte Carlo Methods in Biology}.
Chapman and Hall/CRC.
\doi{10.1201/9781315273075}.\cr
}
\seealso{
\link{CMDist}, \link{CoCA}, \link{get_direction}
}
\author{
Marshall Taylor and Dustin Stoltz
}
