\name{terra-package}
\alias{terra-package}

\alias{terra}

\docType{package}

\title{The terra package}

\description{

\code{terra} provides methods to manipulate geographic (spatial) data in "raster" and "vector" form. Raster data divide space into rectangular cells (pixels) and they are commonly used to represent spatially continuous phenomena, such as elevation or the weather. Satellite images also have this data structure. In contrast, "vector" spatial data (points, lines, polygons) are typically used to represent discrete spatial entities, such as a road, country, or bus stop. 

The package implements two main classes (R data types): \code{SpatRaster} and \code{SpatVector}. \code{SpatRaster} supports handling large raster files that cannot be loaded into memory; local, focal, zonal, and global raster operations; polygon, line and point to raster conversion; integration with modeling methods to make spatial predictions; and more. \code{SpatVector} supports all types of geometric operations such as intersections.

Additional classes include \code{SpatExtent}, which is used to define a spatial extent (bounding box); \code{SpatRasterDataset}, which represents a collection of sub-datasets for the same area. Each sub-dataset is a SpatRaster with possibly many layers, and may, for example, represent different weather variables; and \code{SpatRasterCollection} and \code{SpatVectorCollection} that are vectors of SpatRaster or SpatVector. 

These classes hold a C++ pointer to the data and they cannot be directly saved to a ".Rds" file or used in cluster computing. They cannot be recovered from a saved R session either. See \code{\link{wrap}} or \code{\link{writeRaster}} to work around that limitation. 

The \code{terra} package is conceived as a replacement of the \code{raster} package. \code{terra} has a very similar, but simpler, interface, and it is faster than \code{raster}. At the bottom of this page there is a table that shows differences in the methods between the two packages. 

Below is a list of some of the most important methods grouped by theme. Some of these may not have been implemented yet (they are not hyperlinked). 
}

\section{}{
---------------------------------------------------------------------------------------------------------------------

\bold{SpatRaster}

---------------------------------------------------------------------------------------------------------------------}

\section{I. Creating, combining and sub-setting SpatRaster objects}{

  \tabular{ll}{
    \code{\link{rast}}\tab Create a SpatRaster from scratch, file, or another object\cr
    \code{\link{c}} \tab Combine SpatRasters (multiple layers)\cr
    \code{\link{add<-}} \tab Add a SpatRaster to another one\cr
    \code{\link{subset}} or \code{[[}, or \code{$} \tab Select layers of a SpatRaster\cr
    \code{\link{selectRange}} \tab Select cell values from different layers using an index layer\cr
--------------------------- \tab ------------------------------------------------------------------------------------------ \cr
}
}


\section{II. Changing the spatial extent and/or resolution of a SpatRaster}{
Also see the methods in section VIII

  \tabular{ll}{
    \code{\link{merge}} \tab Combine SpatRasters with different extents (but same origin and resolution) \cr
    \code{\link{mosaic}} \tab Combine SpatRasters with different extents using a function for overlapping cells \cr
    \code{\link{crop}} \tab Select a geographic subset of a SpatRaster \cr
    \code{\link{extend}} \tab Enlarge a SpatRaster \cr
    \code{\link{trim}} \tab Trim a SpatRaster by removing exterior rows and/or columns that only have NAs\cr
    \code{\link{aggregate}} \tab Combine cells of a SpatRaster to create larger cells \cr
    \code{\link{disagg}} \tab Subdivide cells \cr
    \code{\link{resample}} \tab Resample (warp) values to a SpatRaster with a different origin and/or resolution \cr 
	\code{\link{project}} \tab Project (warp) values to a SpatRaster with a different coordinate reference system \cr
    \code{\link{shift}} \tab Adjust the location of SpatRaster \cr
    \code{\link{flip}} \tab  Flip values horizontally or vertically \cr
    \code{\link{rotate}} \tab Rotate values around the date-line (for lon/lat data) \cr
    \code{\link{t}} \tab Transpose a SpatRaster\cr
--------------------------- \tab ------------------------------------------------------------------------------------------ \cr
  }
}


  
\section{III. Local (cell based) computation}{

\subsection{Apply-like methods}{

  \tabular{ll}{
    \code{\link{app}} \tab Apply a function to all cells, across layers, typically to summarize (as in \code{base::apply}) \cr
    \code{\link{tapp}} \tab Apply a function to groups of layers (as in \code{base::tapply} and \code{stats::aggregate})\cr
    \code{\link{lapp}} \tab Apply a function to using the layers of a SpatRaster as variables\cr
    \code{\link{sapp}} \tab Apply a function to each layer\cr
    \code{\link{rapp}} \tab Apply a function to a spatially variable range of layers\cr
	}
}

\subsection{Arithmetic, logical, and standard math methods}{

  \tabular{ll}{
    \code{\link{Arith-methods}} \tab Standard arithmetic methods (\code{+, -, *, ^, \%\%, \%/\%, /}) \cr
    \code{\link{Compare-methods}} \tab Comparison methods for SpatRaster (\code{==, !=, >, <, <=, >=}) \cr
    \code{\link{Logic-methods}} \tab Boolean methods (\code{!, &, |}) \cr
	\code{\link{Math-methods}} \tab \code{abs, sign, sqrt, ceiling, floor, trunc, cummax, cummin, cumprod,} \cr	
	\tab \code{cumsum, log, log10, log2, log1p, acos, acosh, asin, asinh, atan, atanh,} \cr	 
	\tab \code{exp, expm1, cos, cosh, sin, sinh, tan, tanh, round, signif}\cr
    \code{\link{Summary-methods}} \tab \code{mean, max, min, median, sum, range, prod,} \cr
    \tab \code{any, all, stdev, which.min, which.max}\cr
	}
}

\subsection{Other methods}{

  \tabular{ll}{
    \code{\link{cellSize}} \tab Compute the area of cells \cr
    \code{\link{classify}} \tab (Re-)classify values \cr
    \code{\link{cover}} \tab First layer covers second layer except where the first layer is \code{NA} \cr
    \code{\link{init}} \tab Initialize cells with new values \cr	
    \code{\link{mask}} \tab Replace values in a SpatRaster based on values in another SpatRaster\cr
    \code{\link{subst}} \tab Substitute (replace) cell values \cr
    \code{\link{which.lyr}} \tab which is the first layer that is \code{TRUE}?\cr	
	
  --------------------------- \tab ------------------------------------------------------------------------------------------ \cr
  }
  }
}

\section{IV. Zonal and global computation}{
  \tabular{ll}{
    \code{\link{expanse}} \tab Compute the summed area of cells \cr
    \code{\link{crosstab}} \tab Cross-tabulate two SpatRasters\cr
    \code{\link{freq}} \tab Frequency table of SpatRaster cell values \cr
    \code{\link{global}} \tab Summarize SpatRaster cell values with a function \cr
    \code{\link{quantile}} \tab Quantiles \cr
    \code{\link{stretch}} \tab Stretch values \cr
    \code{\link{scale}} \tab Scale values \cr
    \code{\link[terra]{summary}} \tab Summary of the values of a SpatRaster (quartiles and mean) \cr
    \code{\link{unique}} \tab Get the unique values in a SpatRaster \cr
    \code{\link{zonal}} \tab Summarize a SpatRaster by zones in another SpatRaster \cr
  --------------------------- \tab ------------------------------------------------------------------------------------------ \cr
  }
}


\section{V. Focal and other spatial contextual computation}{

  \tabular{ll}{
    \code{\link{focal}} \tab Focal (neighborhood; moving window) functions \cr
    \code{\link{adjacent}} \tab Identify cells that are adjacent to a set of cells of a SpatRaster \cr
    \code{\link{boundaries}} \tab Detection of boundaries (edges)\cr
    \code{\link{distance}} \tab Shortest distance to a cell that is not \code{NA} or to or from a vector object\cr
    \code{direction} \tab Direction (azimuth) to or from cells that are not \code{NA}\cr
    \code{localFun} \tab Local association (using neighborhoods) functions \cr
    \code{\link{patches}} \tab Find patches (clumps) \cr
    \code{\link{terrain}} \tab Compute slope, aspect and other terrain characteristics from elevation data \cr	
    \code{\link{shade}} \tab Compute hill shade from slope and aspect layers \cr	
    \code{\link{autocor}} \tab Compute global or local spatial autocorrelation  \cr	
  --------------------------- \tab ------------------------------------------------------------------------------------------ \cr
  }
}

\section{VI. Model predictions}{

  \tabular{ll}{
    \code{\link{predict}} \tab Predict a non-spatial model to a SpatRaster \cr
    \code{\link{interpolate}} \tab Predict a spatial model to a SpatRaster \cr
  --------------------------- \tab ------------------------------------------------------------------------------------------ \cr
}
}


\section{VII. Accessing cell values}{

Ap	art from the function listed below, you can also use indexing with \code{[} with cell numbers, and row and/or column numbers \cr

\tabular{ll}{
    \code{\link{values}} \tab cell values (fails with very large rasters)\cr
    \code{\link{values<-}} \tab Set new values to the cells of a SpatRaster \cr
    \code{\link{setValues}} \tab Set new values to the cells of a SpatRaster \cr
    \code{\link{as.matrix}} \tab Get cell values as a matrix \cr
    \code{\link{as.array}} \tab Get cell values as an array \cr
    \code{\link{extract}} \tab Extract cell values from a SpatRaster (e.g., by cell, coordinates, polygon)\cr
    \code{\link{spatSample}} \tab Regular or random sample \cr
    \code{\link{minmax}} \tab Get the minimum and maximum value of the cells of a SpatRaster (if known) \cr
    \code{\link{setMinMax}} \tab Compute the minimum and maximum value of a SpatRaster if these are not known \cr
    \code{\link{extract}} \tab spatial queries of a SpatRaster with a SpatVector\cr
	
  --------------------------- \tab ------------------------------------------------------------------------------------------ \cr
  }
}


\section{VIII. Getting and setting SpatRaster dimensions }{

Get or set basic parameters of SpatRasters. If there are values associated with a SpatRaster object (either in memory or via a link to a file) these are lost when you change the number of columns or rows or the resolution. This is not the case when the extent is changed (as the number of columns and rows will not be affected). Similarly, with \bold{crs} you can set the coordinate reference system, but this does not transform the data (see \link{project} for that).   

  \tabular{ll}{
    \code{\link{ncol}}\tab The number of columns \cr
    \code{\link{nrow}} \tab The number of rows \cr
    \code{\link{ncell}} \tab The number of cells (can not be set directly, only via ncol or nrow) \cr
    \code{\link{res}} \tab The resolution (x and y) \cr
    \code{\link{nlyr}} \tab Get or set the number of layers \cr	
    \code{\link{names}} \tab Get or set the layer names \cr	
    \code{\link{xres}} \tab The x resolution (can be set with res) \cr
    \code{\link{yres}} \tab The y resolution (can be set with res)\cr
    \code{\link{xmin}} \tab The minimum x coordinate (or longitude) \cr
    \code{\link{xmax}} \tab The maximum x coordinate (or longitude) \cr
    \code{\link{ymin}} \tab The minimum y coordinate (or latitude) \cr
    \code{\link{ymax}} \tab The maximum y coordinate (or latitude) \cr
	\code{\link{ext}} \tab Get or set the extent (minimum and maximum x and y coordinates ("bounding box") \cr
	\code{\link{origin}} \tab The origin of a SpatRaster\cr
    \code{\link{crs}} \tab The coordinate reference system (map projection) \cr	
    \code{\link{is.lonlat}} \tab Test if an object has (or may have) a longitude/latitude coordinate reference system\cr
    \code{\link{sources}} \tab Get the filename(s) to which a SpatRaster is linked \cr	
    \code{\link{inMemory}} \tab Are the data sources in memory (or on disk)? \cr	
    \code{\link{compareGeom}} \tab Compare the geometry of SpatRasters \cr		
    \code{\link{NAflag}} \tab Set the \code{NA} value (for reading from a file with insufficient metadata) \cr
  --------------------------- \tab ------------------------------------------------------------------------------------------ \cr
}
}


\section{IX. Computing row, column, cell numbers and coordinates}{

Cell numbers start at 1 in the upper-left corner. They increase within rows, from left to right, and then row by row from top to bottom. Likewise, row numbers start at 1 at the top of the raster, and column numbers start at 1 at the left side of the raster. 

  \tabular{ll}{
    \code{\link{xFromCol}} \tab x-coordinates from column numbers \cr
    \code{\link{yFromRow}} \tab y-coordinates from row numbers \cr
    \code{\link{xFromCell}} \tab x-coordinates from row numbers \cr
    \code{\link{yFromCell}} \tab y-coordinates from cell numbers \cr
    \code{\link{xyFromCell}} \tab x and y coordinates from cell numbers \cr
    \code{\link{colFromX}} \tab Column numbers from x-coordinates (or longitude) \cr
    \code{\link{rowFromY}} \tab Row numbers from y-coordinates (or latitude) \cr
    \code{\link{rowColFromCell}} \tab Row and column numbers from cell numbers\cr
    \code{\link{cellFromXY}} \tab Cell numbers from x and y coordinates \cr
    \code{\link{cellFromRowCol}} \tab Cell numbers from row and column numbers \cr
    \code{\link{cellFromRowColCombine}} \tab Cell numbers from all combinations of row and column numbers \cr    
	\code{\link{cells}}\tab Cell numbers from an SpatVector or SpatExtent\cr
  --------------------------- \tab ------------------------------------------------------------------------------------------ \cr
  }
}


\section{X. Writing SpatRaster files}{

\subsection{Basic}{
  \tabular{ll}{
    \code{\link{writeRaster}} \tab Write all values of SpatRaster to disk. You can set the filetype, datatype, compression. \cr
    \code{\link{writeCDF}} \tab Write SpatRaster data to a netCDF file\cr
 	}
}

\subsection{Advanced}{
  \tabular{ll}{
   \code{\link{writeStart}} \tab Open a file for writing \cr
   \code{\link{writeValues}} \tab Write some values  \cr
   \code{\link{writeStop}} \tab Close the file after writing \cr
    --------------------------- \tab ------------------------------------------------------------------------------------------ \cr
}
}
}	

\section{XI. Miscellaneous SpatRaster methods}{

\tabular{ll}{
    \code{\link{terraOptions}} \tab Show, set, or get session options, mostly to control memory use and to set write options\cr
    \code{\link{sources}} \tab Show the data sources of a SpatRaster \cr
    \code{\link{tmpFiles}} \tab Show or remove temporary files \cr
    \code{\link{mem_info}} \tab memory needs and availability \cr
    \code{\link{readStart}} \tab Open file connections for efficient multi-chunk reading \cr
    \code{\link{readStop}} \tab Close file connections \cr
    \code{\link{inMemory}} \tab Are the cell values in memory? \cr
}
}


\section{}{
---------------------------------------------------------------------------------------------------------------------

\bold{SpatRasterDataSet}

---------------------------------------------------------------------------------------------------------------------}


\section{XII. SpatRasterDataset}{

A SpatRasterDataset contains SpatRaster objects that are sub-datasets for the same area. They all have the same extent and resolution.

\tabular{ll}{
  \code{\link{sds}} \tab Create a SpatRasterDataset from a file with subdatasets (ncdf or hdf) or from SpatRaster objects \cr
  \code{[} or \code{$} \tab Extract a SpatRaster \cr
  \code{\link{names}} \tab Get the names of the sub-datasets \cr
}
}


\section{}{
---------------------------------------------------------------------------------------------------------------------

\bold{SpatVector}

---------------------------------------------------------------------------------------------------------------------}


\section{XIII. Create SpatVector objects}{
\tabular{ll}{
  \code{\link{vect}} \tab Create a SpatVector from a file (for example a "shapefile") or from another object\cr
  \code{rbind} \tab append SpatVectors of the same geometry type\cr
  \code{\link{unique}} \tab remove duplicates \cr
  \code{\link{na.omit}} \tab remove empty geometries and/or fields that are \code{NA} \cr
  \code{\link{project}} \tab Project a SpatVector to a different coordinate reference system \cr
  \code{\link{writeVector}} \tab Write SpatVector data to disk \cr
  \code{\link{centroids}} \tab Get the centroids of a SpatVector\cr
  \code{\link{voronoi}} \tab Voronoi diagram \cr
  \code{\link{delauny}} \tab Delauny triangles\cr
  \code{\link{convHull}} \tab Compute the convex hull of a SpatVector \cr  
  \code{\link{fillHoles}}\tab Remove or extract holes from polygons\cr
  --------------------------- \tab ------------------------------------------------------------------------------------------ \cr  
}}


\section{XIV. Properties of SpatVector objects}{
\tabular{ll}{
  \code{\link{geom}} \tab returns the geometries as matrix or WKT\cr
  \code{\link{crds}} \tab returns the coordinates as a matrix\cr
  \code{\link{linearUnits}} \tab returns the linear units of the crs (in meter)\cr
  \code{\link{ncol}}\tab The number of columns (of the attributes)\cr
  \code{\link{nrow}} \tab The number of rows (of the geometries and attributes)\cr
  \code{\link{names}} \tab Get or set the layer names \cr	
  \code{\link{ext}} \tab Get the extent (minimum and maximum x and y coordinates ("bounding box") \cr
  \code{\link{crs}} \tab The coordinate reference system (map projection) \cr	
  \code{\link{is.lonlat}} \tab Test if an object has (or may have) a longitude/latitude coordinate reference system\cr
  --------------------------- \tab ------------------------------------------------------------------------------------------ \cr  
}}


\section{XV. Geometric queries}{
\tabular{ll}{
  \code{\link{adjacent}} \tab find adjacent polygons\cr
  \code{\link{expanse}} \tab computes the area covered by polygons\cr
  \code{\link{nearby}} \tab find nearby geometries\cr
  \code{\link{nearest}} \tab find the nearest geometries\cr
  \code{\link{relate}} \tab geometric relationships such as "intersects", "overlaps", and "touches"\cr
  \code{\link{perim}} \tab computes the length of the perimeter of polygons, and the length of lines\cr
  --------------------------- \tab ------------------------------------------------------------------------------------------ \cr  
}}


\section{XVI. Geometric operations}{

\tabular{ll}{
  \code{\link{erase}} or "-" \tab erase (parts of) geometries\cr
  \code{\link{intersect}} or "*"  \tab intersect geometries\cr
  \code{\link{union}} or "+" \tab Merge geometries\cr
  \code{\link{cover}} \tab update polygons\cr
  \code{\link{symdif}} \tab symmetrical difference of two polygons \cr 
  \code{\link{aggregate}} \tab dissolve smaller polygons into larger ones \cr
  \code{\link{buffer}} \tab buffer geometries \cr
  \code{\link{disagg}} \tab split multi-geometries into separate geometries \cr
  \code{\link{crop}} \tab clip geometries using a rectangle (SpatExtent) or SpatVector \cr
  --------------------------- \tab ------------------------------------------------------------------------------------------ \cr  
}}


\section{XVII. SpatVector attributes}{
We use the term "attributes" for the tabular data (data.frame) associated with vector geometries.

\tabular{ll}{
  \code{\link{extract}} \tab spatial queries between SpatVector and SpatVector (e.g. point in polygons) \cr
  \code{\link{sel}} \tab select - interactively select geometries\cr
  \code{\link{click}} \tab identify attributes by clicking on a map\cr
  \code{\link{merge}} \tab Join a table with a SpatVector \cr
  \code{\link{as.data.frame}} \tab get attributes as a data.frame\cr
  \code{\link{values}} \tab Get the attributes of a SpatVector \cr
  \code{\link{values<-}} \tab Set new attributes to the geometries of a SpatRaster \cr
  
  --------------------------- \tab ------------------------------------------------------------------------------------------ \cr  
}}


\section{XVIII. Change geometries (for display, experimentation)}{

\tabular{ll}{
  \code{\link{shift}} \tab change the position geometries by shifting their coordinates in horizontal and/or vertical direction\cr
  \code{\link{spin}}\tab rotate geometries around an origin\cr
  \code{\link{rescale}} \tab shrink (or expand) geometries, for example to make an inset map \cr
  \code{\link{flip}} \tab flip geometries vertically or horizontally\cr
  \code{\link{t}} \tab transpose geometries (switch x and y)\cr

}}


\section{XIX. Geometry properties and topology}{

\tabular{ll}{
  \code{\link{width}} \tab the minimum diameter of the geometries \cr
  \code{\link{clearance}} \tab the minimum clearance of the geometries \cr
  \code{\link{sharedPaths}} \tab shared paths (arcs) between line or polygon geometries\cr
  \code{\link{simplify}} \tab simplify geometries\cr
  \code{\link{gaps}} \tab find gaps between polygon geometries \cr
  \code{\link{fillHoles}} \tab get or remove the polygon holes\cr
  \code{\link{makeNodes}} \tab create nodes on lines \cr
  \code{\link{mergeLines}} \tab connect lines to form polygons  \cr
  \code{\link{removeDupNodes}} \tab remove duplicate nodes in geometries and optionally rounds the coordinates  \cr
  \code{\link{snap}} \tab make boundaries of geometries identical if they are very close to each other  \cr
  \code{\link{erase} (single argument)} \tab remove parts of geometries that overlap \cr
  \code{\link{union} (single argument)} \tab create new polygons such that there are no overlapping polygons \cr
}}


\section{}{
---------------------------------------------------------------------------------------------------------------------

\bold{Spat* Collections}

---------------------------------------------------------------------------------------------------------------------}


\section{XX. Collections}{
A SpatRasterCollection is a vector of SpatRaster objects. Unlike for a SpatRasterDataset, there the extent and resolution of the SpatRasters do not need to match each other. A SpatVectorCollection is a vector of SpatVector objects. 

\tabular{ll}{
\code{\link{svc}} \tab create a SpatRasterCollection from a set of SpatRaster objects\cr
\code{\link{length}} \tab how many SpatRasters does the SpatRasterCollection have?\cr
\code{\link{[}} \tab extract a SpatRastert\cr
}}



\section{}{
---------------------------------------------------------------------------------------------------------------------

\bold{SpatExtent}

---------------------------------------------------------------------------------------------------------------------}



\section{XXI. SpatExtent}{

  \tabular{ll}{
    \code{\link{ext}} \tab Create a SpatExtent object. For example to \code{\link{crop}} a Spatial dataset\cr
    \code{\link{intersect}} \tab Intersect two SpatExtent objects, same as \code{-}  \cr
    \code{\link{union}} \tab Combine two SpatExtent objects, same as \code{+} \cr
    \code{\link{Math-methods}} \tab round/floor/ceiling of a SpatExtent \cr			
    \code{\link{align}} \tab Align a SpatExtent with a SpatRaster \cr
    \code{\link{draw}} \tab Create a SpatExtent by drawing it on top of a map (plot) \cr		
  }
}




\section{}{
---------------------------------------------------------------------------------------------------------------------

\bold{General methods}

---------------------------------------------------------------------------------------------------------------------}


\section{XXII. Data type conversion}{

You can coerce SpatRasters to Raster* objects after loading the \code{raster} package with \code{as(object, "Raster")}, or \code{raster(object)} or \code{brick(object)} or \code{stack(object)}
  
  \tabular{ll}{
    \code{\link{rast}} \tab SpatRaster from matrix and other objects\cr
    \code{\link{vect}} \tab SpatVector from \code{sf} or \code{Spatial*} vector data\cr
    \code{sf::st_as_sf} \tab sf object from SpatVector\cr
    \code{\link{rasterize}} \tab Rasterizing points, lines or polygons\cr
    \code{\link{as.points}} \tab Create points from a SpatRaster or SpatVector \cr
    \code{\link{as.lines}} \tab Create points from a SpatRaster or SpatVector\cr
    \code{\link{as.polygons}} \tab Create polygons from a SpatRaster \cr
    \code{\link{as.contour}}  \tab Contour lines from a SpatRaster \cr
  --------------------------- \tab ------------------------------------------------------------------------------------------ \cr
  }
}

\section{XXIII. Plotting}{

\subsection{Maps}{
  \tabular{ll}{
    \code{\link{plot}} \tab Plot a SpatRaster or SpatVector. The main method to create a map \cr
	\code{\link{points}} \tab Add points to a map\cr
	\code{\link{lines}} \tab Add lines to a map\cr
	\code{\link{polys}} \tab Add polygons to a map \cr
    \code{\link{text}} \tab Add text (such as the values of a SpatRaster or SpatVector) to a map \cr
    \code{\link{image}} \tab Alternative to plot to make a map with a SpatRaster \cr
    \code{\link{plotRGB}} \tab Combine three layers (red, green, blue channels) into a single "real color" plot \cr
    \code{\link{RGB2col}} \tab Combine three layers (red, green, blue channels) into a single layer with a color-table \cr
    \code{\link{dots}} \tab Make a dot-density map \cr
    \code{\link{cartogram}} \tab Make a cartogram \cr
    \code{\link{persp}} \tab Perspective plot of a SpatRaster \cr
    \code{\link{contour}} \tab Contour plot or filled-contour plot of a SpatRaster \cr
    \code{\link{inset}} \tab Add a small inset (overview) map\cr
    \code{\link{sbar}} \tab Add a scalebar\cr
	}
}

\subsection{Interacting with a map}{
  \tabular{ll}{
    \code{\link{zoom}} \tab Zoom in to a part of a map by drawing a bounding box on it \cr
    \code{\link{click}} \tab Query values of SpatRaster or SpatVector by clicking on a map \cr
    \code{\link{sel}} \tab Select a spatial subset of a SpatRaster or SpatVector by drawing on a map\cr
    \code{\link{draw}} \tab Create a SpatExtent or SpatVector by drawing on a map \cr
	}
}
	
\subsection{Other plots}{
  \tabular{ll}{
    \code{\link{plot}} \tab x-y scatter plot of the values of (a sample of) the layers of two SpatRaster objects\cr
    \code{\link{hist}} \tab Histogram of SpatRaster values \cr
    \code{\link{barplot}} \tab Bar plot of a SpatRaster \cr
    \code{\link{density}} \tab Density plot of SpatRaster values \cr
    \code{\link{pairs}} \tab Pairs plot for layers in a SpatRaster \cr
    \code{\link{boxplot}} \tab Box plot of the values of a SpatRaster\cr
	}
}
}



\section{}{
---------------------------------------------------------------------------------------------------------------------

\bold{Comparison with the raster package}

---------------------------------------------------------------------------------------------------------------------}


\section{XXIV. New method names}{


\code{terra} has a single class \code{SpatRaster} for which \code{raster} has three (\code{RasterLayer, RasterStack, RasterBrick}). Likewise there is a single class for vector data \code{SpatVector} that replaces six \code{Spatial*} classes. Most method names are the same, but note the following important differences in methods names with the \code{raster} package


\tabular{ll}{
\bold{raster package} \tab \bold{terra package}\cr
\code{raster, brick, stack}\tab\code{\link{rast}}\cr
\code{rasterFromXYZ}\tab\code{\link{rast}( , type="xyz")}\cr
\code{stack, addLayer} (combining Raster* objects or files)\tab\code{\link{c}}\cr
\code{addLayer} \tab\code{\link{add<-}}\cr
\code{area} \tab\code{\link{cellSize}}\cr
\code{disaggregate} \tab\code{\link{disagg}}\cr
\code{extent}\tab\code{\link{ext}}\cr
\code{calc}\tab\code{\link{app}}\cr
\code{overlay}\tab\code{\link{lapp}}\cr
\code{stackApply}\tab\code{\link{tapp}}\cr
\code{nlayers}\tab\code{\link{nlyr}}\cr
\code{NAvalue}\tab\code{\link{NAflag}}\cr
\code{stackSelect}\tab\code{\link{selectRange}}\cr
\code{reclassify, subs, cut}\tab\code{\link{classify}}\cr
\code{cellStats}\tab\code{\link{global}}\cr
\code{projectRaster}\tab\code{\link{project}}\cr
\code{dropLayer}\tab\code{\link{subset}}\cr
\code{isLonLat, isGlobalLonLat, couldBeLonLat}\tab\code{\link{is.lonlat}}\cr
\code{shapefile}\tab\code{\link{vect}}\cr
\code{gridDistance, distanceFromPoints} \tab \code{\link{distance}}\cr
\code{drawExtent, drawPoly, drawLine} \tab \code{\link{draw}}\cr
\code{compareRaster}\tab\code{\link{compareGeom}}\cr
\code{sampleRandom, sampleRegular}\tab\code{\link{spatSample}}\cr
\code{rasterToPoints}\tab\code{\link{as.points}}\cr
\code{rasterToPolygons}\tab\code{\link{as.polygons}}\cr
\code{cellFromLine, cellFromPolygon, cellsFromExtent}\tab\code{\link{cells}}\cr
\code{layerize}\tab\code{\link{segregate}}\cr
\code{clump}\tab\code{\link{patches}}\cr
}
}

\section{XXV. Changed behavior}{

Also note that even if function names are the same in \code{terra} and \code{raster}, their output can be different. In most cases this was done to get more consistency in the returned values (and thus fewer errors in the downstream code that uses them). It other cases it simply seemed better. Here are some examples:

\tabular{ll}{
\code{\link{as.polygons}}\tab By default, \code{terra} returns dissolved polygons\cr
\code{\link{extract}}\tab By default, \code{terra} returns a matrix, with the first column the sequential ID of the vectors. \cr
\tab \code{raster} returns a list (for lines or polygons) or a matrix (for points, but without the ID\cr
\tab column. You can use \code{list=TRUE} to get the results as a list\cr
\code{\link{values}}\tab \code{terra} always returns a matrix. \code{raster} returns a vector for a \code{RasterLayer}\cr
\code{\link{Summary-methods}}\tab With \code{raster}, \code{mean(x, y)} and \code{mean(stack(x, y)} return the same result, a single\cr
\tab layer with the mean of all cell values. This is also what \code{terra} returns with \cr
\tab \code{mean(c(x, y))}, but with \code{mean(x, y)} the parallel mean is returned -- that is, the\cr
\tab computation is done layer-wise, and the number of layers in the output is the same as\cr
\tab  that of \code{x} and \code{y} (or the larger of the two if they are not the same). This affects \cr
\tab all summary functions (\code{sum}, \code{mean}, \code{median}, \code{which.min}, \code{which.max}, \code{min}, \code{max},\cr
\tab \code{prod}, \code{any}, \code{all}, \code{stdev}), except \code{range}, which is not implemented for this case \cr
\tab (you can use \code{min} and \code{max} instead) \cr
}
}


\section{}{
---------------------------------------------------------------------------------------------------------------------}

\section{Author}{
Except where indicated otherwise, the methods and functions in this package were written by Robert Hijmans. The configuration scripts were written by Roger Bivand for \code{rgdal} and \code{sf}. Some of the C++ code for GDAL/GEOS was adapted from code by Edzer Pebesma for \code{sf}. The progress bar code is by Karl Forner (RcppProgress). Jeroen Ooms provided the compiled GDAL and GEOS libraries for installation on windows}

\section{Acknowledgments}{
This package is an attempt to climb on the shoulders of giants (GDAL, PROJ, GEOS, NCDF, GeographicLib, Rcpp, R). Many people have contributed by asking questions or \href{https://github.com/rspatial/terra}{raising issues}. The feedback and suggestions by Kendon Bell, Jean-Luc Dupouey, Krzysztof Dyba, Jakub Nowosad, Gerald Nelson, and Michael Sumner have been especially helpful.
}

\keyword{ package }
\keyword{ spatial }

