% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tar_cue_age.R
\name{tar_cue_age}
\alias{tar_cue_age}
\title{Cue to run a target when the last output reaches a certain age}
\usage{
tar_cue_age(
  name,
  age,
  command = TRUE,
  depend = TRUE,
  format = TRUE,
  repository = TRUE,
  iteration = TRUE,
  file = TRUE
)
}
\arguments{
\item{name}{Symbol, name of the target.}

\item{age}{A \code{difftime} object of length 1, such as
\code{as.difftime(3, units = "days")}. If the target's output data
files are older than \code{age} (according to the most recent
time stamp over all the target's output files)
then the target will rerun.
On the other hand, if at least one data file is
younger than \code{Sys.time() - age}, then the ordinary
invalidation rules apply, and the target may or not rerun.
If you want to force the target to run every 3 days,
for example, set \code{age = as.difftime(3, units = "days")}.}

\item{command}{Logical, whether to rerun the target if command changed
since last time.}

\item{depend}{Logical, whether to rerun the target if the value of one
of the dependencies changed.}

\item{format}{Logical, whether to rerun the target if the user-specified
storage format changed. The storage format is user-specified through
\code{\link[targets:tar_target]{tar_target()}} or \code{\link[targets:tar_option_set]{tar_option_set()}}.}

\item{repository}{Logical, whether to rerun the target if the user-specified
storage repository changed. The storage repository is user-specified
through \code{\link[targets:tar_target]{tar_target()}} or \code{\link[targets:tar_option_set]{tar_option_set()}}.}

\item{iteration}{Logical, whether to rerun the target if the user-specified
iteration method changed. The iteration method is user-specified through
\code{\link[targets:tar_target]{tar_target()}} or \code{\link[targets:tar_option_set]{tar_option_set()}}.}

\item{file}{Logical, whether to rerun the target if the file(s) with the
return value changed or at least one is missing.}
}
\value{
A cue object. See the "Cue objects" section for background.
}
\description{
\code{tar_cue_age()} creates a cue object to
rerun a target if the most recent output data becomes old enough.
The age of the target is determined by \code{targets::tar_timestamp()},
and the way the time stamp is calculated is explained
in the Details section of the help file of that function.
}
\details{
\code{tar_cue_age()} uses the time stamps from \code{tar_meta()$time}.
If no time stamp is recorded, the cue defaults to the ordinary
invalidation rules (i.e. \code{mode = "thorough"} in \code{targets::tar_cue()}).
}
\section{Dynamic branches at regular time intervals}{

Time stamps are not recorded for whole dynamic targets,
so \code{tar_age()} is not a good fit for dynamic branching.
To invalidate dynamic branches at regular intervals,
it is recommended to use \code{targets::tar_older()} in combination
with \code{targets::tar_invalidate()} right before calling \code{tar_make()}.
For example,
\code{tar_invalidate(any_of(tar_older(Sys.time - as.difftime(1, units = "weeks"))))} # nolint
invalidates all targets more than a week old. Then, the next \code{tar_make()}
will rerun those targets.
}

\section{Cue objects}{

A cue object is an object generated by \code{targets::tar_cue()},
\code{tarchetypes::tar_cue_force()}, or similar. It is
a collection of decision rules that decide when a target
is invalidated/outdated (e.g. when \code{tar_make()} or similar
reruns the target). You can supply these cue objects to the
\code{tar_target()} function or similar. For example,
\code{tar_target(x, run_stuff(), cue = tar_cue(mode = "always"))}
is a target that always calls \code{run_stuff()} during \code{tar_make()}
and always shows as invalidated/outdated in \code{tar_outdated()},
\code{tar_visnetwork()}, and similar functions.
}

\examples{
if (identical(Sys.getenv("TAR_LONG_EXAMPLES"), "true")) {
targets::tar_dir({ # tar_dir() runs code from a temporary directory.
targets::tar_script({
  library(tarchetypes)
  list(
    targets::tar_target(
      data,
      data.frame(x = seq_len(26)),
      cue = tarchetypes::tar_cue_age(
        name = data,
        age = as.difftime(0.5, units = "secs")
      )
    )
  )
})
targets::tar_make()
Sys.sleep(0.6)
targets::tar_make()
})
}
}
\seealso{
Other cues: 
\code{\link{tar_age}()},
\code{\link{tar_cue_age_raw}()},
\code{\link{tar_cue_force}()},
\code{\link{tar_cue_skip}()}
}
\concept{cues}
