% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom_superberrorbar.R
\name{geom_superberrorbar}
\alias{geom_superberrorbar}
\title{geom_superberrorbar for expanded error bar displays}
\usage{
geom_superberrorbar(
  mapping = NULL,
  data = NULL,
  stat = "identity",
  position = "identity",
  ...,
  na.rm = FALSE,
  orientation = NA,
  show.legend = NA,
  inherit.aes = TRUE
)
}
\arguments{
\item{mapping}{(as usual) see geom_errorbar}

\item{data}{(as usual) see geom_errorbar}

\item{stat}{(as usual) see geom_errorbar}

\item{position}{(as usual) see geom_errorbar}

\item{...}{all additional parameters are sent to the underlying geom_path. Includes
\itemize{
\item \code{pointing} either "up", "down" or "both" up and down (default is "both");
\item \code{direction}  "left", "right" or "both" (Default is "both")
\item \code{tipformat}  "single", "double" or "triple" to add additional
marker lines to the tips (default is "single")
\item \code{tipgap} The spacing between the markers when "double" or "triple" is used (default 0.1)
\item \code{vcolour} for the vertical part of the error bar
\item \code{wcolour} if specified, for the second half of the vertical part of the error bar.
}}

\item{na.rm}{(as usual) see \code{geom_errorbar()}}

\item{orientation}{(as usual) see \code{geom_errorbar()}}

\item{show.legend}{(as usual) see \code{geom_errorbar()}}

\item{inherit.aes}{(as usual) see \code{geom_errorbar()}}
}
\value{
a layer containing error bars in a ggplot object
}
\description{
\code{geom_superberrorbar()} is a geom for ggplots; it is based on
the original \code{geom_errorbar()} (and is totally compatible with it) but
expands this geom in four different ways. First, it is possible to
decide the error bar tips \code{direction} which can be unidirectional, pointing to
the "left" or to the "right" or go in "both" directions.
Second, it is possible set \code{tipformat} to "double" or "triple" the horizontal marks
at the extremities of the error bar, with a \code{tipgap} of your liking.
Third, an additional characteristic is \code{vcolour} to set a different colour
for the vertical part of the error bar or the pair
\code{vcolour} and \code{wcolour} for the top half and bottom half of the vertical
error bar. The colour(s) can also be "NA" to
have it invisible. Lastly, the error bar can be \code{pointing} "up" and "down"
or go in "both" (the default)
}
\examples{
library(superb) # to import the geom_superberrorbar
library(ggplot2)

# let's have a fake data frame
dta <- data.frame(grp = c(1,2,3), center=c(1,2,3), width = c(1,1,1.5) )

# an example with none of the new features = a regular error bar
ggplot(dta, aes(ymin=center-width, ymax=center+width, x = grp ) ) +
  geom_superberrorbar()

# an example with left-pointing error bars
ggplot(dta, aes(ymin=center-width, ymax=center+width, x = grp ) ) +
  geom_superberrorbar(direction="left", width = 0.1)

# an example with doubled-tipped error bar and the default tipgap
ggplot(dta, aes(ymin=center-width, ymax=center+width, x = grp ) ) +
  geom_superberrorbar(tipformat = "double", width = 0.1)

# an example with left-pointing tripled-tip error bars with small gaps
ggplot(dta, aes(ymin=center-width, ymax=center+width, x = grp ) ) +
  geom_superberrorbar(tipformat = "triple", width= 0.1, tipgap = 0.04, direction = "left")

# an example with unidirectional error bars (here "up" bars)
ggplot(dta, aes(y= center, ymin=center-width, ymax=center+width, x = grp ) ) +
  geom_bar(stat="identity", fill = "yellow") + 
  geom_superberrorbar(pointing = "up")

# a final example with two-coloured, left-pointing tripled-tip error bars with small gaps
ggplot(dta, aes(ymin=center-width, ymax=center+width, x = grp ) ) +
  geom_superberrorbar(tipformat = "triple", width= 0.1, tipgap = 0.04, direction = "left",
           colour = "black", vcolour = "orange")

# This new geom is integrated inside superb() so that you can vary the 
# error bar shapes. Let's see examples:

# using GRD to generate random data with a moderate effect
options(superb.feedback = 'none') # shut down 'warnings' and 'design' interpretation messages
test <- GRD(SubjectsPerGroup  = 20,
		   WSFactors = "Moment(5)", 
            Effects = list("Moment" = extent(10) ),
            Population = list(mean = 100, stddev = 15, rho = 0.8) ) 

ornate = list(
        labs(title =paste("(left)            95\% confidence intervals",
                        "\n(right)          99\% confidence intervals",
                        "\n(center, up) 99.9\% confidence intervals")),
        xlab("Moment"), ylab("Score"),
        coord_cartesian( ylim = c(85,125) )
)

plt1 <- superb(
            crange(DV.1, DV.5) ~ ., 
            test, 
            WSFactors = "Moment(5)",  
            adjustments=list(purpose = "difference", decorrelation = "CA"), 
            errorbarParams = list(direction = "left", color="purple", 
                                  width = 0.2, position = position_nudge(-0.05) ),
            gamma     = 0.95,
            plotLayout = "line" ) + ornate
plt2 <- superb( 
            crange(DV.1, DV.5) ~ ., 
            test, 
            WSFactors = "Moment(5)",  
            adjustments=list(purpose = "difference", decorrelation = "CA"), 
            errorbarParams = list(direction = "right", tipgap = 0.40, tipformat = "double", 
                                  width = 0.2, position = position_nudge(+0.05) ),
            gamma     = 0.99,
            plotLayout = "line" ) + ornate 
plt3 <- superb( 
            crange(DV.1, DV.5) ~ ., 
            test, 
            WSFactors = "Moment(5)",  
            adjustments=list(purpose = "difference", decorrelation = "CA"), 
            errorbarParams = list(direction = "both", tipformat = "single", pointing="up", 
                                  width = 0.2, position = position_nudge(0) ),
            gamma     = 0.999,
            plotLayout = "line" ) + ornate 

# transform the ggplots into "grob" so that they can be manipulated
plt1 <- ggplotGrob(plt1)
plt2 <- ggplotGrob(plt2 + makeTransparent() )
plt3 <- ggplotGrob(plt3 + makeTransparent() )

# put the grobs onto an empty ggplot 
ggplot() + 
    annotation_custom(grob=plt1) + 
    annotation_custom(grob=plt2) + 
    annotation_custom(grob=plt3)


# all of them as aesthetics
set.seed(1)
library(dplyr)
dat <- data.frame(Trial = c(rep("Pre",9),rep("Post",9)), 
                     Time = rep.int(seq(0,120,15),2), 
                     var = c(rnorm(9,15,2),rnorm(9,22,2)),
                     var_sd = c(rnorm(18,3,1)))
dat <- mutate(dat, point = ifelse(Trial == "Pre","down","up"))
dat <- mutate(dat, direc = ifelse(Trial == "Pre","left","right"))
dat <- mutate(dat, tipfo = ifelse(Trial == "Pre","double","triple"))
dat <- mutate(dat, vcolo = ifelse(Trial == "Pre","red","blue"))
   
ggplot(data = dat, 
       aes(x = Time, y = var, group = Trial)) +
   geom_line(aes(linetype = Trial)) +  
   geom_point(aes(shape= Trial, fill = Trial), size=2) +
   geom_superberrorbar(aes(ymin=var-var_sd, 
                           ymax=var+var_sd,
                           direction = direc,
                           pointing = point, 
                           wcolour = vcolo,  
                           vcolour = "green",
                           tipgap = 0.40,
                           tipformat = tipfo
       ), 
       width = 4)

}
