% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wrappers.R
\name{svsample}
\alias{svsample}
\alias{svtsample}
\alias{svlsample}
\alias{svtlsample}
\alias{svsample2}
\title{Markov Chain Monte Carlo (MCMC) Sampling for the Stochastic Volatility (SV)
Model}
\usage{
svsample(
  y,
  draws = 10000,
  burnin = 1000,
  designmatrix = NA,
  priormu = c(0, 100),
  priorphi = c(5, 1.5),
  priorsigma = 1,
  priornu = 0,
  priorrho = NA,
  priorbeta = c(0, 10000),
  priorlatent0 = "stationary",
  priorspec = NULL,
  thin = 1,
  thinpara = thin,
  thinlatent = thin,
  keeptime = "all",
  quiet = FALSE,
  startpara = NULL,
  startlatent = NULL,
  parallel = c("no", "multicore", "snow"),
  n_cpus = 1L,
  cl = NULL,
  n_chains = 1L,
  print_progress = "automatic",
  expert = NULL,
  ...
)

svtsample(
  y,
  draws = 10000,
  burnin = 1000,
  designmatrix = NA,
  priormu = c(0, 100),
  priorphi = c(5, 1.5),
  priorsigma = 1,
  priornu = 0.1,
  priorrho = NA,
  priorbeta = c(0, 10000),
  priorlatent0 = "stationary",
  priorspec = NULL,
  thin = 1,
  thinpara = thin,
  thinlatent = thin,
  keeptime = "all",
  quiet = FALSE,
  startpara = NULL,
  startlatent = NULL,
  parallel = c("no", "multicore", "snow"),
  n_cpus = 1L,
  cl = NULL,
  n_chains = 1L,
  print_progress = "automatic",
  expert = NULL,
  ...
)

svlsample(
  y,
  draws = 20000,
  burnin = 2000,
  designmatrix = NA,
  priormu = c(0, 100),
  priorphi = c(5, 1.5),
  priorsigma = 1,
  priornu = 0,
  priorrho = c(4, 4),
  priorbeta = c(0, 10000),
  priorlatent0 = "stationary",
  priorspec = NULL,
  thin = 1,
  thinpara = thin,
  thinlatent = thin,
  keeptime = "all",
  quiet = FALSE,
  startpara = NULL,
  startlatent = NULL,
  parallel = c("no", "multicore", "snow"),
  n_cpus = 1L,
  cl = NULL,
  n_chains = 1L,
  print_progress = "automatic",
  expert = NULL,
  ...
)

svtlsample(
  y,
  draws = 20000,
  burnin = 2000,
  designmatrix = NA,
  priormu = c(0, 100),
  priorphi = c(5, 1.5),
  priorsigma = 1,
  priornu = 0.1,
  priorrho = c(4, 4),
  priorbeta = c(0, 10000),
  priorlatent0 = "stationary",
  priorspec = NULL,
  thin = 1,
  thinpara = thin,
  thinlatent = thin,
  keeptime = "all",
  quiet = FALSE,
  startpara = NULL,
  startlatent = NULL,
  parallel = c("no", "multicore", "snow"),
  n_cpus = 1L,
  cl = NULL,
  n_chains = 1L,
  print_progress = "automatic",
  expert = NULL,
  ...
)

svsample2(
  y,
  draws = 10000,
  burnin = 1000,
  designmatrix = NA,
  priormu = c(0, 100),
  priorphi = c(5, 1.5),
  priorsigma = 1,
  priornu = 0,
  priorrho = NA,
  priorbeta = c(0, 10000),
  priorlatent0 = "stationary",
  thinpara = 1,
  thinlatent = 1,
  keeptime = "all",
  quiet = FALSE,
  startpara = NULL,
  startlatent = NULL
)
}
\arguments{
\item{y}{numeric vector containing the data (usually log-returns), which
must not contain zeros. Alternatively, \code{y} can be an \code{svsim}
object. In this case, the returns will be extracted and a message is signalled.}

\item{draws}{single number greater or equal to 1, indicating the number of
draws after burn-in (see below). Will be automatically coerced to integer.
The default value is 10000.}

\item{burnin}{single number greater or equal to 0, indicating the number of
draws discarded as burn-in. Will be automatically coerced to integer. The
default value is 1000.}

\item{designmatrix}{regression design matrix for modeling the mean. Must
have \code{length(y)} rows. Alternatively, \code{designmatrix} may be a
string of the form \code{"arX"}, where \code{X} is a nonnegative integer. To
fit a constant mean model, use \code{designmatrix = "ar0"} (which is
equivalent to \code{designmatrix = matrix(1, nrow = length(y))}). To fit an
AR(1) model, use \code{designmatrix = "ar1"}, and so on. If some elements of
\code{designmatrix} are \code{NA}, the mean is fixed to zero (pre-1.2.0
behavior of \pkg{stochvol}).}

\item{priormu}{numeric vector of length 2, indicating mean and standard
deviation for the Gaussian prior distribution of the parameter \code{mu},
the level of the log-volatility. The default value is \code{c(0, 100)},
which constitutes a practically uninformative prior for common exchange rate
datasets, stock returns and the like.}

\item{priorphi}{numeric vector of length 2, indicating the shape parameters
for the Beta prior distribution of the transformed parameter
\code{(phi + 1) / 2}, where \code{phi} denotes the persistence of the
log-volatility. The default value is \code{c(5, 1.5)}, which constitutes a
prior that puts some belief in a persistent log-volatility but also
encompasses the region where \code{phi} is around 0.}

\item{priorsigma}{single positive real number, which stands for the scaling
of the transformed parameter \code{sigma^2}, where \code{sigma} denotes the
volatility of log-volatility. More precisely, \code{sigma^2 ~ priorsigma *
chisq(df = 1)}. The default value is \code{1}, which constitutes a
reasonably vague prior for many common exchange rate datasets, stock returns
and the like.}

\item{priornu}{single non-negative number, indicating the rate parameter
for the exponential prior distribution of the parameter; can be \code{Inf}
\code{nu}, the degrees-of-freedom parameter of the conditional innovations
t-distribution. The default value is \code{0}, fixing the
degrees-of-freedom to infinity. This corresponds to conditional standard
normal innovations, the pre-1.1.0 behavior of \pkg{stochvol}.}

\item{priorrho}{either \code{NA} for the no-leverage case or a numeric
vector of length 2 that specify the beta prior distribution for
\code{(rho+1)/2}}

\item{priorbeta}{numeric vector of length 2, indicating the mean and
standard deviation of the Gaussian prior for the regression parameters. The
default value is \code{c(0, 10000)}, which constitutes a very vague prior
for many common datasets. Not used if \code{designmatrix} is \code{NA}.}

\item{priorlatent0}{either a single non-negative number or the string
\code{'stationary'} (the default, also the behavior before version 1.3.0).
When \code{priorlatent0} is equal to \code{'stationary'}, the stationary
distribution of the latent AR(1)-process is used as the prior for the
initial log-volatility \code{h_0}. When \code{priorlatent0} is equal to a
number \eqn{B}, we have \eqn{h_0 \sim N(\mu, B\sigma^2)} a priori.}

\item{priorspec}{in case one needs different prior distributions than the
ones specified by \code{priormu}, \code{...}, \code{priorrho}, a \code{priorspec}
object can be supplied here. A smart constructor for this usecase is
\link{specify_priors}.}

\item{thin}{single number greater or equal to 1, coercible to integer.
Every \code{thinpara}th parameter and latent draw is kept and returned. The default
value is 1, corresponding to no thinning of the parameter draws i.e. every
draw is stored.}

\item{thinpara}{single number greater or equal to 1, coercible to integer.
Every \code{thinpara}th parameter draw is kept and returned. The default
value is \code{thin}.}

\item{thinlatent}{single number greater or equal to 1, coercible to integer.
Every \code{thinlatent}th latent variable draw is kept and returned. The
default value is \code{thin}}

\item{keeptime}{Either 'all' (the default) or 'last'. Indicates which latent
volatility draws should be stored.}

\item{quiet}{logical value indicating whether the progress bar and other
informative output during sampling should be omitted. The default value is
\code{FALSE}, implying verbose output.}

\item{startpara}{\emph{optional} named list, containing the starting values
for the parameter draws. If supplied, \code{startpara} must contain three
elements named \code{mu}, \code{phi}, and \code{sigma}, where \code{mu} is
an arbitrary numerical value, \code{phi} is a real number between \code{-1}
and \code{1}, and \code{sigma} is a positive real number. Moreover, if
\code{priornu} is not \code{0}, \code{startpara} must also contain an
element named \code{nu} (the degrees of freedom parameter for the
t-innovations). The default value is equal to the prior mean. 
In case of parallel execution with \code{cl} provided, \code{startpara} can be a list of
named lists that initialize the parallel chains.}

\item{startlatent}{\emph{optional} vector of length \code{length(y)},
containing the starting values for the latent log-volatility draws. The
default value is \code{rep(-10, length(y))}.
In case of parallel execution with \code{cl} provided, \code{startlatent} can be a list of
named lists that initialize the parallel chains.}

\item{parallel}{\emph{optional} one of \code{"no"} (default), \code{"multicore"}, or \code{"snow"},
indicating what type of parallellism is to be applied. Option
\code{"multicore"} is not available on Windows.}

\item{n_cpus}{\emph{optional} positive integer, the number of CPUs to be used in case of
parallel computations. Defaults to \code{1L}. Ignored if parameter
\code{cl} is supplied and \code{parallel != "snow"}.}

\item{cl}{\emph{optional} so-called SNOW cluster object as implemented in package
\code{parallel}. Ignored unless \code{parallel == "snow"}.}

\item{n_chains}{\emph{optional} positive integer specifying the number of independent MCMC chains}

\item{print_progress}{\emph{optional} one of \code{"automatic"}, \code{"progressbar"},
or \code{"iteration"}, controls the output. Ignored if \code{quiet} is \code{TRUE}.}

\item{expert}{\emph{optional} named list of expert parameters. For most
applications, the default values probably work best. Interested users are
referred to the literature provided in the References section. If
\code{expert} is provided, it may contain the following named elements:
\itemize{
\item{interweave}{Logical value. If \code{TRUE} (the default),
then ancillarity-sufficiency interweaving strategy (ASIS) is applied
to improve on the sampling efficiency for the parameters.
Otherwise one parameterization is used.}
\item{correct_model_misspecification}{Logical value. If \code{FALSE}
(the default), then auxiliary mixture sampling is used to sample the latent
states. If \code{TRUE}, extra computations are made to correct for model
misspecification either ex-post by reweighting or on-line using a
Metropolis-Hastings step.}
}}

\item{\dots}{Any extra arguments will be forwarded to
\code{\link{updatesummary}}, controlling the type of statistics calculated
for the posterior draws.}
}
\value{
The value returned is a list object of class \code{svdraws} holding
\item{para}{\code{mcmc.list} object containing the \emph{parameter} draws from
the posterior distribution.}
\item{latent}{\code{mcmc.list} object containing the
\emph{latent instantaneous log-volatility} draws from the posterior
distribution.}
\item{latent0}{\code{mcmc.list} object containing the \emph{latent
initial log-volatility} draws from the posterior distribution.}
\item{tau}{\code{mcmc.list} object containing the \emph{latent variance inflation
factors} for the sampler with conditional t-innovations \emph{(optional)}.}
\item{beta}{\code{mcmc.list} object containing the \emph{regression coefficient}
draws from the posterior distribution \emph{(optional)}.}
\item{y}{the left hand side of the observation equation, usually
the argument \code{y}. In case of an AR(\code{k}) specification, the
first \code{k} elements are removed.}
\item{runtime}{\code{proc_time} object containing the
run time of the sampler.}
\item{priors}{a \code{priorspec} object containing the parameter
values of the prior distributions for \code{mu},
\code{phi}, \code{sigma}, \code{nu}, \code{rho}, and
\code{beta}s, and the variance of specification for \code{latent0}.}
\item{thinning}{\code{list} containing the thinning
parameters, i.e. the arguments \code{thinpara}, \code{thinlatent} and
\code{keeptime}.}
\item{summary}{\code{list} containing a collection of
summary statistics of the posterior draws for \code{para}, \code{latent},
and \code{latent0}.}
\item{meanmodel}{\code{character} containing information about how \code{designmatrix}
was employed.}

To display the output, use \code{print}, \code{summary} and \code{plot}. The
\code{print} method simply prints the posterior draws (which is very likely
a lot of output); the \code{summary} method displays the summary statistics
currently stored in the object; the \code{plot} method
\code{\link{plot.svdraws}} gives a graphical overview of the posterior
distribution by calling \code{\link{volplot}}, \code{\link{traceplot}} and
\code{\link{densplot}} and displaying the results on a single page.
}
\description{
\code{svsample} simulates from the joint posterior distribution of the SV
parameters \code{mu}, \code{phi}, \code{sigma} (and potentially \code{nu} and \code{rho}),
along with the latent log-volatilities \code{h_0,...,h_n} and returns the
MCMC draws. If a design matrix is provided, simple Bayesian regression can
also be conducted.
}
\details{
Functions \code{svtsample}, \code{svlsample}, and \code{svtlsample} are
wrappers around \code{svsample} with convenient default values for the SV
model with t-errors, leverage, and both t-errors and leverage, respectively.

For details concerning the algorithm please see the paper by Kastner and
Frühwirth-Schnatter (2014).
}
\note{
If \code{y} contains zeros, you might want to consider de-meaning your
returns or use \code{designmatrix = "ar0"}.
}
\examples{
###############
# Full examples
###############

# Example 1
## Simulate a short and highly persistent SV process 
sim <- svsim(100, mu = -10, phi = 0.99, sigma = 0.2)

## Obtain 5000 draws from the sampler (that's not a lot)
draws <-
  svsample(sim, draws = 5000, burnin = 100,
           priormu = c(-10, 1), priorphi = c(20, 1.5), priorsigma = 0.2)

## Check out the results
summary(draws)
plot(draws)


\donttest{
# Example 2
## Simulate an asymmetric and conditionally heavy-tailed SV process
sim <- svsim(150, mu = -10, phi = 0.96, sigma = 0.3, nu = 10, rho = -0.3)

## Obtain 10000 draws from the sampler
## Use more advanced prior settings
## Run two parallel MCMC chains
advanced_draws <-
  svsample(sim, draws = 10000, burnin = 5000,
           priorspec = specify_priors(mu = sv_normal(-10, 1),
                                      sigma2 = sv_gamma(0.5, 2),
                                      rho = sv_beta(4, 4),
                                      nu = sv_constant(5)),
           parallel = "snow", n_chains = 2, n_cpus = 2)

## Check out the results
summary(advanced_draws)
plot(advanced_draws)


# Example 3
## AR(1) structure for the mean
data(exrates)
len <- 3000
ahead <- 100
y <- head(exrates$USD, len)

## Fit AR(1)-SVL model to EUR-USD exchange rates
res <- svsample(y, designmatrix = "ar1")

## Use predict.svdraws to obtain predictive distributions
preddraws <- predict(res, steps = ahead)

## Calculate predictive quantiles
predquants <- apply(preddraws$y[[1]], 2, quantile, c(.1, .5, .9))

## Visualize
expost <- tail(head(exrates$USD, len+ahead), ahead)
ts.plot(y, xlim = c(length(y)-4*ahead, length(y)+ahead),
	       ylim = range(c(predquants, expost, tail(y, 4*ahead))))
for (i in 1:3) {
  lines((length(y)+1):(length(y)+ahead), predquants[i,],
        col = 3, lty = c(2, 1, 2)[i])
}
lines((length(y)+1):(length(y)+ahead), expost,
      col = 2)


# Example 4
## Predicting USD based on JPY and GBP in the mean
data(exrates)
len <- 3000
ahead <- 30
## Calculate log-returns
logreturns <- apply(exrates[, c("USD", "JPY", "GBP")], 2,
                    function (x) diff(log(x)))
logretUSD <- logreturns[2:(len+1), "USD"]
regressors <- cbind(1, as.matrix(logreturns[1:len, ]))  # lagged by 1 day

## Fit SV model to EUR-USD exchange rates
res <- svsample(logretUSD, designmatrix = regressors)

## Use predict.svdraws to obtain predictive distributions
predregressors <- cbind(1, as.matrix(logreturns[(len+1):(len+ahead), ]))
preddraws <- predict(res, steps = ahead,
                     newdata = predregressors)
predprice <- exrates[len+2, "USD"] * exp(t(apply(preddraws$y[[1]], 1, cumsum)))

## Calculate predictive quantiles
predquants <- apply(predprice, 2, quantile, c(.1, .5, .9))

## Visualize
priceUSD <- exrates[3:(len+2), "USD"]
expost <- exrates[(len+3):(len+ahead+2), "USD"]
ts.plot(priceUSD, xlim = c(len-4*ahead, len+ahead+1),
	       ylim = range(c(expost, predquants, tail(priceUSD, 4*ahead))))
for (i in 1:3) {
  lines(len:(len+ahead), c(tail(priceUSD, 1), predquants[i,]),
        col = 3, lty = c(2, 1, 2)[i])
}
lines(len:(len+ahead), c(tail(priceUSD, 1), expost),
      col = 2)
}

########################
# Further short examples
########################

y <- svsim(50, nu = 10, rho = -0.1)$y

# Supply initial values
res <- svsample(y,
                startpara = list(mu = -10, sigma = 1))

\donttest{
# Supply initial values for parallel chains
res <- svsample(y,
                startpara = list(list(mu = -10, sigma = 1),
                                 list(mu = -11, sigma = .1, phi = 0.9),
                                 list(mu = -9, sigma = .3, phi = 0.7)),
                parallel = "snow", n_chains = 3, n_cpus = 2)

# Parallel chains with with a pre-defined cluster object
cl <- parallel::makeCluster(2, "PSOCK", outfile = NULL)  # print to console
res <- svsample(y,
                parallel = "snow", n_chains = 3, cl = cl)
parallel::stopCluster(cl)
}

# Turn on correction for model misspecification
## Since the approximate model is fast and it is working very
##   well in practice, this is turned off by default
res <- svsample(y,
                expert = list(correct_model_misspecification = TRUE))
print(res)

\dontrun{
# Parallel multicore chains (not available on Windows)
res <- svsample(y, draws = 30000, burnin = 10000,
                parallel = "multicore", n_chains = 3, n_cpus = 2)

# Plot using a color palette
palette(rainbow(coda::nchain(para(res, "all"))))
plot(res)

# Use functionality from package 'coda'
## E.g. Geweke's convergence diagnostics
coda::geweke.diag(para(res, "all")[, c("mu", "phi", "sigma")])

# Use functionality from package 'bayesplot'
bayesplot::mcmc_pairs(res, pars = c("sigma", "mu", "phi", "h_0", "h_15"))
}

}
\references{
Kastner, G. and Frühwirth-Schnatter, S. (2014).
Ancillarity-sufficiency interweaving strategy (ASIS) for boosting MCMC
estimation of stochastic volatility models. \emph{Computational Statistics &
Data Analysis}, \bold{76}, 408--423,
\url{http://dx.doi.org/10.1016/j.csda.2013.01.002}.
}
\seealso{
\code{\link{svsim}}, \code{\link{specify_priors}}
}
\keyword{models}
\keyword{ts}
