% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/counterFactuals.R, R/plotMethods.R,
%   R/printMethods.R
\name{cfact_fore}
\alias{cfact_fore}
\alias{plot.cfactfore}
\alias{print.cfactfore}
\title{Simulate counterfactual forecast scenarios for structural STVAR models.}
\usage{
cfact_fore(
  stvar,
  nsteps,
  nsim = 1000,
  pi = 0.95,
  pred_type = c("mean", "median"),
  exo_weights = NULL,
  cfact_type = c("fixed_path", "muted_response"),
  policy_var = 1,
  mute_var = NULL,
  cfact_start = 1,
  cfact_end = 1,
  cfact_path = NULL
)

\method{plot}{cfactfore}(x, ..., nt, trans_weights = TRUE)

\method{print}{cfactfore}(x, ..., digits = 3)
}
\arguments{
\item{stvar}{an object of class \code{'stvar'} defining a structural or reduced form
STVAR model. For a reduced form model (that is not readily identified statiscally),
the shocks are automatically identified by the lower triangular Cholesky decomposition.}

\item{nsteps}{how many steps ahead should be predicted, i.e., the forecast horizon?}

\item{nsim}{to how many independent simulations should the forecast be based on?}

\item{pi}{a numeric vector specifying the confidence levels of the prediction intervals.}

\item{pred_type}{should the pointforecast be based on sample "median" or "mean"?}

\item{exo_weights}{if \code{weight_function="exogenous"}, provide a size \eqn{(nsteps x M)} matrix of exogenous
transition weights for the regimes: \code{[step, m]} for \eqn{step} steps ahead and regime \eqn{m} weight. Ignored
if \code{weight_function!="exogenous"}.}

\item{cfact_type}{a character string indicating the type of counterfactual to be computed: should the path of the policy
variable be fixed to some hypothetical path (\code{cfact_type="fixed_path"}) in given forecast horizons or should the responses
of the policy variable to lagged and contemporaneous movements of some given variable be muted (\code{cfact_type="muted_response"})?
See details for more information.}

\item{policy_var}{a positive integer between \eqn{1} and \eqn{d} indicating the index of the policy variable considered in the
counterfactual scenario.}

\item{mute_var}{a positive integer between \eqn{1} and \eqn{d} indicating the index of the variable to whose movements the policy variable
specified in the argument \code{policy_var} should not react to in the counterfactual scenario. This indicates also the index of the shock
to which the policy variable should not react to. It is assumed that \code{mute_var != policy_var}. This argument is only used when
\code{cfact_type="muted_response"}.}

\item{cfact_start}{a positive integer between \eqn{1} and \code{nsteps} indicating the starting forecast horizon period for the counterfactual
behavior of the specified policy variable.}

\item{cfact_end}{a positive integer between \code{cfact_start} and \code{nsteps} indicating the ending period for the counterfactual
behavior of the specified policy variable.}

\item{cfact_path}{a numeric vector of length \code{cfact_end-cfact_start+1} indicating the hypothetical path of the policy variable
specified in the argument \code{policy_var}. This argument is only used when \code{cfact_type="fixed_path"}.}

\item{x}{object of class \code{'cfactfore'} created by the function \code{cfact_fore}.}

\item{...}{parameters passed to \code{print.stvarpred} printing the forecast.}

\item{nt}{a positive integer specifying the number of observations to be plotted
along with the forecast.}

\item{trans_weights}{should forecasts for transition weights be plotted?}

\item{digits}{how many significant digits to print?}
}
\value{
Returns a class \code{'cfactfore'} list with the following elements:
  \describe{
    \item{$cfact_pred}{Counterfactual forecast in a class '\code{stvarpred}' object (see \code{?predict.stvar}).}
    \item{$pred}{Forecast that does not impose the counterfactual scenario, in a class '\code{stvarpred}' object.}
    \item{stvar}{The original STVAR model object.}
    \item{input}{A list containing the arguments used to calculate the counterfactual.}
 }

Returns the input object \code{x} invisibly.
}
\description{
\code{cfact_fore} simulates counterfactual forecast scenarios for structural STVAR models.
}
\details{
Two types of counterfactual forecast scenarios are accommodated where in given forecast horizons
 either (1) the policy variable of interest takes some hypothetical path (\code{cfact_type="fixed_path"}), or (2)
 its responses to lagged and contemporaneous movements of some given variable are shut off (\code{cfact_type="muted_response"}).
 In both cases, the counterfactual scenarios are simulated by creating hypothetical shocks to the policy variable of interest
 that yield the counterfactual outcome. This approach has the appealing feature that the counterfactual deviations from the
 policy reaction function are treated as policy surprises, allowing them to propagate normally, so that the dynamics of the model
 are not, per se, tampered but just the policy surprises are.

 \strong{Important:} This function assumes that when the policy variable of interest is the \eqn{i_1}th variable, the shock
 to it that is manipulated is the \eqn{i_1}th shock. This should be automatically satisfied for recursively identified models,
 whereas for model identified by heteroskedasticity or non-Gaussianity, the ordering of the shocks can be generally changed
 without loss of generality with the function \code{reorder_B_columns}. In Type (2) counterfactuals it is additionally assumed
 that, if the variable to whose movements the policy variable should not react to is the \eqn{i_2}th variable, the shock to it
 is the \eqn{i_2}th shock. If it is not clear whether the \eqn{i_2}th shock can be interpreted as a shock to a variable
 (but has a broader definition such as "a demand shock"), the Type (2) counterfactual scenario is interpreted as follows: the \eqn{i_1}th
 variable does not react to lagged movements of the \eqn{i_2}th variable nor to the \eqn{i_2}th shock.

 See the seminal paper of Bernanke et al (1997) for discussing about the "Type (1)" counterfactuals and
 Kilian and Lewis (2011) for discussion about the "Type (2)" counterfactuals. See Kilian and Lütkepohl (2017), Section 4.4
 for further discussion about counterfactual forecast scenarios. The literature cited about considers linear models, but it is
 explained in the vignette of this package how this function computes the counterfactual forecast scenarios for the STVAR models in
 a way that accommodates nonlinear time-varying dynamics.
}
\section{Functions}{
\itemize{
\item \code{plot(cfactfore)}: plot method

\item \code{print(cfactfore)}: print method

}}
\examples{
# Recursively identified logistic Student's t STVAR(p=3, M=2) model with the first
# lag of the second variable as the switching variable:
params32logt <- c(0.5959, 0.0447, 2.6279, 0.2897, 0.2837, 0.0504, -0.2188, 0.4008,
  0.3128, 0.0271, -0.1194, 0.1559, -0.0972, 0.0082, -0.1118, 0.2391, 0.164, -0.0363,
  -1.073, 0.6759, 3e-04, 0.0069, 0.4271, 0.0533, -0.0498, 0.0355, -0.4686, 0.0812,
  0.3368, 0.0035, 0.0325, 1.2289, -0.047, 0.1666, 1.2067, 7.2392, 11.6091)
mod32logt <- STVAR(gdpdef, p=3, M=2, params=params32logt, weight_function="logistic",
  weightfun_pars=c(2, 1), cond_dist="Student", identification="recursive")

# Counteractual forecast scenario 5 steps ahead (using only 100 Monte Carlo repetitions
# to save computation time), where the first variable takes values 1, -2, and 3 in the
# horizons 1, 2, and 3, respectively:
set.seed(1)
cfact1 <- cfact_fore(mod32logt, nsteps=5, nsim=100, cfact_type="fixed_path", policy_var=1,
 cfact_start=1, cfact_end=3, cfact_path=c(1, -2, 3))
cfact1 # Print the results
plot(cfact1) # Plot the factual and counterfactual forecasts

# Counteractual forecast scenario 5 steps ahead (using only 100 Monte Carlo repetitions
# to save computation time), where the first variable does not respond to lagged
# movements of the second variable nor to the second shock in time periods from 1 to 3:
set.seed(1)
cfact2 <- cfact_fore(mod32logt, nsteps=5, nsim=100, cfact_type="muted_response", policy_var=1,
 mute_var=2, cfact_start=1, cfact_end=3)
cfact2 # Print the results
plot(cfact2) # Plot the factual and counterfactual forecasts
}
\references{
\itemize{
   \item Bernanke B., Gertler M., Watson M. 1997. Systematic monetary policy and the effects of oilprice shocks.
     \emph{Brookings Papers on Economic Activity}, 1, 91—142.
   \item Kilian L., Lewis L. 2011. Does the fed respond to oil price shocks? \emph{The Economic Journal}, \strong{121}:555.
   \item Kilian L., Lütkepohl H. 2017. Structural Vector Autoregressive Analysis. 1st edition.
     \emph{Cambridge University Press}, Cambridge.
 }
}
\seealso{
\code{\link{GIRF}}, \code{\link{GFEVD}}, \code{\link{linear_IRF}}, \code{\link{hist_decomp}}, \code{\link{cfact_hist}},
 \code{\link{cfact_girf}},  \code{\link{fitSSTVAR}}
}
