% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Q.test.R
\name{Q.test}
\alias{Q.test}
\title{A function to compute Q test for spatial qualitative data using
       asymptotic distribution}
\usage{
Q.test(formula = NULL, data = NULL, na.action,
fx = NULL, coor = NULL, m = 3, r = 1, distr = "asymptotic",
control = list())
}
\arguments{
\item{formula}{a symbolic description of the factor(s).}

\item{data}{an (optional) data frame or a sf object
with points/multipolygons geometry containing the
variable(s) to be tested.}

\item{na.action}{action with NA values}

\item{fx}{a factor  or a matrix of factors in columns}

\item{coor}{(optional) a 2xN vector with spatial coordinates.
Used when *data* is not a spatial object}

\item{m}{length of m-surrounding (default = 3).}

\item{r}{maximum overlapping between any two m-surroundings (default = 1).}

\item{distr}{character. Distribution type "asymptotic" (default) or "mc".}

\item{control}{Optional argument. See Control Argument section.}
}
\value{
An list of two object of the class \code{htest}. Each element of the list return the:
  \tabular{ll}{
    \code{data.name} \tab a character string giving the names of the data.\cr
    \code{statistic} \tab Value of the Q test \cr
    \code{N} \tab total number of observations.\cr
    \code{R} \tab total number of symbolized observations.\cr
    \code{m} \tab length m-surrounding.\cr
    \code{r} \tab degree of overlapping.\cr
    \code{df} \tab degree of freedom.\cr
    \code{distr} \tab type of distribution used to get the significance of the Q test.\cr
    \code{type} \tab type of symbols.\cr
    }
}
\description{
A function to compute Q test for spatial qualitative data
}
\section{Control arguments}{

\describe{
\item{distance}{character to select the type of distance.
Default = "Euclidean" for Cartesian coordinates only: one of Euclidean,
Hausdorff or Frechet; for geodetic coordinates,
great circle distances are computed (see sf::st_distance())}
\item{dtmaxabs}{Delete degenerate surrounding based on the absolute distance between observations.}
\item{dtmaxpc}{A value between 0 and 1. Delete degenerate surrounding based on the distance. Delete m-surrounding when the maximum distance between observation
is upper than k percentage of maximum distance between anywhere observation.}
\item{dtmaxknn}{A integer value 'k'. Delete degenerate surrounding based on the near neighborhood criteria. Delete m-surrounding
is a element of the m-surrounding is not include in the set of k near neighborhood of the first element}
\item{nsim}{number of simulations for get the Monte Carlo distribution.
 Default = 999}
\item{seedinit}{seed to select the initial element to star
the algorithm to get compute the m-surroundings or to start the simulation}
}
}

\examples{

# Case 1: With coordinates
N <- 100
cx <- runif(N)
cy <- runif(N)
coor <- cbind(cx,cy)
p <- c(1/6,3/6,2/6)
rho = 0.5
listw <- spdep::nb2listw(spdep::knn2nb(spdep::knearneigh(cbind(cx,cy), k = 4)))
fx <- dgp.spq(list = listw, p = p, rho = rho)
q.test <- Q.test(fx = fx, coor = coor, m = 3, r = 1)
summary(q.test)
plot(q.test)
print(q.test)
\donttest{
q.test.mc <- Q.test(fx = fx, coor = coor, m = 3, r = 1, distr = "mc")
summary(q.test.mc)
plot(q.test.mc)
print(q.test.mc)


# Case 2: With a sf object
data("FastFood.sf")
f1 <- ~ Type
q.test <- Q.test(formula = f1, data = FastFood.sf, m = c(3, 4),
r = c(1, 2, 3), control = list(distance ="Euclidean"))
summary(q.test)
plot(q.test)
print(q.test)

# Case 3: With a sf object with isolated areas
data("provinces_spain")
sf::sf_use_s2(FALSE)
provinces_spain$Male2Female <- factor(provinces_spain$Male2Female > 100)
levels(provinces_spain$Male2Female) = c("men","woman")
provinces_spain$Older <- cut(provinces_spain$Older, breaks = c(-Inf,19,22.5,Inf))
levels(provinces_spain$Older) = c("low","middle","high")
f1 <- ~ Older + Male2Female
q.test <- Q.test(formula = f1,
data = provinces_spain, m = 3, r = 1, control = list(seedinit = 1111))
summary(q.test)
print(q.test)
plot(q.test)
q.test.mc <- Q.test(formula = f1, data = provinces_spain, m = 4, r = 3, distr = "mc",
control = list(seedinit = 1111))
summary(q.test.mc)
print(q.test.mc)
plot(q.test.mc)

# Case 4: Examples with multipolygons
library(sf)
fname <- system.file("shape/nc.shp", package="sf")
nc <- sf::st_read(fname)
qb79 <- quantile(nc$BIR79)
nc$QBIR79 <- (nc$BIR79 > qb79[2]) + (nc$BIR79 > qb79[3]) +
(nc$BIR79 >= qb79[4]) + 1
nc$QBIR79 <- as.factor(nc$QBIR79)
plot(nc["QBIR79"], pal = c("#FFFEDE","#FFDFA2", "#FFA93F", "#D5610D"),
     main = "BIR79 (Quartiles)")
sid79 <- quantile(nc$SID79)
nc$QSID79 <- (nc$SID79 > sid79[2]) + (nc$SID79 > sid79[3]) +
(nc$SID79 >= sid79[4]) + 1
nc$QSID79 <- as.factor(nc$QSID79)
plot(nc["QSID79"], pal = c("#FFFEDE","#FFDFA2", "#FFA93F", "#D5610D"),
     main = "SID79 (Quartiles)")
f1 <- ~ QSID79 + QBIR79
lq1nc <- Q.test(formula = f1, data = nc, m = 5, r = 2,
control = list(seedinit = 1111, dtmaxpc = 0.5, distance = "Euclidean") )
print(lq1nc)

lq2nc <- Q.test(formula = f1, data = nc, m = 5, r = 2,
control = list(dtmaxpc = 0.2) )
print(lq2nc)

lq3nc <- Q.test(formula = f1, data = nc, m = 5, r = 2,
control = list(dtmaxknn = 5) )
print(lq3nc)

# Case 5: Examples with points and matrix of variables
fx <- matrix(c(nc$QBIR79, nc$QSID79), ncol = 2, byrow = TRUE)
mctr <- suppressWarnings(sf::st_centroid(st_geometry(nc)))
mcoor <- st_coordinates(mctr)[,c("X","Y")]
q.test <- Q.test(fx = fx, coor = mcoor, m = 5, r = 2,
                 control = list(seedinit = 1111, dtmaxpc = 0.5))
print(q.test)
plot(q.test)

}
}
\references{
\itemize{
    \item Ruiz M, López FA, A Páez. (2010). \emph{Testing for spatial association of qualitative
    data using symbolic dynamics}. Journal of Geographical Systems. 12 (3) 281-309
    \item López, FA, and A Páez. (2012). \emph{Distribution-free inference for Q(m) based on permutational bootstrapping: an application
    to the spatial co-location pattern of firms in Madrid} Estadística Española, 177, 135-156.
  }
}
\author{
\tabular{ll}{
  Fernando López  \tab \email{fernando.lopez@upct.es} \cr
  Román Mínguez  \tab \email{roman.minguez@uclm.es} \cr
  Antonio Páez \tab \email{paezha@gmail.com} \cr
  Manuel Ruiz \tab \email{manuel.ruiz@upct.es} \cr
  }
}
