% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/raster.kendall.R
\name{raster.kendall}
\alias{raster.kendall}
\title{Kendall tau trend with continuity correction for raster time-series}
\usage{
raster.kendall(
  x,
  intercept = TRUE,
  p.value = TRUE,
  confidence = TRUE,
  tau = TRUE,
  min.obs = 6,
  method = c("zhang", "yuepilon", "none"),
  ...
)
}
\arguments{
\item{x}{A multiband terra SpatRaster object with at least 5 layers}

\item{intercept}{(FALSE/TRUE) return a raster with the pixel 
wise intercept values}

\item{p.value}{(FALSE/TRUE) return a raster with the pixel 
wise p.values}

\item{confidence}{(FALSE/TRUE) return a raster with the pixel 
wise 95 pct confidence levels}

\item{tau}{(FALSE/TRUE) return a raster with the pixel wise 
tau correlation values}

\item{min.obs}{The threshold of minimum number of observations (default 6)}

\item{method}{Kendall method to use c("zhang", "yuepilon","none"), see kendall function}

\item{...}{Additional arguments passed to the terra app function}
}
\value{
Depending on arguments, a raster layer or rasterBrick object containing:
\itemize{
  \item {raster layer 1} {slope for trend, always returned}
  \item {raster layer 2} {Kendall's tau two-sided test, reject null at 0, if tau TRUE}
  \item {raster layer 3} {intercept for trend if intercept TRUE}
  \item {raster layer 4} {p value for trend fit if p.value TRUE}
  \item {raster layer 5} {lower confidence level at 95 pct, if confidence TRUE}
  \item {raster layer 6} {upper confidence level at 95 pct, if confidence TRUE}
}
}
\description{
Calculates a nonparametric statistic for a monotonic trend 
             based on the Kendall tau statistic and the Theil-Sen slope 
             modification
}
\details{
This function implements Kendall's nonparametric test for a monotonic trend 
using the Theil-Sen (Theil 1950; Sen 1968; Siegel 1982) method to estimate 
the slope and related confidence intervals.
}
\examples{
\donttest{
 library(terra)

 # note; nonsense example with n=9
 r <- c(rast(system.file("ex/logo.tif", package="terra")),
        rast(system.file("ex/logo.tif", package="terra")),
        rast(system.file("ex/logo.tif", package="terra"))) 
 
 # Calculate trend slope with p-value and confidence level(s)
 # ("slope","intercept", "p.value","z.value", "LCI","UCI","tau")
   k <- raster.kendall(r, method="none")
   plot(k)
}

}
\references{
Theil, H. (1950) A rank invariant method for linear and polynomial regression 
  analysis.  Nederl. Akad. Wetensch. Proc. Ser. A 53:386-392 (Part I), 
  53:521-525 (Part II), 53:1397-1412 (Part III).

Sen, P.K. (1968) Estimates of Regression Coefficient Based on Kendall's tau. 
  Journal of the American Statistical Association. 63(324):1379-1389.

Siegel, A.F. (1982) Robust Regression Using Repeated Medians. 
  Biometrika, 69(1):242-244
}
\seealso{
\code{\link[zyp]{zyp.trend.vector}} for model details

\code{\link[terra]{app}} for available ... arguments
}
\author{
Jeffrey S. Evans  <jeffrey_evans@tnc.org>
}
