\name{sjp.pca}
\alias{sjp.pca}
\title{Plot PCA results}
\usage{
sjp.pca(data, numberOfFactors = NULL, factorLoadingTolerance = 0.1,
  plotEigenvalues = FALSE, title = NULL, titleSize = 1.3,
  titleColor = "black", axisLabels.y = NULL, type = "tile",
  geomAlpha = 0.8, valueLabelColor = "black", valueLabelSize = 4.5,
  valueLabelAlpha = 1, circleSize = 10, outlineColor = "black",
  outlineSize = 0.2, axisColor = NULL, borderColor = NULL,
  axisLabelSize = 1.1, axisLabelColor = "gray30", axisLabelAngle.x = 0,
  axisLabelAngle.y = 0, breakTitleAt = 50, breakLabelsAt = 20,
  hideLegend = TRUE, legendTitle = NULL, showValueLabels = TRUE,
  showTickMarks = FALSE, showCronbachsAlpha = TRUE, fillColor = NULL,
  majorGridColor = NULL, minorGridColor = NULL, theme = NULL,
  printPlot = TRUE)
}
\arguments{
  \item{data}{A data frame with factors (each columns one
  variable) that should be used to compute a PCA, or a
  \code{\link{prcomp}} object.}

  \item{numberOfFactors}{A predefined number of factors to
  use for the calculating the varimax rotation. By default,
  this value is \code{NULL} and the amount of factors is
  calculated according to the Kaiser-criteria. See
  paramater \code{plotEigenvalues}.}

  \item{factorLoadingTolerance}{Specifies the minimum
  difference a variable needs to have between factor
  loadings (components) in order to indicate a clear
  loading on just one factor and not diffusing over all
  factors. For instance, a variable with 0.8, 0.82 and 0.84
  factor loading on 3 possible factors can not be clearly
  assigned to just one factor and thus would be removed
  from the principal component analysis. By default, the
  minimum difference of loading values between the highest
  and 2nd highest factor should be 0.1}

  \item{plotEigenvalues}{If \code{TRUE}, a plot showing the
  Eigenvalues according to the Kaiser criteria is plotted
  to determine the number of factors.}

  \item{title}{Title of the diagram, plotted above the
  whole diagram panel.}

  \item{titleSize}{The size of the plot title. Default is
  1.3.}

  \item{titleColor}{The color of the plot title. Default is
  \code{"black"}.}

  \item{axisLabels.y}{The item labels that are printed on
  the y-axis. If no item labels are provided (default), the
  data frame's column names are used. Item labels must be a
  string vector, e.g.: \code{axisLabels.y=c("Var 1", "Var
  2", "Var 3")}.}

  \item{type}{Indicates whether \code{"circle"} (default)
  or \code{"tile"} geoms should be used for plotting.}

  \item{geomAlpha}{Specify the transparancy (alpha value)
  of geom objects (circles or tiles).  Default is 0.8.}

  \item{valueLabelColor}{The color of the value labels
  (numbers) inside the diagram.  Default is
  \code{"black"}.}

  \item{valueLabelSize}{The size of value labels in the
  diagram. Default is 4.5, recommended values range between
  2 and 8.}

  \item{valueLabelAlpha}{Specify the transparancy (alpha
  value) of value labels.  Default is 0.8}

  \item{circleSize}{Specifies the circle size factor. The
  circle size depends on the correlation value
  multiplicated with this factor. Default is 10.}

  \item{outlineColor}{Defines the outline color of geoms
  (circles or tiles). Default is \code{"black"}.}

  \item{outlineSize}{Defines the outline size of geoms
  (circles or tiles). Default is 1.}

  \item{axisColor}{User defined color of axis border (y-
  and x-axis, in case the axes should have different colors
  than the diagram border).}

  \item{borderColor}{User defined color of whole diagram
  border (panel border).}

  \item{axisLabelSize}{The size of variable labels at the
  axes. Default is 1.1, recommended values range between
  0.5 and 3.0.}

  \item{axisLabelColor}{User defined color for axis labels.
  If not specified, a default dark gray color palette will
  be used for the labels.}

  \item{axisLabelAngle.x}{Angle for x-axis-labels.}

  \item{axisLabelAngle.y}{Angle for y-axis-labels.}

  \item{breakTitleAt}{Wordwrap for diagram title.
  Determines how many chars of the title are displayed in
  one line and when a line break is inserted into the
  title. Default is 50.}

  \item{breakLabelsAt}{Wordwrap for diagram labels.
  Determines how many chars of the category labels are
  displayed in one line and when a line break is inserted.
  Default is 12.}

  \item{hideLegend}{Show or hide the legend. The legend
  indicates the strength of correlations by gradient colour
  fill. Default is \code{TRUE}, hence the legend is
  hidden.}

  \item{legendTitle}{The legend title, provided as string,
  e.g. \code{legendTitle=c("Factor loading")}.  Default is
  \code{NULL}, hence no legend title is used.}

  \item{showValueLabels}{Whether factor loading values
  should be plotted to each geom.  Default is \code{TRUE}.}

  \item{showTickMarks}{Whether tick marks should be plotted
  or not. Default is \code{FALSE}.}

  \item{showCronbachsAlpha}{If \code{TRUE} (default), the
  cronbach's alpha value for each factor scale will be
  calculated, i.e. all variables with the highest loading
  for a factor are taken for the reliability test. The
  result is an alpha value for each factor dimension.  Only
  applies when \code{data} is a data frame and no
  \code{\link{prcomp}} object.}

  \item{fillColor}{A color palette for fillng the geoms. If
  not specified, the 5th diverging color palette from the
  color brewer palettes (RdBu) is used, resulting in red
  colors for negative and blue colors for positive factor
  loadings, that become lighter the weaker the loadings
  are. Use any color palette that is suitbale for the
  \code{scale_fill_gradientn} parameter of ggplot2.}

  \item{majorGridColor}{Specifies the color of the major
  grid lines of the diagram background.}

  \item{minorGridColor}{Specifies the color of the minor
  grid lines of the diagram background.}

  \item{theme}{Specifies the diagram's background theme.
  Default (parameter \code{NULL}) is a gray background with
  white grids.  \itemize{ \item Use \code{"bw"} for a white
  background with gray grids \item \code{"classic"} for a
  classic theme (black border, no grids) \item
  \code{"minimal"} for a minimalistic theme (no border,gray
  grids) or \item \code{"none"} for no borders, grids and
  ticks.  }}

  \item{printPlot}{If \code{TRUE} (default), plots the
  results as graph. Use \code{FALSE} if you don't want to
  plot any graphs. In either case, the ggplot-object will
  be returned as value.}
}
\value{
(Invisibly) returns a \code{\link{structure}} with
\itemize{ \item the varimax-rotated factor loading matrix
(\code{varim}) \item the column indices of removed
variables (for more details see next list item)
(\code{removed.colindex}) \item an updated data frame
containing all factors that have a clear loading on a
specific scale in case \code{data} was a data frame (See
parameter \code{factorLoadingTolerance} for more details)
(\code{removed.df}) \item the ggplot-object (\code{plot}),
\item the data frame that was used for setting up the
ggplot-object (\code{df}).  }
}
\description{
Performes a principle component analysis on a data frame or
matrix and plots the factor solution as ellipses or tiles.
\cr \cr In case a data frame is used as parameter, the
cronbach's alpha value for each factor scale will be
calculated, i.e. all variables with the highest loading for
a factor are taken for the reliability test. The result is
an alpha value for each factor dimension.
}
\note{
This PCA uses the \code{\link{prcomp}} function and the
\code{\link{varimax}} rotation.
}
\examples{
# randomly create data frame with 7 items, each consisting of 4 categories
likert_4 <- data.frame(sample(1:4, 500, replace=TRUE, prob=c(0.2,0.3,0.1,0.4)),
                       sample(1:4, 500, replace=TRUE, prob=c(0.5,0.25,0.15,0.1)),
                       sample(1:4, 500, replace=TRUE, prob=c(0.4,0.15,0.25,0.2)),
                       sample(1:4, 500, replace=TRUE, prob=c(0.25,0.1,0.4,0.25)),
                       sample(1:4, 500, replace=TRUE, prob=c(0.1,0.4,0.4,0.1)),
                       sample(1:4, 500, replace=TRUE),
                       sample(1:4, 500, replace=TRUE, prob=c(0.35,0.25,0.15,0.25)))

# Create variable labels
colnames(likert_4) <- c("V1", "V2", "V3", "V4", "V5", "V6", "V7")

# plot results from PCA as square-tiled "heatmap"
sjp.pca(likert_4)

# manually compute PCA
pca <- prcomp(na.omit(likert_4), retx=TRUE, center=TRUE, scale.=TRUE)
# plot results from PCA as circles, including Eigenvalue-diagnostic.
# note that this plot does not compute the Cronbach's Alpha
sjp.pca(pca, plotEigenvalues=TRUE, type="circle")


# -------------------------------
# Data from the EUROFAMCARE sample dataset
# -------------------------------
data(efc)

# retrieve variable and value labels
varlabs <- sji.getVariableLabels(efc)

# recveive first item of COPE-index scale
start <- which(colnames(efc)=="c82cop1")
# recveive last item of COPE-index scale
end <- which(colnames(efc)=="c90cop9")

# create data frame with COPE-index scale
df <- as.data.frame(efc[,c(start:end)])
colnames(df) <- varlabs[c(start:end)]

sjp.pca(df)
}
\references{
\url{http://strengejacke.wordpress.com/sjplot-r-package/}
\cr \cr
\url{http://strengejacke.wordpress.com/2013/07/08/plotting-principal-component-analysis-with-ggplot-rstats/}
}

