% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simode.R
\name{simode}
\alias{simode}
\title{Statistical inference of ordinary differential equations using
separable integral-matching}
\usage{
simode(equations, pars, time, obs, mc_sets = 1, nlin_pars = NULL,
  likelihood_pars = NULL, fixed = NULL, start = NULL, lower = NULL,
  upper = NULL, im_method = c("separable", "non-separable"),
  gen_obs = NULL, calc_nll = NULL, simode_ctrl = simode.control(), ...)
}
\arguments{
\item{equations}{Named vector. The equations describing the ODE system.
Each element of the vector should contain a character representation
of the right-hand side of an equation, and should be named according to
the left-hand side of the equation (i.e., the variable name).
An equation can contain parameters appearing in \code{pars},
variables appearing in the equations names and/or any function of 't',
which is a reserved symbol for the time domain.}

\item{pars}{The names of the parameters and initial conditions
to be estimated. An initial condition name for a certain variable
is the name given to the relevant equation in \code{equations}
(e.g., if an equation is named 'x' than its initial condition should be named 'x'
as well). Note: The symbol 't' is reserved for the time domain and cannot be
used as a parameter name.}

\item{time}{Time points of the observations. Either a vector,
if the same time points were used for observing all variables,
or a list of vectors the length of \code{obs}, of which each element
is the length of the relevant element in \code{obs}.}

\item{obs}{Named list. The observations. When \code{mc_sets=1}, \code{obs}
should contain a list of vectors with length <= equations (either partial or fully observed
system). Each list member should be named according to the relevant equation name and should
be the length of the relevant time vector. When \code{mc_sets>1}, \code{obs} should contain
a list the length of \code{mc_sets}, where each list member is a list that fits the
description in the case of \code{mc_sets=1}.}

\item{mc_sets}{Number of monte-carlo expriments. When \code{mc_sets>1}, the function will
fit each set of observations separately either sequentially or in parallel, according
to the value of \code{parallel} in \code{simode_ctrl}.}

\item{nlin_pars}{Names of parameters or initial conditions
that will be estimated in stage 1 using nonlinear least squares optimization.
The parameter names in \code{nlin_pars} must appear in \code{pars}.}

\item{likelihood_pars}{Names of likelihood parameters not appearing in the ODE system,
which are needed for the the user-defined function \code{calc_nll}.
The parameter names in \code{likelihood_pars} must appear in \code{pars}.}

\item{fixed}{Named vector. Fixed values for one or more of the ODE system parameters or
initial conditions. Parameters in this list will not be estimated.}

\item{start}{Named vector. Starting values for optimization of parameters/initial conditions.
Must contain starting values for all the parameters in \code{nlin_pars}
and \code{likelihood_pars}. If im_method="non-seperable", can optionally contain
starting values for any other parameter/initial condition.}

\item{lower}{Named vector. Lower bounds for any parameter/initial condition.}

\item{upper}{Named vector. Upper bounds for any parameter/initial condition.}

\item{im_method}{The method to use for integral-matching. Default "separable"
means that linear parameters are estimated directly while "non-separable"
means that linear parameters are estimated using nonlinear least squares optimization.
If none of the parameters are linear then the default can be used.}

\item{gen_obs}{A user-defined function for completing missing observations (see Details).}

\item{calc_nll}{A user-defined function for calculating negative log-likelihood for
the model (see Details).}

\item{simode_ctrl}{Various control parameters. See \code{\link{simode.control}}.}

\item{...}{Additional arguments passed to \code{optim}, \code{gen_obs} and \code{calc_nll}}
}
\value{
If mc_sets=1, returns a \code{simode} object containing the
parameter estimates after integral-matching (stage 1) and after
nonlinear least squares optimization (stage 2). If \code{mc_sets>1} returns a
\code{list.simode} object which is a list of \code{simode} objects the length of
\code{mc_sets}.
}
\description{
Estimating the parameters of an ODE system in two stages:
1) Estimate the parameters using separable integral-matching,
2) Estimate the parameters using nonlinear least squares
starting from the values obtained in stage 1.
}
\details{
\code{gen_obs} can be used in cases of a partially observed system,
for which observations of the missing variables can be generated given values for the
system parameters. The function will be called during the optimization using integral-matching.\cr
It must be defined as
\code{gen_obs <- function(equations, pars, x0, time, obs, ...)}, where:
\itemize{
\item \code{equations} the ODE equations
\item \code{pars} the parameter values
\item \code{x0} the initial conditions
\item \code{time} the timing of the observations (vector or list)
\item \code{obs} the observations
\item \code{...} additional parameters passed from the call to \code{\link{simode}}
}
The function should return a list with two items:
\itemize{
\item \code{time} the vector or list of time points of the observations
\item \code{obs} the list of observations with the newly generated observations
}

\code{calc_nll} allows the user to pass his own likelihood function to be used in the
optimization in the second stage (if not defined, the default nonlinear least squares
optimization will be used). The likelihood function will also be used in a following
call to \code{\link{profile}}, for the calculation of likelihood profiles.
It must be defined as
\code{calc_nll <- function(pars, time, obs, model_out, ...)}, where:
\itemize{
\item \code{pars} the parameter values
\item \code{time} the timing of the observations (vector or list)
\item \code{obs} the observations
\item \code{model_out} the model output returned from a call to \code{\link{solve_ode}}.
If \code{time} is a list with possibly different times for each variable
then \code{model_out} will contain a union of all these times.
\item \code{...} additional parameters passed from the call to \code{\link{simode}}
}
The function should return the negative log-likelihood.
}
\examples{

## =================================================
## Predator-Prey Lotka-Volterra model
## =================================================

## generate model equations and parameters (X=Prey,Y=Predator)
pars <- c('alpha','beta','gamma','delta')
vars <- c('X','Y')
eq_X <- 'alpha*X-beta*X*Y'
eq_Y <- 'delta*X*Y-gamma*Y'
equations <- c(eq_X,eq_Y)
names(equations) <- vars
x0 <- c(0.9,0.9)
names(x0) <- vars
theta <- c(2/3,4/3,1,1)
names(theta) <- pars

## generate observations
n <- 50
time <- seq(0,25,length.out=n)
model_out <- solve_ode(equations,theta,x0,time)
x_det <- model_out[,vars]
set.seed(1000)
sigma <- 0.05
obs <- list()
for(i in 1:length(vars)) {
  obs[[i]] <- pmax(0, rnorm(n,x_det[,i],sigma))
}
names(obs) <- vars

## estimate model parameters with known initial conditions
simode_fit1 <- simode(equations=equations, pars=pars, fixed=x0, time=time, obs=obs)
plot(simode_fit1, type='fit', time=seq(0,25,length.out=100), pars_true=theta, mfrow=c(2,1))
plot(simode_fit1, type='est', pars_true=theta)

\donttest{
## estimate model parameters and initial conditions
simode_fit2 <- simode(equations=equations, pars=c(pars,vars), time=time, obs=obs)
plot(simode_fit2, type='fit', time=seq(0,25,length.out=100), pars_true=c(theta,x0), mfrow=c(2,1))
plot(simode_fit2, type='est', pars_true=c(theta,x0))

profiles_fit2 <- profile(simode_fit2,step_size=0.01,max_steps=50)
plot(profiles_fit2,mfrow=c(2,3))
ci_fit2 <- confint(profiles_fit2)
ci_fit2
plot(ci_fit2,pars_true=c(theta,x0),legend=T)
}

}
\references{
Dattner & Klaassen (2015). Optimal Rate of Direct Estimators in Systems of Ordinary Differential Equations Linear in Functions of the Parameters,
Electronic Journal of Statistics, Vol. 9, No. 2, 1939-1973.

Dattner, Miller, Petrenko, Kadouriz, Jurkevitch & Huppert (2017). Modelling and Parameter Inference of Predator-prey Dynamics in Heterogeneous Environments Using The Direct Integral Approach,
Journal of The Royal Society Interface 14.126: 20160525.
}
