% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ssq.R
\name{ssq}
\alias{ssq}
\title{Single-Server Queue Simulation}
\usage{
ssq(
  maxArrivals = Inf,
  seed = NA,
  interarrivalFcn = NULL,
  serviceFcn = NULL,
  interarrivalType = "M",
  serviceType = "M",
  maxTime = Inf,
  maxDepartures = Inf,
  maxInSystem = Inf,
  maxEventsPerSkyline = 15,
  saveAllStats = FALSE,
  saveInterarrivalTimes = FALSE,
  saveServiceTimes = FALSE,
  saveWaitTimes = FALSE,
  saveSojournTimes = FALSE,
  saveNumInQueue = FALSE,
  saveNumInSystem = FALSE,
  saveServerStatus = FALSE,
  showOutput = TRUE,
  animate = FALSE,
  showQueue = NULL,
  showSkyline = NULL,
  showSkylineSystem = FALSE,
  showSkylineQueue = FALSE,
  showSkylineServer = FALSE,
  showTitle = TRUE,
  showProgress = TRUE,
  plotQueueFcn = defaultPlotSSQ,
  plotSkylineFcn = defaultPlotSkyline,
  jobImage = NA,
  plotDelay = NA,
  respectLayout = FALSE
)
}
\arguments{
\item{maxArrivals}{maximum number of customer arrivals allowed to enter the system}

\item{seed}{initial seed to the random number generator (NA uses current state of
random number generator; NULL seeds using system clock)}

\item{interarrivalFcn}{function for generating interarrival times for queue simulation.
Default value (\code{NA}) will result in use of default interarrival
function based on \code{interarrivalType}.  See examples.}

\item{serviceFcn}{function for generating service times for queue simulation.
Default value (\code{NA}) will result in use of default service function
based on \code{serviceType}.  See examples.}

\item{interarrivalType}{string representation of desired interarrival process.
Options are "M" -- exponential with rate 1;
"G" -- uniform(0,2), having mean 1; and "D" -- deterministic with
constant value 1.  Default is "M".}

\item{serviceType}{string representation of desired service process .
Options are "M" -- exponential with rate 10/9;
"G" -- uniform(0, 1.8), having mean 9/10; and "D" --
deterministic with constant value 9/10.  Default is "M".}

\item{maxTime}{maximum time to simulate}

\item{maxDepartures}{maximum number of customer departures to process}

\item{maxInSystem}{maximum number of customers that the system can hold (server(s) plus queue).
Infinite by default.}

\item{maxEventsPerSkyline}{maximum number of events viewable at a time in the skyline plot.
A large value for this parameter may result in plotting delays.
This parameter does not impact the final plotting, which will show all
end-of-simulation results.}

\item{saveAllStats}{if \code{TRUE}, returns all vectors of statistics (see below)
collected by the simulation}

\item{saveInterarrivalTimes}{if \code{TRUE}, returns a vector of all interarrival times generated}

\item{saveServiceTimes}{if \code{TRUE}, returns a vector of all service times generated}

\item{saveWaitTimes}{if \code{TRUE}, returns a vector of all wait times (in the queue) generated}

\item{saveSojournTimes}{if \code{TRUE}, returns a vector of all sojourn times (time spent in the system)
generated}

\item{saveNumInQueue}{if \code{TRUE}, returns a vector of times and a vector of counts for whenever
the number in the queue changes}

\item{saveNumInSystem}{if \code{TRUE}, returns a vector of times and a vector of counts for whenever
the number in the system changes}

\item{saveServerStatus}{if \code{TRUE}, returns a vector of times and a vector of
server status (0:idle, 1:busy) for whenever the status changes}

\item{showOutput}{if \code{TRUE}, displays summary statistics upon completion}

\item{animate}{logical; if \code{FALSE}, no animation plots will be shown.}

\item{showQueue}{logical; if \code{TRUE}, displays a visualization of the queue}

\item{showSkyline}{If \code{NULL} (default), defers to each individual showSkyline...
parameter below; otherwise, supersedes individual showSkyline...
parameter values.
If \code{TRUE}, displays full skyline plot; \code{FALSE} suppresses
skyline plot. Can alternatively be specified using chmod-like octal
component specification: use 1, 2, 4 for system, queue, and server
respectively, summing to indicate desired combination (e.g., 7 for all).
Can also be specified as a binary vector (e.g., c(1,1,1) for all).}

\item{showSkylineSystem}{logical; if \code{TRUE}, includes number in system
as part of skyline plot.  Value for \code{showSkyline}
supersedes this parameter's value.}

\item{showSkylineQueue}{logical; if \code{TRUE}, includes number in queue
as part of skyline plot.  Value for \code{showSkyline}
supersedes this parameter's value.}

\item{showSkylineServer}{logical; if \code{TRUE}, includes number in server
as part of skyline plot.  Value for \code{showSkyline}
supersedes this parameter's value.}

\item{showTitle}{if \code{TRUE}, display title at the top of the main plot}

\item{showProgress}{if TRUE, displays a progress bar on screen during no-animation execution}

\item{plotQueueFcn}{plotting function to display queue visualization.
By default, this is provided by \code{defaultPlotSSQ}. Please refer to
that associated help for more details about required arguments.}

\item{plotSkylineFcn}{plotting function to display Skyline visualization.
By default, this is provided by \code{defaultPlotSkyline}.  Please refer
to that associated help for more details about required arguments.}

\item{jobImage}{a vector of URLs/local addresses of images to use as jobs. Requires
package \code{'magick'}.}

\item{plotDelay}{a positive numeric value indicating seconds between plots.
A value of -1 enters 'interactive' mode, where the state will pause
for user input at each step.  A value of 0 will display only the final
end-of-simulation plot.}

\item{respectLayout}{logical; if \code{TRUE}, plot layout (i.e. par, device, etc.) settings
will be respected.}
}
\value{
The function returns a list containing:
\itemize{
\item the number of arrivals to the system (\code{customerArrivals}),
\item the number of customers processed (\code{customerDepartures}),
\item the ending time of the simulation (\code{simulationEndTime}),
\item average wait time in the queue (\code{avgWait}),
\item average time in the system (\code{avgSojourn}),
\item average number in the system (\code{avgNumInSystem}),
\item average number in the queue (\code{avgNumInQueue}), and
\item server utilization (\code{utilization}).
}
of the queue as computed by the simulation.
When requested via the ``save...'' parameters, the list may also contain:
\itemize{
\item a vector of interarrival times (\code{interarrivalTimes}),
\item a vector of wait times (\code{waitTimes}),
\item a vector of service times (\code{serviceTimes}),
\item a vector of sojourn times (\code{sojournTimes}),
\item two vectors (time and count) noting changes to number in the system
(\code{numInSystemT}, \code{numInSystemN}),
\item two vectors (time and count) noting changes to number in the queue
(\code{numInQueueT}, \code{numInQueueN}), and
\item two vectors (time and status) noting changes to server status
(\code{serverStatusT}, \code{serverStatusN}).
}
}
\description{
A next-event simulation of a single-server queue, with
extensible arrival and service processes.
}
\details{
Implements a next-event implementation of a single-server queue simulation.

The \code{seed} parameter can take one of three valid
argument types:
\itemize{
\item \code{NA} (default), which will use the current state of the random
number generator without explicitly setting a new seed (see examples);
\item a positive integer, which will be used as the initial seed passed in
an explicit call to \code{\link{set.seed}}; or
\item \code{NULL}, which will be passed in an explicit call to to
\code{\link{set.seed}}, thereby setting the initial seed using the
system clock.
}
}
\examples{
 # process 100 arrivals, R-provided seed (via NULL seed)
 ssq(100, NULL)

 ssq(maxArrivals = 100, seed = 54321)
 ssq(maxDepartures = 100, seed = 54321)
 ssq(maxTime = 100, seed = 54321)

 ############################################################################
 # example to show use of seed = NA (default) to rely on current state of generator
 output1 <- ssq(200, 8675309, showOutput = FALSE, saveAllStats = TRUE)
 output2 <- ssq(300,          showOutput = FALSE, saveAllStats = TRUE)
 set.seed(8675309)
 output3 <- ssq(200,          showOutput = FALSE, saveAllStats = TRUE)
 output4 <- ssq(300,          showOutput = FALSE, saveAllStats = TRUE)
 sum(output1$sojournTimes != output3$sojournTimes) # should be zero
 sum(output2$sojournTimes != output4$sojournTimes) # should be zero

 myArrFcn <- function() { vexp(1, rate = 1/4, stream = 1)  }  # mean is 4
 mySvcFcn <- function() { vgamma(1, shape = 1, rate = 0.3) }  # mean is 3.3

 output <- ssq(maxArrivals = 100, interarrivalFcn = myArrFcn, serviceFcn = mySvcFcn,
              saveAllStats = TRUE)
 mean(output$interarrivalTimes)
 mean(output$serviceTimes)
 meanTPS(output$numInQueueT, output$numInQueueN) # compute time-averaged num in queue
 meanTPS(output$serverStatusT, output$serverStatusN) # compute server utilization

 ############################################################################
 # example to show use of (simple) trace data for arrivals and service times;
 # ssq() will need one more interarrival (arrival) time than jobs processed
 #
 arrivalTimes      <- NULL
 interarrivalTimes <- NULL
 serviceTimes      <- NULL

 initTimes <- function() {
     arrivalTimes      <<- c(15, 47, 71, 111, 123, 152, 232, 245, 99999)
     interarrivalTimes <<- c(arrivalTimes[1], diff(arrivalTimes))
     serviceTimes      <<- c(43, 36, 34, 30, 38, 30, 31, 29)
 }

 getInterarr <- function() {
     nextInterarr <- interarrivalTimes[1]
     interarrivalTimes <<- interarrivalTimes[-1]  # remove 1st element globally
     return(nextInterarr)
 }

 getService <- function() {
     nextService <- serviceTimes[1]
     serviceTimes <<- serviceTimes[-1]  # remove 1st element globally
     return(nextService)
 }

 initTimes()
 numJobs <- length(serviceTimes)
 output <- ssq(maxArrivals = numJobs, interarrivalFcn = getInterarr,
               serviceFcn = getService, saveAllStats = TRUE)
 mean(output$interarrivalTimes)
 mean(output$serviceTimes)


 ############################################################################
 # example to show use of (simple) trace data for arrivals and service times,
 # allowing for reuse (recycling) of trace data times
 arrivalTimes      <- NULL
 interarrivalTimes <- NULL
 serviceTimes      <- NULL

 initArrivalTimes <- function() {
   arrivalTimes      <<- c(15, 47, 71, 111, 123, 152, 232, 245)
   interarrivalTimes <<- c(arrivalTimes[1], diff(arrivalTimes))
 }

 initServiceTimes <- function() {
     serviceTimes    <<- c(43, 36, 34, 30, 38, 30, 31, 29)
 }

 getInterarr <- function() {
     if (length(interarrivalTimes) == 0)  initArrivalTimes()

     nextInterarr <- interarrivalTimes[1]
     interarrivalTimes <<- interarrivalTimes[-1] # remove 1st element globally
     return(nextInterarr)
 }

 getService <- function() {
     if (length(serviceTimes) == 0)  initServiceTimes()

     nextService <- serviceTimes[1]
     serviceTimes <<- serviceTimes[-1]  # remove 1st element globally
     return(nextService)
 }

 initArrivalTimes()
 initServiceTimes()
 output <- ssq(maxArrivals = 100, interarrivalFcn = getInterarr,
               serviceFcn = getService, saveAllStats = TRUE)
 mean(output$interarrivalTimes)
 mean(output$serviceTimes)

 ############################################################################
 # Testing with visualization

 # Visualizing ssq with a set seed, infinite queue capacity, 20 arrivals,
 # interactive mode (default), showing skyline for all 3 attributes (default)
 if (interactive()) {
   ssq(seed = 1234, maxArrivals = 20, animate = TRUE)
 }

 # Same as above, but jump to final queue visualization using plotDelay 0
 ssq(seed = 1234, maxArrivals = 20, animate = TRUE, plotDelay = 0)

 # Perform simulation again with finite queue of low capacity. Note same
 # variate generation but different outcomes due to rejection pathway
 ssq(seed = 1234, maxArrivals = 25, animate = TRUE, maxInSystem = 5, plotDelay = 0)

 # Using default distributions to simulate a default M/G/1 Queue
 ssq(seed = 1234, maxDepartures = 10, interarrivalType = "M", serviceType = "G", 
     animate = TRUE, plotDelay = 0)

}
\seealso{
\code{\link{rstream}}, \code{\link{set.seed}},
\code{\link[=runif]{stats::runif}}
}
\author{
Barry Lawson (\email{blawson@bates.edu}), \cr
Larry Leemis (\email{leemis@math.wm.edu}), \cr
Vadim Kudlay (\email{vkudlay@nvidia.com})
}
\concept{queueing}
\keyword{utilities}
