#' @title Prints infection curves.
#'
#' @param dataInput a data frame composed of two columns. One is for time and the other is for intensity. Should be normalized data generated by normalizeData function.
#' @param sigmoidalFitVector the output of sigmoidalFitFunction. Default is NULL.
#' @param doubleSigmoidalFitVector the output of double sigmoidal fit function. Default is NULL.
#' @param showParameterRelatedLines if equal to TRUE, figure will show parameter related lines on the curves. Default is FALSE.
#' @param xlabelText the x-axis name; with default "time"
#' @param ylabelText the y-axis name; with default "intensity"
#'
#' @description Generates figures using ggplot that shows the input data and the fitted curves.
#' @return Returns infection curve figures.
#' @export
#'
#' @examples
#'time=seq(3,24,0.1)
#'
#'#simulate intensity data and add noise
#'noise_parameter=0.2
#'intensity_noise=runif(n = length(time),min = 0,max = 1)*noise_parameter
#'intensity=doublesigmoidalFitFormula(time,
#'                                    finalAsymptoteIntensity=.3,
#'                                    maximum=4,
#'                                    slope1=1,
#'                                    midPoint1=7,
#'                                    slope2=1,
#'                                    midPointDistance=8)
#'intensity=intensity+intensity_noise
#'
#'dataInput=data.frame(intensity=intensity,time=time)
#'normalizedInput = normalizeData(dataInput,dataInputName="batch_01_21_2016_samp007623")
#'
#'
#'# Do the double sigmoidal fit
#'doubleSigmoidalModel=fitFunction(dataInput=normalizedInput,
#'                                           model="doublesigmoidal",
#'                                           n_runs_min=20,
#'                                           n_runs_max=500,
#'                                           showDetails=FALSE)
#'
#'doubleSigmoidalModel = numericalReCalculation(doubleSigmoidalModel,
#'                                                        stepSize=0.00001)
#'
#'fig01=printInfectionCurves(dataInput=normalizedInput,
#'                           doubleSigmoidalFitVector=doubleSigmoidalModel,
#'                           showParameterRelatedLines=TRUE)
#'print(fig01)
#'
#'
printInfectionCurves<-function(dataInput,
                               sigmoidalFitVector=NULL,
                               doubleSigmoidalFitVector=NULL,
                               showParameterRelatedLines=FALSE,
                               xlabelText="time", ylabelText="intensity")
{
  # get data from data input
  dataOutputVariable = dataCheck(dataInput) # check if the data structure is correct
  sameSourceDataCheck(dataInput,
                      sigmoidalFitVector,
                      doubleSigmoidalFitVector) # check if all data comes from same source

  # get data from data structure
  isalist=(is.list(dataInput) & !is.data.frame(dataInput))
  if(isalist){
    dataInput=unnormalizeData(dataInput)
    dataFrameInput=dataInput$timeIntensityData
  }
  isadataframe=(is.data.frame(dataInput))
  if(isadataframe){
    dataFrameInput=dataInput
  }

  # SIGMOIDAL
  if(!is.null(sigmoidalFitVector))
  {
    if(!sigmoidalFitVector$model=="sigmoidal")
    {stop("provided sigmoidalFitVector is not a sigmoidal fit vector")}
    if(!sigmoidalFitVector$isThisaFit)
    {warning("provided sigmoidal fit vector does not include a fit!")}
    if(sigmoidalFitVector$isThisaFit)
    {
      # GENERATE THE TIME SERIES FOR FITTED DATA
      time=seq(0,
               sigmoidalFitVector$dataScalingParameters.timeRatio,
               sigmoidalFitVector$dataScalingParameters.timeRatio/1000)
      intensityTheoreticalSigmoidal=sigmoidalFitFormula(time,
                                                        maximum=sigmoidalFitVector$maximum_Estimate,
                                                        slope=sigmoidalFitVector$slope_Estimate,
                                                        midPoint=sigmoidalFitVector$midPoint_Estimate)
      intensityTheoreticalSigmoidalDf=data.frame(time,intensityTheoreticalSigmoidal)
      dplyr::full_join(intensityTheoreticalSigmoidalDf,dataFrameInput)->dataFrameInput
    }
  }

  # DOUBLE SIGMOIDAL
  if(!is.null(doubleSigmoidalFitVector))
  {
    if(!doubleSigmoidalFitVector$model=="doublesigmoidal")
    {stop("provided doubleSigmoidalFitVector is not a double sigmoidal fit vector")}
    if(!doubleSigmoidalFitVector$isThisaFit)
    {warning("provided double sigmoidal fit vector does not include a fit!")}
    if(doubleSigmoidalFitVector$isThisaFit)
    {
      # GENERATE THE TIME SERIES FOR FITTED DATA
      time=seq(0,
               doubleSigmoidalFitVector$dataScalingParameters.timeRatio,
               doubleSigmoidalFitVector$dataScalingParameters.timeRatio/1000)
      intensityTheoreticalDoubleSigmoidal=doublesigmoidalFitFormula(time,
                                                                    finalAsymptoteIntensity=doubleSigmoidalFitVector$finalAsymptoteIntensity_Estimate,
                                                                    maximum=doubleSigmoidalFitVector$maximum_Estimate,
                                                                    slope1=doubleSigmoidalFitVector$slope1_Estimate,
                                                                    midPoint1=doubleSigmoidalFitVector$midPoint1_Estimate,
                                                                    slope2=doubleSigmoidalFitVector$slope2_Estimate,
                                                                    midPointDistance=doubleSigmoidalFitVector$midPointDistance_Estimate)
      intensityTheoreticalDoubleSigmoidalDf=data.frame(time,intensityTheoreticalDoubleSigmoidal)
      dplyr::full_join(intensityTheoreticalDoubleSigmoidalDf,dataFrameInput)->dataFrameInput
    }
  }

  output=ggplot2::ggplot(dataFrameInput)+
    ggplot2::geom_point(ggplot2::aes(x=dataFrameInput$time, y=dataFrameInput$intensity))+
    ggplot2::expand_limits(x = 0, y = 0)+
    ggplot2::theme_bw()+
    ggplot2::theme(panel.grid.minor = ggplot2::element_blank())+
    ggplot2::xlab(xlabelText)+ggplot2::ylab(ylabelText)

  if(!is.null(sigmoidalFitVector))
  {
    if(sigmoidalFitVector$isThisaFit)
    {
      output=output+ggplot2::geom_line(ggplot2::aes(x=time,y=intensityTheoreticalSigmoidal),color="orange",size=1.5)
      if(showParameterRelatedLines)
      {
        # Points related with the sigmoidal fit line
        output=output+
          ggplot2::geom_point(ggplot2::aes(x=sigmoidalFitVector$midPoint_Estimate,
                                           y=(sigmoidalFitVector$maximum_Estimate)/2),
                              colour="red",size=6,shape=13)

        # Lines related with the sigmoidal fit line
        output=output+
          ggplot2::geom_hline(ggplot2::aes(yintercept=0),colour="#bdbdbd",size=0.5,linetype="longdash")+
          ggplot2::geom_hline(ggplot2::aes(yintercept=sigmoidalFitVector$maximum_Estimate),
                              colour="#bdbdbd",size=0.5,linetype="longdash")+
          ggplot2::geom_segment(ggplot2::aes(x = sigmoidalFitVector$midPoint_Estimate -
                                               2/(1*sigmoidalFitVector$slope_Estimate),
                                             y = 0,
                                             xend = sigmoidalFitVector$midPoint_Estimate +
                                               2/(1*sigmoidalFitVector$slope_Estimate),
                                             yend = sigmoidalFitVector$maximum_Estimate),
                                colour="#bdbdbd",size=0.5,linetype="longdash")
      }
    }
  }

  if(!is.null(doubleSigmoidalFitVector))
  {
    if(doubleSigmoidalFitVector$isThisaFit)
    {
      output=output+ggplot2::geom_line(ggplot2::aes(x=time,y=intensityTheoreticalDoubleSigmoidal),color="orange",size=1.5)
      if(showParameterRelatedLines)
      {
        if(!doubleSigmoidalFitVector$numericalParameters)
        {stop("to show parameter related lines one needs to run numericalReCalculation for doubleSigmoidalModel ")}
        if(doubleSigmoidalFitVector$numericalParameters)
        {
          # Points related with the double sigmoidal fit line (with numerical correction)
          output=output+
            ggplot2::geom_point(ggplot2::aes(x=doubleSigmoidalFitVector$numerical.maximum_x_Estimate,
                                             y=doubleSigmoidalFitVector$numerical.maximum_y_Estimate),
                                colour="red",size=6,shape=13)+
            ggplot2::geom_point(ggplot2::aes(x=doubleSigmoidalFitVector$numerical.midPoint1_x_Estimate,
                                             y=doubleSigmoidalFitVector$numerical.midPoint1_y_Estimate),
                                colour="red",size=6,shape=13)+
            ggplot2::geom_point(ggplot2::aes(x=doubleSigmoidalFitVector$numerical.midPoint2_x_Estimate,
                                             y=doubleSigmoidalFitVector$numerical.midPoint2_y_Estimate),
                                colour="red",size=6,shape=13)

          # Lines related with the double sigmoidal fit line (with numerical correction)
          output=output+
            ggplot2::geom_hline(ggplot2::aes(yintercept=0),colour="#bdbdbd",size=0.5,linetype="longdash")+
            ggplot2::geom_hline(ggplot2::aes(yintercept=doubleSigmoidalFitVector$numerical.maximum_y_Estimate),
                                colour="#bdbdbd",size=0.5,linetype="longdash")+
            ggplot2::geom_segment(ggplot2::aes(x = doubleSigmoidalFitVector$numerical.maximum_x_Estimate,
                                               y = doubleSigmoidalFitVector$numerical.maximum_y_Estimate*
                                                 doubleSigmoidalFitVector$finalAsymptoteIntensity_Estimate,
                                               xend = Inf,
                                               yend = doubleSigmoidalFitVector$numerical.maximum_y_Estimate*
                                                 doubleSigmoidalFitVector$finalAsymptoteIntensity_Estimate),
                                  colour="#bdbdbd",size=0.5,linetype="longdash")+
            ggplot2::geom_segment(ggplot2::aes(x = doubleSigmoidalFitVector$numerical.midPoint1_x_Estimate -
                                                 doubleSigmoidalFitVector$maximum_Estimate/(doubleSigmoidalFitVector$numerical.slope1_Estimate*2),
                                               y = 0,
                                               xend = doubleSigmoidalFitVector$numerical.midPoint1_x_Estimate +
                                                 doubleSigmoidalFitVector$maximum_Estimate/(doubleSigmoidalFitVector$numerical.slope1_Estimate*2),
                                               yend = doubleSigmoidalFitVector$maximum_Estimate),
                                  colour="#bdbdbd",size=0.5,linetype="longdash")+
            ggplot2::geom_segment(ggplot2::aes(x = doubleSigmoidalFitVector$numerical.midPoint2_x_Estimate -
                                                 doubleSigmoidalFitVector$maximum_Estimate*(1-doubleSigmoidalFitVector$finalAsymptoteIntensity_Estimate)/(-doubleSigmoidalFitVector$numerical.slope2_Estimate*2),
                                               y = doubleSigmoidalFitVector$maximum_Estimate,
                                               xend = doubleSigmoidalFitVector$numerical.midPoint2_x_Estimate +
                                                 doubleSigmoidalFitVector$maximum_Estimate*(1-doubleSigmoidalFitVector$finalAsymptoteIntensity_Estimate)/(-doubleSigmoidalFitVector$numerical.slope2_Estimate*2),
                                               yend = doubleSigmoidalFitVector$maximum_Estimate*doubleSigmoidalFitVector$finalAsymptoteIntensity_Estimate),
                                  colour="#bdbdbd",size=0.5,linetype="longdash")

        }
      }
    }
  }

  # data should contain same source


  return(output)
}

#' @title Check is data came from the same source.
#'
#' @param dataInput a data frame composed of two columns. One is for time and the other is for intensity. Should be normalized data generated by normalizeData.
#' @param sigmoidalFitVector is the output of sigmoidalFitFunction. Default is NULL.
#' @param doubleSigmoidalFitVector is the output of double sigmoidal fit function. Default is NULL.
#'
#' @description Checks if the provided data and models came from same source by looking to ".dataInputName" columns of the inputs.
#' @return Returns TRUE if models can from same source, FALSE otherwise.
#' @export
#'
sameSourceDataCheck<-function(dataInput,
                              sigmoidalFitVector,
                              doubleSigmoidalFitVector)
{
  # do the checks for data input
  # decide if data input is a data frame or a normalized data frame
  isalist=(is.list(dataInput) & !is.data.frame(dataInput))
  if(isalist){sameSourceDataCheck.dataInput=dataInput$dataInputName}
  isadataframe=(is.data.frame(dataInput))
  if(isadataframe){sameSourceDataCheck.dataInput=NA}


  # do the checks for sigmoidal fit vector
  # decide if sigmoidal fit is provided or not
  if(is.null(sigmoidalFitVector)){sameSourceDataCheck.sigmoidal=NA}
  if(!is.null(sigmoidalFitVector))
  {sameSourceDataCheck.sigmoidal=sigmoidalFitVector$dataInputName}

  # do the checks for double sigmoidal fit vector
  # decide if double sigmoidal fit is provided or not
  if(is.null(doubleSigmoidalFitVector)){sameSourceDataCheck.doublesigmoidal=NA}
  if(!is.null(doubleSigmoidalFitVector))
  {sameSourceDataCheck.doublesigmoidal=doubleSigmoidalFitVector$dataInputName}

  # make decision
  sameSourceDataCheckVector=c(dataInput=sameSourceDataCheck.dataInput,
                              sigmoidal=sameSourceDataCheck.sigmoidal,
                              doublesigmoidal=sameSourceDataCheck.doublesigmoidal)
  sameSourceDataCheckVector <- sameSourceDataCheckVector[!is.na(sameSourceDataCheckVector)]
  if(length(sameSourceDataCheckVector)!=0)
  {
    if(!all(sameSourceDataCheckVector[1]==sameSourceDataCheckVector))
    {
      print(sameSourceDataCheckVector)
      stop("all data need to come from same source!")
    }
  }
}

