% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shrinkTVP.R
\name{shrinkTVP}
\alias{shrinkTVP}
\title{Markov Chain Monte Carlo (MCMC) for time-varying parameter models with shrinkage}
\usage{
shrinkTVP(formula, data, niter = 10000, nburn = round(niter/2),
  nthin = 1, learn_a_xi = TRUE, learn_a_tau = TRUE, a_xi = 0.1,
  a_tau = 0.1, learn_kappa2 = TRUE, learn_lambda2 = TRUE,
  kappa2 = 20, lambda2 = 20, hyperprior_param, c_tuning_par_xi = 1,
  c_tuning_par_tau = 1, display_progress = TRUE, ret_beta_nc = FALSE,
  sv = FALSE, sv_param, LPDS = FALSE, y_test, x_test)
}
\arguments{
\item{formula}{an object of class "formula": a symbolic representation of the model, as in the
function \code{lm}. For details, see \code{\link{formula}}.}

\item{data}{\emph{optional} data frame containing the response variable and the covariates. If not found in \code{data},
the variables are taken from \code{environment(formula)}, typically the environment from which \code{shrinkTVP}
is called. No \code{NA}s are allowed in the response variable and the covariates.}

\item{niter}{positive integer, indicating the number of MCMC iterations
to perform, including the burn-in. Has to be larger than or equal to \code{nburn} + 2. The default value is 10000.}

\item{nburn}{non-negative integer, indicating the number of iterations discarded
as burn-in. Has to be smaller than or equal to \code{niter} - 2. The default value is \code{round(niter / 2)}.}

\item{nthin}{positive integer, indicating the degree of thinning to be performed. Every \code{nthin} draw is kept and returned.
The default value is 1, implying that every draw is kept.}

\item{learn_a_xi}{logical value indicating whether to learn a_xi, the local adaptation parameter of the state variances.
The default value is \code{TRUE}.}

\item{learn_a_tau}{logical value indicating whether to learn a_tau, the local adaptation parameter of the mean of the
initial values of the states. The default value is \code{TRUE}.}

\item{a_xi}{positive, real number, indicating the (fixed) value for a_xi. Ignored if
\code{learn_a_xi} is \code{TRUE}. The default value is 0.1.}

\item{a_tau}{positive, real number, indicating the (fixed) value for a_tau. Ignored if
\code{learn_a_tau} is \code{TRUE}. The default value is 0.1.}

\item{learn_kappa2}{logical value indicating whether to learn kappa2, the global level of shrinkage for
the state variances. The default value is \code{TRUE}.}

\item{learn_lambda2}{logical value indicating whether to learn the lambda^2 parameter,
the global level of shrinkage for the mean of the initial values of the states. The default value is \code{TRUE}.}

\item{kappa2}{positive, real number, indicating the (fixed) value for kappa2. Ignored if
\code{learn_kappa2} is \code{TRUE}. The default value is 20.}

\item{lambda2}{positive, real number, indicating the (fixed) value for lambda2. Ignored if
\code{learn_lambda2} is \code{TRUE}. The default value is 20.}

\item{hyperprior_param}{\emph{optional} named list containing hyperparameter values.
Not all have to be supplied, with those missing being replaced by the default values.
Any list elements that are misnamed will be ignored and a warning will be thrown.
All hyperparameter values have to be positive, real numbers. The following hyperparameters can be
supplied:
\itemize{
\item \code{c0}: The default value is 2.5.
\item \code{g0}: The default value is 5.
\item \code{G0}: The default value is 5 / (2.5 - 1).
\item \code{cp}: The default value is 1.
\item \code{np}: The default value is 20.
\item \code{e1}: The default value is 0.001.
\item \code{e2}: The default value is 0.001.
\item \code{d1}: The default value is 0.001.
\item \code{d2}: The default value is 0.001.
\item \code{b_xi}: The default value is 10.
\item \code{b_tau}: The default value is 10.
\item \code{nu_xi}: The default value is 5.
\item \code{nu_tau}: The default value is 5.
}}

\item{c_tuning_par_xi}{positive, real number. Determines the standard deviation of the proposal
distribution for the Metropolis Hastings step for a_xi. Ignored if \code{learn_a_xi} is \code{FALSE}. The default
value is 1.}

\item{c_tuning_par_tau}{positive, real number. Determines the standard deviation of the proposal
distribution for the Metropolis Hastings step for a_tau. Ignored if \code{learn_a_tau} is \code{FALSE}. The default
value is 1.}

\item{display_progress}{logical value indicating whether the progress bar and other informative output should be
displayed. The default value is \code{TRUE}.}

\item{ret_beta_nc}{logical value indicating whether to output the non-centered states in addition to the centered ones.
The default value is \code{FALSE}.}

\item{sv}{logical value indicating whether to use stochastic volatility for the error of the observation
equation. For details please see \code{\link{stochvol}}, in particular \code{\link{svsample}}. The default value is
\code{FALSE}.}

\item{sv_param}{\emph{optional} named list containing hyperparameter values for the stochastic volatility
parameters. Not all have to be supplied, with those missing being replaced by the default values.
Any list elements that are misnamed will be ignored and a warning will be thrown. Ignored if
\code{sv} is \code{FALSE}. The following elements can be supplied:
\itemize{
\item \code{Bsigma_sv}: positive, real number. The default value is 1.
\item \code{a0_sv}: positive, real number. The default value is 5.
\item \code{b0_sv}: positive, real number. The default value is 1.5.
\item \code{bmu}: real number. The default value is 0.
\item \code{Bmu}: real number. larger than 0. The default value is 1.
}}

\item{LPDS}{logical value indicating whether the one step ahead log predictive density score should be returned.
If \code{LPDS} is \code{TRUE}, both \code{x_test} and \code{y_test} have to be supplied. The default value is \code{FALSE}.}

\item{y_test}{\emph{optional} real number. If \code{LPDS} is \code{TRUE} this value will be taken to be the true one step
ahead value and will be used to calculate the log predictive density score. Ignored if \code{LPDS} is \code{FALSE}.}

\item{x_test}{\emph{optional} object that can be coerced to a 1 x d data frame containing the covariates for the
calculation of the one step ahead log predictive density score. The column names have to exactly match the
names of the covariates provided in the formula or an error will be thrown. An intercept will be added to
\code{x_test} if an intercept was added to the covariates by the formula interface. Ignored if \code{LPDS} is \code{FALSE}.}
}
\value{
The value returned is a list object of class \code{shrinkTVP_res} containing
\item{\code{sigma2}}{\code{mcmc} object containing the parameter draws from the posterior distribution of \code{sigma2}.
If \code{sv} is \code{TRUE}, \code{sigma2} is additionally an \code{mcmc.tvp} object.}
\item{\code{theta_sr}}{an \code{mcmc} object containing the parameter draws from the posterior distribution of the square root of theta.}
\item{\code{beta_mean}}{an \code{mcmc} object containing the parameter draws from the posterior distribution of beta_mean.}
\item{\code{beta_nc}}{\emph{(optional)} \code{list} object containing an \code{mcmc.tvp} object for the parameter draws from the posterior
distribution of the non-centered states, one for each covariate. In the case that there is only one covariate, this becomes just
a single \code{mcmc.tvp} object. Not returned if \code{ret_beta_nc} is \code{FALSE}.}
\item{\code{beta}}{\code{list} object containing an \code{mcmc.tvp} object for the parameter draws from the posterior distribution of the centered
states, one for each covariate. In the case that there is only one covariate, this becomes just a single \code{mcmc.tvp} object.}
\item{\code{xi2}}{\code{mcmc} object containing the parameter draws from the posterior distribution of xi2.}
\item{\code{a_xi}}{\emph{(optional)} \code{mcmc} object containing the parameter draws from the posterior distribution of a_xi.
Not returned if \code{learn_a_xi} is \code{FALSE}.}
\item{\code{a_xi_acceptance}}{\emph{(optional)} \code{list} object containing acceptance statistics for the Metropolis Hastings algorithm for
a_xi. Not returned if \code{learn_a_xi} is \code{FALSE}.}
\item{\code{tau2}}{\code{mcmc} object containing the parameter draws from the posterior distribution of tau2.}
\item{\code{a_tau}}{\emph{(optional)} \code{mcmc} object containing the parameter draws from the posterior distribution of a_tau.
Not returned if \code{learn_a_tau} is \code{FALSE}.}
\item{\code{a_tau_acceptance}}{\emph{(optional)} \code{list} containing acceptance statistics for the Metropolis Hastings algorithm for
a_tau. Not returned if \code{learn_a_tau} is \code{FALSE}.}
\item{\code{kappa2}}{\emph{(optional)} \code{mcmc} object containing the parameter draws from the posterior distribution of kappa2.
Not returned if \code{learn_kappa2} is \code{FALSE}.}
\item{\code{lambda2}}{\emph{(optional)} \code{mcmc} object containing the parameter draws from the posterior distribution of lambda2.
Not returned if \code{learn_lambda2} is \code{FALSE}.}
\item{\code{C0}}{\emph{(optional)} \code{mcmc} object containing the parameter draws from the posterior distribution of C0.
Not returned if \code{sv} is \code{TRUE}.}
\item{\code{sv_mu}}{\emph{(optional)} \code{mcmc} object containing the parameter draws from the posterior distribution of the mu
parameter for the stochastic volatility model on the errors. Not returned if \code{sv} is \code{FALSE}.}
\item{\code{sv_phi}}{\emph{(optional)} \code{mcmc} object containing the parameter draws from the posterior distribution of the phi
parameter for the stochastic volatility model on the errors. Not returned if \code{sv} is \code{FALSE}.}
\item{\code{sv_sigma2}}{\emph{(optional)} \code{mcmc} object containing the parameter draws from the posterior distribution of the sigma2
parameter for the stochastic volatility model on the errors. Not returned if \code{sv} is \code{FALSE}.}
\item{\code{priorvals}}{\code{list} object containing hyperparameter values of the prior distributions, as specified by the user.}
\item{\code{model}}{\code{list} object containing the model matrix and model response used.}
\item{\code{summaries}}{\code{list} object containing a collection of summary statistics of the posterior draws.}
\item{\code{LPDS}}{\emph{(optional)} value of the log predictive density score, calculated with \code{y_test} and \code{x_test}.}
To display the output, use \code{plot} and \code{summary}. The \code{summary} method displays the specified prior values stored in
\code{priorvals} and the posterior summaries stored in \code{summaries}, while the \code{plot} method calls \code{coda}'s \code{plot.mcmc}
or the \code{plot.mcmc.tvp} method. Furthermore, all functions that can be appplied to \code{coda::mcmc} objects
(e.g. \code{coda::acfplot}) can be applied to all output elements that are \code{coda} compatible.
}
\description{
\code{shrinkTVP} samples from the joint posterior distribution of the parameters of a time-varying
parameter model with shrinkage, potentially including stochastiv volatility (SV), and returns the MCMC draws.
}
\details{
For details concerning the algorithm please refer to the paper by Bitto and Frühwirth-Schnatter (2019).
}
\examples{
\donttest{

## Example 1, learn everything
set.seed(123)
sim <- simTVP(theta = c(0.2, 0, 0), beta_mean = c(1.5, -0.3, 0))
data <- sim$data

res <- shrinkTVP(y ~ x1 + x2, data = data)
# summarize output
summary(res)


## Example 2, hierarchical Bayesian Lasso
res <- shrinkTVP(y ~ x1 + x2, data = data,
                learn_a_xi = FALSE, learn_a_tau = FALSE,
                a_xi = 1, a_tau = 1)


## Example 3, non-hierarchical Bayesian Lasso
res <- shrinkTVP(y ~ x1 + x2, data = data,
                learn_a_xi = FALSE, learn_a_tau = FALSE,
                a_xi = 1, a_tau = 1,
                learn_kappa2 = FALSE, learn_lambda2 = FALSE)


## Example 4, adding stochastic volatility
res <- shrinkTVP(y ~ x1 + x2, data = data,
                sv = TRUE)


## Example 4, changing some of the default hyperparameters
res <- shrinkTVP(y ~ x1 + x2, data = data,
                hyperprior_param = list(cp = 5,
                                        nu_xi = 10))
}

}
\references{
Bitto, A., & Frühwirth-Schnatter, S. (2019). "Achieving shrinkage in a time-varying parameter model framework."
\emph{Journal of Econometrics}, 210(1), 75-97. <doi:10.1016/j.jeconom.2018.11.006>
}
\seealso{
\code{\link{plot.shrinkTVP_res}}, \code{\link{plot.mcmc.tvp}}
}
\author{
Peter Knaus \email{peter.knaus@wu.ac.at}
}
