#' Scatterplot method for `summary.seroincidence.by` objects
#'
#' @param object
#' a `summary.seroincidence.by` object
#' (generated by applying the `summary()`
#' method to the output of [est_seroincidence_by()]).
#' @param xvar the name of a stratifying variable in `object`
#' @param alpha transparency for the points in the graph
#' (1 = no transparency, 0 = fully transparent)
#' @param shape shape argument for `geom_point()`
#' @param dodge_width width for jitter
#' @param CIs [logical], if `TRUE`, add CI error bars
#' @param color_var [character] which variable in `object` to use
#' to determine point color
#' @param group_var [character] which variable in `object` to use
#' to connect points with lines (`NULL` for no lines)
#' @param ... unused
#'
#' @return a [ggplot2::ggplot()] object
#' @export
#' @keywords internal
#' @example inst/examples/exm-strat_ests_scatterplot.R
#'
strat_ests_scatterplot <- function(
    object,
    xvar = strata(object)[1],
    alpha = .7,
    shape = 1,
    dodge_width = 0.001,
    CIs = FALSE, # nolint: object_name_linter
    color_var = "nlm.convergence.code",
    group_var = NULL,
    ...) {

  # Check if xvar exists in the dataset
  if (!is.element(xvar, names(object))) {
    cli::cli_abort(
      class = "unavailable_xvar",
      message = c(
        "The variable `{xvar}` specified by argument `xvar`
        does not exist in `object`.",
        "Please choose a column that exists in `object`."
      )
    )
  }

  color_label <- object[[color_var]] |> labelled::get_label_attribute()
  if (is.null(color_label)) color_label <- color_var

  plot1 <-
    object |>
    ggplot2::ggplot() +
    ggplot2::aes(
      x = get(xvar),
      y = .data$incidence.rate,
      group = if (!is.null(group_var)) .data[[group_var]],
      col = .data[[color_var]]
    ) +
    ggplot2::xlab(xvar) +
    ggplot2::ylab("Estimated incidence rate") +
    ggplot2::theme_linedraw() +
    ggplot2::theme(
      panel.grid.minor.x = ggplot2::element_blank(),
      panel.grid.minor.y = ggplot2::element_blank()
    ) +
    ggplot2::expand_limits(y = 0) +
    ggplot2::labs(col = color_label) +
    ggplot2::theme(legend.position = "bottom")

  if (CIs) {
    plot1 <- plot1 +
      ggplot2::geom_pointrange(
        alpha = alpha,
        position = ggplot2::position_dodge2(width = dodge_width),
        aes(ymin = .data$CI.lwr, ymax = .data$CI.upr)
      )

  } else {
    plot1 <- plot1 +
      ggplot2::geom_point(
        position = ggplot2::position_dodge2(width = dodge_width),
        shape = shape,
        alpha = alpha
      )

  }

  if (!is.null(group_var)) {
    plot1 <- plot1 +
      ggplot2::geom_line(
        position = ggplot2::position_dodge2(width = dodge_width),
        alpha = alpha
      )
  }


  return(plot1)

}
