% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/outreg.R
\name{outreg}
\alias{outreg}
\title{Creates a publication quality result table for
regression models. Works with models fitted with lm, glm, as well
as lme4.}
\usage{
outreg(modelList, type = "latex", modelLabels = NULL, varLabels = NULL,
  tight = TRUE, showAIC = FALSE, float = FALSE, request, runFuns,
  digits = 3, alpha = c(0.05, 0.01, 0.001), SElist = NULL,
  PVlist = NULL, Blist = NULL, title, label, gofNames,
  print.results = TRUE, browse = identical(type, "html") && interactive())
}
\arguments{
\item{modelList}{A regression model or an R list of regression
models. Default model names will be M1, M2, and so forth. User
specified names are allowed, such as \code{list("My Model" = m1,
"Her Model" = m2)}.  This is the currently recommended way to
supply model lables. This is less error prone than the use of the
modelLabels argument.}

\item{type}{Default = "latex". The alternatives are "html" and "csv"}

\item{modelLabels}{This is allowed, but discouraged. A vector of
character string variables, one for each element in
modelList. Will override the names in modelList.}

\item{varLabels}{To beautify the parameter names printed.  Must be
a named vector in the format c(parmname = "displayName", parmname =
"displayName"). Include as many parameters as desired, it is not
necessary to supply new labels for all of the
parameters.}

\item{tight}{Table format. If TRUE, parameter estimates and
standard errors are printed in a single column.  If FALSE,
parameter estimates and standard errors are printed side by side.}

\item{showAIC}{This is a legacy argument, before the \code{request} argument was created.
If TRUE, the AIC estimate is included with the diagnostic values. It has the same effect
as described by \code{request}.}

\item{float}{Default = FALSE. Include boilerplate for a LaTeX table
float, with the tabular markup inside it. Not relevant if type =
"html".}

\item{request}{Extra information to be retrieved from the summary(model)
and displayed. This must be a vector of named arguments, such as
c(adj.r.squared = "adj $R^2$", fstatistic = "F"). The name must be
a valid name of the output object, the value should be the label
the user wants printed in the table. See details.}

\item{runFuns}{A list of functions}

\item{digits}{Default = 3. How many digits after decimal sign are to be displayed.}

\item{alpha}{Default = c(0.05, 0.01, 0.001). I think stars are dumb, but enough
people have asked me for more stars that I'm caving in.}

\item{SElist}{Optional. Replacement standard errors. Must be a
list of named vectors. \code{outreg} uses the R \code{summary} to
retrieve standard errors, but one might instead want to use robust
or bootstrapped standard errors.  This argument may supply a new
SE vector for each fitted regression model, but it is also allowed
to supply the SE replacement for just one of the models. The
format should be \code{list("A Model Label" = c(0.1, 0.3, 0.4),
"Another Model Label" = c(0.4, 0.2, 0.3)}.  On the left, one must
use the same names that are used in the modelList argument.}

\item{PVlist}{Optional. A list of replacement "p values". It must be a list
of named vectors, similar in format to SElist. The which the
elements are the "p values" that the user wants to use for each
model.}

\item{Blist}{Optional. This is only needed in the rare case where a model's
parameters cannot be discerned from its summary. List must have names
for models, and vectors slope coefficient. See discussion of SElist and PVlist.}

\item{title}{A LaTeX caption for the table. Not relevant if type =
"html".}

\item{label}{A string to be used as a LaTeX label in the table to be
created. Not relevant if type = "html".}

\item{gofNames}{Optional pretty names. R regression summaries use
names like "sigma" or "r.squared" that we might want to revise for
presentation. I prefer to refer to "sigma" as "RMSE", but perhaps
you instead prefer something like \code{gofnames = c("sigma" = "That
Estimate I don't understand", "deviance" = "Another Mystery")}. The
words that you might replace are "sigma", "r.squared",
"deviance", "adj.r.squared", "fstatistic".}

\item{print.results}{Default TRUE, marked-up table will be displayed in session.
If FALSE, same result is returned as an object.}

\item{browse}{Display the regression model in a browse? Defaults to TRUE if type = "html"}
}
\value{
A character vector, one element per row of the regression table.
}
\description{
This provides "markup" that the user is will copy into a LaTeX
document. As of rockchalk 1.8.4, can also create HTML markup.
The rockchalk vignette demonstrates use of outreg in Sweave.
}
\details{
\code{outreg} returns a string vector. It is suggested that users
should save the outreg result and then use cat to save it. That is
myMod <- outreg(m1, ...)  cat(myMod, file = "myMod.html") or
cat(myMod, file = "myMod.tex".  In version 1.8.66, we write the
html file to a temporary location and display it in a web
browser. Many word processors will not accept a cut-and paste
transfer from the browser, they will, however, be able to open the
html file itself and automatically re-format it in the native
table format.

In version 1.8.111, an argument \code{print.results} was introduced.
This is TRUE by default, so the marked-up table is printed into
the session, and it is returned as well.  If the function should
run silently (as suggested in the last few versions), include
\code{print.results = TRUE}.

The table includes a minimally sufficient (in my opinion) model
summary.  It offers parameter estimates, standard errors, and
minimally sufficient goodness of fit.  My tastes tend toward
minimal tables, but users request more features, and
\code{outreg}'s interface hass been generalized to allow
specialized requests. See \code{request} and \code{runFuns}
arguments.

I don't want to write a separate table function for every
different kind of regression model that exists (how
exhausting). So I've tried to revise \code{outreg()} to work with
regression functions that follow the standard R framework. It is
known to work \code{lm} and \code{glm}, as well as \code{merMod}
class from \code{lme4}, but it will try to interact with other
kinds of regression models.  Those models should have methods
\code{summary()}, \code{coef()}, \code{vcov()} and \code{nobs()}.
Package writes should provide those, its not my job.

Do you want "robust standard errors"? P values calculated
according to some alternative logic?  Go ahead, calculate them in
your code, outreg will now accept them as arguments. As of Version
1.8.4, users can provide their own standard errors and/or p-values
for each model. Thus, if a model answers in the usual way to the
standard R request \code{coef(summary(model))}, outreg can work if
users supply standard errors.

About the customizations \code{request}.  The \code{request}
argument supplies a list of names of summary output elements that
are desired. The format is a pair, a value to be retrieved from
\code{summary(model)}, and a pretty name to be printed for
it. With the \code{lm()} regression, for example, one might want
the output of the F test and the adjusted R-square: Include
\code{request = c(adj.r.squared = "adj. $R^2$", "fstatistic" =
"F")}. The value on the left is the name of the desired
information in the summary object, while the value on the right is
\emph{any} valid LaTeX (or HTML) markup that the user desires to
display in the table. \code{request} terms that generate a single
numerical value will generally work fine, while requests that ask
for more structured information, such as the F test (including the
2 degrees of freedom values) may work (user feedback needed).

The \code{runFuns} argument is inspired by a user request: could
this include the BIC or other summaries that can be easily
calculated?  Any R function, such as \code{AIC} or \code{BIC},
should work, as long as it returns a single value.  This is a
two-part specification, a function name and a pretty label to be
used in printing. For example, \code{runFuns = c("AIC" = "Akaike
Criterion", "BIC" = "Schwartz Criterion", "logLik" = "LL")}.
}
\note{
There are many R packages that can be used to create LaTeX
regression tables. memisc, texreg, apsrtable, xtables, and rms are
some. This "outreg" version was in use in our labs before we were
aware that those packages were in development. It is not intended
as a competitor, it is just a slightly different version of the
same that is more suited to our needs.
}
\examples{
set.seed(2134234)
dat <- data.frame(x1 = rnorm(100), x2 = rnorm(100))
dat$y1 <- 30 + 5 * rnorm(100) + 3 * dat$x1 + 4 * dat$x2
dat$y2 <- rnorm(100) + 5 * dat$x2
m1 <- lm(y1 ~ x1, data = dat)
m2 <- lm(y1 ~ x2, data = dat)
m3 <- lm(y1 ~ x1 + x2, data = dat)
gm1 <- glm(y1 ~ x1, family = Gamma, data = dat)
outreg(m1, title = "My One Tightly Printed Regression", float = TRUE)
ex1 <- outreg(m1, title = "My One Tightly Printed Regression",
               float = TRUE, print.results = FALSE)
## Show markup, Save to file with cat()
cat(ex1)
## cat(ex1, file = "ex1.tex")
 
ex2 <- outreg(list("Fingers" = m1), tight = FALSE, 
    title = "My Only Spread Out Regressions", float = TRUE,
    alpha = c(0.05, 0.01, 0.001)) 

ex3 <- outreg(list("Model A" = m1, "Model B label with Spaces" = m2),
    varLabels = list(x1 = "Billie"), 
    title = "My Two Linear Regressions", request = c(fstatistic = "F"),
    print.results = FALSE)
cat(ex3)

ex4 <- outreg(list("Model A" = m1, "Model B" = m2),
    modelLabels = c("Overrides ModelA", "Overrides ModelB"),
    varLabels = list(x1 = "Billie"),
    title = "Note modelLabels Overrides model names")
cat(ex4)

ex5 <- outreg(list("Whichever" = m1, "Whatever" = m2),
    title = "Still have showAIC argument, as in previous versions",
    showAIC = TRUE, float = TRUE)
cat(ex5)

\donttest{
## Launches HTML browse
ex5html <- outreg(list("Whichever" = m1, "Whatever" = m2),
    title = "Still have showAIC argument, as in previous versions",
    showAIC = TRUE, type = "html")
## Could instead, make a file:
## fn <- "some_name_you_choose.html"
## cat(ex5html, file = fn)
## browseURL(fn)
## Open that HTML file in LibreOffice or MS Word
}

ex6 <- outreg(list("Whatever" = m1, "Whatever" =m2),
    title = "Another way to get AIC output",
    runFuns = c("AIC" = "Akaike IC"))
cat(ex6)

ex7 <- outreg(list("Amod" = m1, "Bmod" = m2, "Gmod" = m3),
       title = "My Three Linear Regressions", float = FALSE)
cat(ex7)

## A new feature in 1.85 is ability to provide vectors of beta estimates
## standard errors, and p values if desired. 
## Suppose you have robust standard errors!
if (require(car)){
   newSE <- sqrt(diag(car::hccm(m3)))
   ex8 <- outreg(list("Model A" = m1, "Model B" = m2, "Model C" = m3, "Model C w Robust SE" = m3),
        SElist= list("Model C w Robust SE" = newSE))
   cat(ex8)
}

ex11 <- outreg(list("I Love Long Titles" = m1,
          "Prefer Brevity" = m2,
          "Short" = m3), tight = FALSE, float = FALSE)
cat(ex11)

ex12 <- outreg(list("GLM" = gm1), float = TRUE)
cat(ex12)

ex13 <- outreg(list("OLS" = m1, "GLM" = gm1), float = TRUE,
        alpha = c(0.05, 0.01))
cat(ex13)

ex14 <- outreg(list(OLS = m1, GLM = gm1), float = TRUE,
    request = c(fstatistic = "F"), runFuns = c("BIC" = "BIC"))
cat(ex14)

ex15 <- outreg(list(OLS = m1, GLM = gm1), float = TRUE,
    request = c(fstatistic = "F"), runFuns = c("BIC" = "BIC"),
    digits = 5, alpha = c(0.01))

ex16 <- outreg(list("OLS 1" = m1, "OLS 2" = m2,  GLM = gm1), float = TRUE,
    request = c(fstatistic = "F"),
    runFuns = c("BIC" = "BIC", logLik = "ll"),
    digits = 5, alpha = c(0.05, 0.01, 0.001))

ex17 <- outreg(list("Model A" = gm1, "Model B label with Spaces" = m2),
    request = c(fstatistic = "F"),
    runFuns = c("BIC" = "Schwarz IC", "AIC" = "Akaike IC",
    "nobs" = "N Again?"))

## Here's a fit example from lme4.
if (require(lme4) && require(car)){
  fm1 <- lmer(Reaction ~ Days + (Days | Subject), sleepstudy)
  ex18 <- outreg(fm1)
  cat(ex18)
  ## Fit same with lm for comparison
  lm1 <- lm(Reaction ~ Days, sleepstudy)
  ## Get robust standard errors
  lm1rse <- sqrt(diag(car::hccm(lm1)))

  if(interactive()){
  ex19 <- outreg(list("Random Effects" = fm1, 
       "OLS" = lm1, "OLS Robust SE" = lm1),
       SElist = list("OLS Robust SE" = lm1rse), type = "html")
  }
  ## From the glmer examples
  gm2 <- glmer(cbind(incidence, size - incidence) ~ period + (1 | herd),
                   data = cbpp, family = binomial)
  lm2 <- lm(incidence/size ~ period,  data = cbpp)
  lm2rse <- sqrt(diag(car::hccm(lm2)))
  ## Lets see what MASS::rlm objects do? Mostly OK
  rlm2 <- MASS::rlm(incidence/size ~ period, data = cbpp)
  \donttest{
  ex20 <- outreg(list("GLMER" = gm2, "lm" = lm2, "lm w/robust se" = lm2,
            "rlm" = rlm2), SElist = list("lm w/robust se" = lm2rse),
            type = "html")
  } 
}
}
\author{
Paul E. Johnson \email{<pauljohn@ku.edu>}
}
\keyword{regression}
