#' Wrapper around the `params` global variable
#'
#' Returns an element of the global `params` variable that is normally used to send parameters
#' to a script from the `Makefile` generated by `rmake`. Script parameters may be defined with
#' the `params` argument of the [rRule()] or [markdownRule()] functions.
#'
#' @param name Name of the parameter
#' @param default Default value to be returned if the `params` global variable does not exist,
#' which typically occurs if the script is executed not from `Makefile`.
#' @return Function returns an element of given `name` from the `params` variable that is created
#' inside of the `Makefile` recipe. If the `params` global variable does not exist (the script
#' is likely to be executed directly, i.e., not from the Makefile generated by `rmake`),
#' the `default` value is returned and a warning is generated. If the `params` global variable
#' exists but it is not a list or the `name` element does not exist there, an error is thrown.
#' @seealso [rRule()], [markdownRule()]
#' @author Michal Burda
#' @examples
#' task <- getParam('task', 'default')
#'
#' @export
getParam <- function(name, default=NA) {
  if (!exists('params')) {
    warning(paste0('rmake parameters not found - using default value for "', name, '": ', default))
    return(default)
  } else {
    p <- get('params', envir=.GlobalEnv)
    if (is.list(p) && is.element(name, names(p))) {
      return(p[[name]])
    } else {
      stop(paste0('rmake parameter "', name, '" is required'))
    }
  }
}
