% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predictCox.R
\name{predictCox}
\alias{predictCox}
\title{Fast computation of survival probabilities, hazards and cumulative hazards from Cox regression models}
\usage{
predictCox(object, newdata = NULL, times, centered = TRUE,
  type = c("cumhazard", "survival"), keep.strata = TRUE,
  keep.times = TRUE, keep.newdata = FALSE, se = FALSE, band = FALSE,
  iid = FALSE, average.iid = FALSE, nSim.band = 10000,
  conf.level = 0.95, logTransform = TRUE, store.iid = "full")
}
\arguments{
\item{object}{The fitted Cox regression model object either
obtained with \code{coxph} (survival package) or \code{cph}
(rms package).}

\item{newdata}{A \code{data.frame} or \code{data.table} containing
the values of the predictor variables defining subject specific
predictions. Should have the same structure as the data set
used to fit the \code{object}.}

\item{times}{Time points at which to evaluate the predictions.}

\item{centered}{Logical. If \code{TRUE} return prediction at the
mean values of the covariates \code{fit$mean}, if \code{FALSE}
return a prediction for all covariates equal to zero.  in the
linear predictor. Will be ignored if argument \code{newdata} is
used.}

\item{type}{the type of predicted value. Choices are \itemize{
\item \code{"hazard"} the baseline hazard function when
argument \code{newdata} is not used and the hazard function
when argument \code{newdata} is used.  \item \code{"cumhazard"}
the cumulative baseline hazard function when argument
\code{newdata} is not used and the cumulative hazard function
when argument \code{newdata} is used.  \item \code{"survival"}
the survival baseline hazard function when argument
\code{newdata} is not used and the cumulative hazard function
when argument \code{newdata} is used.  } Several choices can be
combined in a vector of strings that match (no matter the case)
strings \code{"hazard"},\code{"cumhazard"}, \code{"survival"}.}

\item{keep.strata}{Logical. If \code{TRUE} add the (newdata) strata
to the output. Only if there any.}

\item{keep.times}{Logical. If \code{TRUE} add the evaluation times
to the output.}

\item{keep.newdata}{Logical. If \code{TRUE} add the value of the
covariates used to make the prediction in the output list.}

\item{se}{Logical. If \code{TRUE} add the standard error to the output.}

\item{band}{Logical. If \code{TRUE} add the confidence band to the output.}

\item{iid}{Logical. If \code{TRUE} add the influence function to the output.}

\item{average.iid}{Logical. If \code{TRUE} add the average of the influence function over \code{newdata} to the output.}

\item{nSim.band}{the number of simulations used to compute the quantiles
for the confidence bands..}

\item{conf.level}{Level of confidence.}

\item{logTransform}{Should the confidence intervals/bands be computed on the log (hazard) and
log(-log) (survival) scale and be backtransformed.
Otherwise they are computed on the original scale and truncated (if necessary).}

\item{store.iid}{Implementation used to estimate the influence function and the standard error.
Can be \code{"full"} or \code{"minimal"}. See the details section of \code{\link{calcSeCox}}.}

\item{...}{arguments to be passed to the function \code{iidCox}.}
}
\value{
A list with some or all of the following elements:
\itemize{
\item{times}: the time points at which the other elements are evaluated.
\item{hazard}: When argument \code{newdata} is not used the baseline hazard function, otherwise the predicted hazard function. 
\item{cumhazard}: When argument \code{newdata} is not used the cumulative baseline hazard function, otherwise the predicted cumulative hazard function. 
\item{cumhazard.se}: The standard errors of the predicted cumulative hazard function.
\item{survival}: When argument \code{newdata} is not used the survival probabilities corresponding to the baseline hazard, otherwise the predicted survival probabilities.
\item{survival.se}: The standard errors of the predicted survival.
\item{strata}: The strata variable.
}
}
\description{
Fast routine to get baseline hazards and subject specific hazards
as well as survival probabilities from a \code{survival::coxph} or \code{rms::cph} object
}
\details{
Not working with time varying predictor variables or
    delayed entry.  The centered argument enables us to reproduce
    the results obtained with the \code{basehaz} function from the
    survival package.
    
When setting \code{logTransform} to \code{TRUE}, the standard error that is returned is 
before back-transformation to the original scale.
}
\examples{
library(survival)

set.seed(10)
d <- sampleData(40,outcome="survival")
nd <- sampleData(4,outcome="survival")
d$time <- round(d$time,1)
fit <- coxph(Surv(time,event)~X1 + strata(X2) + X6,
             data=d, ties="breslow", x = TRUE, y = TRUE)
# table(duplicated(d$time))

predictCox(fit)
predictCox(fit,centered=FALSE,type="hazard")
predictCox(fit,centered=TRUE,type="hazard")
predictCox(fit, newdata=nd, times=c(3,8),se=TRUE)
predictCox(fit, newdata=nd, times=c(3,8),se=TRUE, logTransform = TRUE)
predictCox(fit, newdata=nd, times=c(3,8),se=TRUE, store.iid = "minimal")
predictCox(fit, newdata=nd, times = 5,iid=TRUE)

cbind(survival::basehaz(fit),predictCox(fit,type="cumhazard")$cumhazard)

# one strata variable
fitS <- coxph(Surv(time,event)~strata(X1)+X2,
              data=d, ties="breslow", x = TRUE, y = TRUE)

predictCox(fitS)
predictCox(fitS, newdata=nd, times = 1)

# two strata variables
set.seed(1)
d$U=sample(letters[1:5],replace=TRUE,size=NROW(d))
d$V=sample(letters[4:10],replace=TRUE,size=NROW(d))
nd$U=sample(letters[1:5],replace=TRUE,size=NROW(nd))
nd$V=sample(letters[4:10],replace=TRUE,size=NROW(nd))
fit2S <- coxph(Surv(time,event)~X1+strata(U)+strata(V)+X2,
              data=d, ties="breslow", x = TRUE, y = TRUE)

cbind(survival::basehaz(fit2S),predictCox(fit2S,type="cumhazard")$cumhazard)
predictCox(fit2S)
predictCox(fitS, newdata=nd, times = 3)


}
\author{
Brice Ozenne broz@sund.ku.dk, Thomas A. Gerds tag@biostat.ku.dk
}
