% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/flexscan.R
\name{rflexscan}
\alias{rflexscan}
\title{Detect spatial disease clusters using the flexible/circular scan statistic}
\usage{
rflexscan(x, y, lat, lon, name, observed, expected, population, nb,
  clustersize = 15, radius = 6370, stattype = "ORIGINAL",
  scanmethod = "FLEXIBLE", ralpha = 0.2, simcount = 999,
  rantype = "MULTINOMIAL", comments = "", verbose = FALSE)
}
\arguments{
\item{x}{An array of X-coordinates.}

\item{y}{An array of Y-coordinates.}

\item{lat}{An array of latitude.}

\item{lon}{An array of longitude.}

\item{name}{The name of each area.}

\item{observed}{An array of observed number of diseases.}

\item{expected}{An array of expected number of diseases under the null hypothesis. 
This is used on "Poisson" model.}

\item{population}{An array of background population at risk in each area. 
This is used on "Binomial" model.}

\item{nb}{A neighbours list or an adjacency matrix.}

\item{clustersize}{The number of maximum spatial cluster size to scan.}

\item{radius}{Radius of Earth to calculate a distance between two sets of latitude and
longitude. It is approximately 6370 km in Japan.}

\item{stattype}{Statistic type to be used (case-insensitive).
\describe{
  \item{"ORIGINAL"}{the likelihood ratio statistic by Kulldorff and
  Nagarwalla (1995)}
  \item{"RESTRICTED"}{the restricted likelihood ratio statistic by Tango 
  (2008), with a preset parameter \code{ralpha} for restriction}
}}

\item{scanmethod}{Scanning method to be used (case-insensitive).
\describe{
  \item{"FLEXIBLE"}{flexible scan statistic by Tango and Takahashi (2005)}
  \item{"CIRCULAR"}{circular scan statistic by Kulldorff (1997)}
}}

\item{ralpha}{Parameter for the restricted likelihood ratio statistic.}

\item{simcount}{The number of Monte Carlo replications to calculate a p-value for statistical
test.}

\item{rantype}{The type of random number for Monte Carlo simulation (case-insensitive).
\describe{
  \item{"MULTINOMIAL"}{Total number of cases in whole area is fixed. It can 
  be chosen in either Poisson or Binomial model.}
  \item{"POISSON"}{Total number of cases is not fixed. It can be chosen in 
  Poisson model.}
}}

\item{comments}{Comments for the analysis which will be written in summary.}

\item{verbose}{Print progress messages.}
}
\value{
An \code{rflexscan} object which contains analysis results and specified
parameters.
}
\description{
This function analyzes spatial count data using the flexible spatial scan 
statistic developed by Tango and Takahashi (2005) or Kulldorff's circular 
spatial scan statistic (1997), and detect spatial disease clusters.
}
\examples{
# load sample data (North Carolina SIDS data)
library(spdep)
data("nc.sids")

# calculate the expected numbers of cases
expected <- nc.sids$BIR74 * sum(nc.sids$SID74) / sum(nc.sids$BIR74)

# run FleXScan
fls <- rflexscan(x = nc.sids$x, y = nc.sids$y,
                 observed = nc.sids$SID74,
                 expected = expected,
                 name = rownames(nc.sids),
                 clustersize = 10,
                 nb = ncCR85.nb)

# print rflexscan object
print(fls)

# print properties of the most likely cluster
print(fls$cluster[[1]])

# print summary to the terminal
summary(fls)

# plot graph
plot(fls, col = palette())
labs <- 1:length(fls$cluster)
legend("bottomleft", legend = labs, col = palette(), lty = 1)

}
\references{
Tango T. and Takahashi K. (2005). A flexibly shaped spatial scan 
  statistic for detecting clusters, International Journal of Health 
  Geographics 4:11.
  
  Kulldorff M. and Nagarwalla N. (1995). Spatial disease clusters: 
  Detection and Inference. Statistics in Medicine 14:799-810.
  
  Kulldorff M. (1997). A spatial scan statistic. Communications in 
  Statistics: Theory and Methods, 26:1481-1496.
  
  Tango T. (2008). A spatial scan statistic with a restricted 
  likelihood ratio. Japanese Journal of Biometrics 29(2):75-95.
}
\seealso{
\link{summary.rflexscan}, \link{plot.rflexscan}, \link{choropleth}
}
