% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/roll_elements.R
\name{roll_elements}
\alias{roll_elements}
\alias{roll_elements_vec}
\title{Roll elements}
\usage{
roll_elements(
  data,
  cols = NULL,
  n = NULL,
  n_fn = NULL,
  n_col_name = ".n",
  overwrite = FALSE,
  ...
)

roll_elements_vec(data, n = NULL, n_fn = NULL, ...)
}
\arguments{
\item{data}{\code{vector} or \code{data.frame} to roll elements of. When a \code{data.frame} is
grouped, the rolling is applied group-wise.}

\item{cols}{Names of columns in \code{`data`} to roll.
If \code{NULL}, the \emph{index} is rolled and used to reorder \code{`data`}.

\strong{N.B.} only used when \code{`data`} is a \code{data.frame}.}

\item{n}{Number of positions to roll. A positive number rolls \emph{left/up}.
A negative number rolls \emph{right/down}.}

\item{n_fn}{Function to find \code{`n`}. Useful when \code{`data`} is a
grouped \code{data.frame} and \code{`n`} should depend on the rows in the group.

\strong{Input}: Each specified \code{vector}/\code{column} in \code{`data`}
is passed to the function as a separate argument.

\strong{Output}: It should return either a \code{vector}
with one \code{integer-like scalar} \emph{per column}
or a single \code{integer-like scalar} to use for all columns.

Can be created with \code{\link[rearrr:create_n_fn]{create_n_fn()}}.
See also \code{\link[rearrr:median_index]{median_index()}} and
\code{\link[rearrr:median_index]{quantile_index()}}.}

\item{n_col_name}{Name of new column with the applied \code{`n`} values.
If \code{NULL}, no column is added.}

\item{overwrite}{Whether to allow overwriting of columns with the
same name as \code{`n_col_name`}. (Logical)}

\item{...}{Extra arguments for \code{`n_fn`}.}
}
\value{
Rolled \code{`data`}.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

Rolls positions of elements.

Example:

Rolling \code{c(1, 2, 3, 4, 5)} with \code{`n = 2`} becomes:

\code{c(3, 4, 5, 1, 2)}

\code{roll_elements_vec()} takes and returns a \code{vector}.

Should not be confused with \code{\link[rearrr:roll_values]{roll_values()}},
which changes the \emph{values} of the elements and wraps to a given range.
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)

# Roll vector left
roll_elements(1:10, n = 2)

# Roll vector right and return the vector
roll_elements_vec(1:10, n = -2)

# Roll vector left by median index (rounded to 6)
roll_elements(3:12, n_fn = median_index)

# Roll vector right by median value (rounded to 8)
roll_elements(3:12, n_fn = create_n_fn(median, negate = TRUE))

# Pass extra arguments (here 'prob') to 'n_fn' via '...'
roll_elements(
  1:10,
  n_fn = quantile_index,
  prob = 0.2
)

#
# Roll data.frame
#

# Set seed
set.seed(1)

# Create a data frame
df <- data.frame(
  "x" = 1:20,
  "y" = runif(20) * 10,
  "g" = rep(1:4, each = 5)
)

# Roll rows left/up
roll_elements(df, n = 2)

# Roll rows right/down
roll_elements(df, n = -2)

# Roll 'x' column right/down
roll_elements(df, cols = "x", n = -2)

# Roll rows right by median index in each group
# Specify 'negate' for the 'median_index' function
roll_elements(
  df \%>\% dplyr::group_by(g),
  n_fn = median_index,
  negate = TRUE
)
}
\seealso{
Other roll functions: 
\code{\link{roll_values}()}

Other rearrange functions: 
\code{\link{center_max}()},
\code{\link{center_min}()},
\code{\link{closest_to}()},
\code{\link{furthest_from}()},
\code{\link{pair_extremes}()},
\code{\link{position_max}()},
\code{\link{position_min}()},
\code{\link{rev_windows}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{rearrange functions}
\concept{roll functions}
