#' Get citations in various formats from CrossRef.
#' 
#' @export
#' 
#' @param dois Search by a single DOI or many DOIs.
#' @param format Name of the format. One of "rdf-xml", "turtle", 
#'   "citeproc-json", "citeproc-json-ish", "text", "ris", "bibtex" (default), 
#'   "crossref-xml", "datacite-xml","bibentry", or "crossref-tdm". The format 
#'   "citeproc-json-ish" is a format that is not quite proper citeproc-json
#' @param style a CSL style (for text format only). See \code{\link{get_styles}}
#'   for options. Default: apa. If there's a style that CrossRef doesn't support
#'   you'll get a  \code{(500) Internal Server Error}
#' @param locale Language locale. See \code{?Sys.getlocale}
#' @param raw (logical) Return raw text in the format given by \code{format} 
#'   parameter. Default: FALSE
#' @template moreargs
#' @details See \url{http://www.crosscite.org/cn/} for more info on the Crossref
#'   Content Negotiation API service.
#'   
#'   DataCite DOIs: Some values of the \code{format} parameter won't work with 
#'   DataCite DOIs, i.e. "citeproc-json", "crossref-xml", "crossref-tdm", 
#'     "onix-xml". 
#'   
#'   MEDRA DOIs only work with "rdf-xml", "turtle", "citeproc-json-ish", "ris", 
#'   "bibtex", "bibentry", "onix-xml".
#'   
#'   See examples below.
#'   
#'   See \code{\link{cr_agency}}
#'   
#'   Note that the format type \code{citeproc-json} uses the CrossRef API at 
#'   \code{api.crossref.org}, while all others are content negotiated via
#'   \code{http://data.crossref.org}, \code{http://data.datacite.org} or 
#'   \code{http://data.medra.org}. DOI agency is checked first (see 
#'   \code{\link{cr_agency}}). 
#'   
#' @examples \dontrun{
#' cr_cn(dois="10.1126/science.169.3946.635")
#' cr_cn(dois="10.1126/science.169.3946.635", "citeproc-json")
#' cr_cn(dois="10.1126/science.169.3946.635", "citeproc-json-ish")
#' cr_cn("10.1126/science.169.3946.635", "rdf-xml")
#' cr_cn("10.1126/science.169.3946.635", "crossref-xml")
#' cr_cn("10.1126/science.169.3946.635", "text")
#' 
#' # return an R bibentry type
#' cr_cn("10.1126/science.169.3946.635", "bibentry")
#' cr_cn("10.6084/m9.figshare.97218", "bibentry")
#' 
#' # return an apa style citation
#' cr_cn("10.1126/science.169.3946.635", "text", "apa")
#' cr_cn("10.1126/science.169.3946.635", "text", "harvard3")
#' cr_cn("10.1126/science.169.3946.635", "text", "elsevier-harvard")
#' cr_cn("10.1126/science.169.3946.635", "text", "ecoscience")
#' cr_cn("10.1126/science.169.3946.635", "text", "heredity")
#' cr_cn("10.1126/science.169.3946.635", "text", "oikos")
#' 
#' # example with many DOIs
#' dois <- cr_r(2)
#' cr_cn(dois, "text", "apa")
#' 
#' # Cycle through random styles - print style on each try
#' stys <- get_styles()
#' foo <- function(x){
#'  cat(sprintf("<Style>:%s\n", x), sep = "\n\n")
#'  cat(cr_cn("10.1126/science.169.3946.635", "text", style=x))
#' }
#' foo(sample(stys, 1))
#' 
#' # Using DataCite DOIs
#' ## some formats don't work
#' # cr_cn("10.5284/1011335", "text")
#' # cr_cn("10.5284/1011335", "crossref-xml")
#' # cr_cn("10.5284/1011335", "crossref-tdm")
#' ## But most do work
#' cr_cn("10.5284/1011335", "datacite-xml")
#' cr_cn("10.5284/1011335", "rdf-xml")
#' # cr_cn("10.5284/1011335", "turtle")
#' cr_cn("10.5284/1011335", "citeproc-json-ish")
#' cr_cn("10.5284/1011335", "ris")
#' cr_cn("10.5284/1011335", "bibtex")
#' cr_cn("10.5284/1011335", "bibentry")
#' 
#' dois <- c('10.5167/UZH-30455','10.5167/UZH-49216','10.5167/UZH-503',
#'           '10.5167/UZH-38402','10.5167/UZH-41217')
#' cat(cr_cn(dois[1]))
#' cat(cr_cn(dois[2]))
#' cat(cr_cn(dois[3]))
#' cat(cr_cn(dois[4]))
#' 
#' # Using Medra DOIs
#' cr_cn("10.3233/ISU-150780", "onix-xml")
#' 
#' # Get raw output
#' cr_cn(dois = "10.1002/app.27716", format = "citeproc-json", raw = TRUE)
#' 
#' # sometimes messy DOIs even work
#' ## in this case, a DOI minting agency can't be found
#' ## but we proceed anyway, just assuming it's "crossref"
#' cr_cn("10.1890/0012-9615(1999)069[0569:EDILSA]2.0.CO;2")
#' }

`cr_cn` <- function(dois, format = "bibtex", style = 'apa', 
                    locale = "en-US", raw = FALSE, .progress="none", ...) {
  
  format <- match.arg(format, c("rdf-xml", "turtle", "citeproc-json",
                                "citeproc-json-ish", "text", "ris", "bibtex", 
                                "crossref-xml", "datacite-xml", "bibentry", 
                                "crossref-tdm", "onix-xml"))

  cn <- function(doi, ...){
    agency_id <- suppressWarnings(GET_agency_id(doi))
    if (is.null(agency_id)) {
      warning(doi, " agency not found - proceeding with 'crossref' ...", 
              call. = FALSE)
      agency_id <- "crossref"
    }
    
    url <- paste0("http://data.", agency_id, ".org/", doi)
    
    # check cn data provider
    if (!format %in% supported_cn_types[[agency_id]]) {
      stop(paste0("Format '", format, "' for '", doi, 
                  "' is not supported by the DOI registration agency: '",
                  agency_id, "'.\nTry one of the following formats: ", 
                  paste0(supported_cn_types[[agency_id]], collapse = ", ")))
    }
    pick <- c(
           "rdf-xml" = "application/rdf+xml",
           "turtle" = "text/turtle",
           "citeproc-json" = 
             "transform/application/vnd.citationstyles.csl+json",
           "citeproc-json-ish" = "application/vnd.citationstyles.csl+json",
           "text" = "text/x-bibliography",
           "ris" = "application/x-research-info-systems",
           "bibtex" = "application/x-bibtex",
           "crossref-xml" = "application/vnd.crossref.unixref+xml",
           "datacite-xml" = "application/vnd.datacite.datacite+xml",
           "bibentry" = "application/x-bibtex",
           "crossref-tdm" = "application/vnd.crossref.unixsd+xml",
           "onix-xml" = "application/vnd.medra.onixdoi+xml")
    type <- pick[[format]]
    if (format == "citeproc-json") {
      response <- GET(file.path("http://api.crossref.org/works", doi, type), 
                      make_rcrossref_ua(), ...)
    } else {
      if (format == "text") {
        type <- paste(type, "; style = ", style, "; locale = ", locale, 
                      sep = "")
      }
      response <- GET(url, ..., 
                      make_rcrossref_ua(),
                      add_headers(Accept = type, followlocation = TRUE))
    }
    warn_status(response)
    if (response$status_code < 202) {
      select <- c(
        "rdf-xml" = "text/xml",
        "turtle" = "text/plain",
        "citeproc-json" = "application/json",
        "citeproc-json-ish" = "application/json",
        "text" = "text/plain",
        "ris" = "text/plain",
        "bibtex" = "text/plain",
        "crossref-xml" = "text/xml",
        "datacite-xml" = "text/xml",
        "bibentry" = "text/plain",
        "crossref-tdm" = "text/xml",
        "onix-xml" = "text/xml")
      parser <- select[[format]]
      if (raw) {
        ct_utf8(response)
      } else {
        out <- ct_utf8(response)
        if (format == "text") {
          out <- gsub("\n", "", out)
        }
        if (format == "bibentry") {
          out <- parse_bibtex(out)
        }
        if (parser == "application/json") {
          out <- jsonlite::fromJSON(out)
        }
        if (parser == "text/xml") {
          out <- xml2::read_xml(out)
        }
        out
      }
    }
  }

  if (length(dois) > 1) {
    llply(dois, function(z, ...) {
      out <- try(cn(z, ...), silent = TRUE)
      if ("try-error" %in% class(out)) {
        warning(
          paste0("Failure in resolving '", z, 
                 "'. See error detail in results."),
          call. = FALSE
        )
        out <- list(doi = z, error = out[[1]])
      }
      return(out)
    }, .progress = .progress)
  } else {
    cn(dois, ...)
  }
}

parse_bibtex <- function(x){
  x <- gsub("@[Dd]ata", "@Misc", x)
  writeLines(x, "tmpscratch.bib")
  output <- read.bib("tmpscratch.bib")
  unlink("tmpscratch.bib")
  output
}

warn_status <- function(x) {
  if (x$status_code > 202) {
    mssg <- ct_utf8(x)
    if (!is.character(mssg)) {
      mssg <- if (x$status_code == 406) {
        "(406) - probably bad format type"
      } else {
        http_status(x)$message
      }
    } else {
      mssg <- paste(sprintf("(%s)", x$status_code), "-", mssg)
    }
    warning(
      sprintf(
        "%s w/ %s", 
        gsub("%2F", "/", httr::parse_url(x$url)$path), 
        mssg
      ),
      call. = FALSE
    )
  }
}

#' Get doi agency id to identify resource location
#' 
#' @param x doi
#' @keywords internal
GET_agency_id <- function(x, ...) {
  if (is.null(x)) {
    stop("no doi for doi agency check provided", call. = FALSE)
  }
  cr_agency(x)$agency$id
}

# Supported content types
# See http://www.crosscite.org/cn/
supported_cn_types <- list(
  crossref = c("rdf-xml", "turtle", "citeproc-json", "citeproc-json-ish", 
               "text", "ris", "bibtex", "crossref-xml", "bibentry", 
               "crossref-tdm"),
  datacite = c("rdf-xml", "turtle", "datacite-xml", "citeproc-json-ish", "text",
               "ris", "bibtex", "bibentry"),
  medra = c("rdf-xml", "turtle", "citeproc-json-ish", "ris", "bibtex", 
            "bibentry", "onix-xml")
)
