/*
 * Decompiled with CFR 0.152.
 */
package org.openscience.cdk.layout;

import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.primitives.Ints;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import javax.vecmath.Point2d;
import org.openscience.cdk.graph.GraphUtil;
import org.openscience.cdk.interfaces.IAtom;
import org.openscience.cdk.interfaces.IAtomContainer;
import org.openscience.cdk.interfaces.IBond;
import org.openscience.cdk.interfaces.IStereoElement;
import org.openscience.cdk.interfaces.ITetrahedralChirality;
import org.openscience.cdk.ringsearch.RingSearch;
import org.openscience.cdk.stereo.ExtendedTetrahedral;
import org.openscience.cdk.tools.LoggingToolFactory;

final class NonplanarBonds {
    private final IAtomContainer container;
    private final int[][] graph;
    private final RingSearch ringSearch;
    private final ITetrahedralChirality[] elements;
    private final Map<IAtom, Integer> atomToIndex;

    public static IAtomContainer assign(IAtomContainer container) {
        if (!Iterables.isEmpty((Iterable)container.stereoElements())) {
            new NonplanarBonds(container);
        }
        return container;
    }

    NonplanarBonds(IAtomContainer container) {
        this(container, GraphUtil.toAdjList((IAtomContainer)container));
    }

    NonplanarBonds(IAtomContainer container, int[][] g) {
        this.container = container;
        this.elements = new ITetrahedralChirality[container.getAtomCount()];
        this.graph = g;
        this.atomToIndex = Maps.newHashMapWithExpectedSize((int)container.getAtomCount());
        this.ringSearch = new RingSearch(container, this.graph);
        for (IBond bond : container.bonds()) {
            switch (bond.getStereo()) {
                case UP: 
                case UP_INVERTED: 
                case DOWN: 
                case DOWN_INVERTED: {
                    bond.setStereo(IBond.Stereo.NONE);
                }
            }
        }
        for (int i = 0; i < container.getAtomCount(); ++i) {
            IAtom atom = container.getAtom(i);
            this.atomToIndex.put(atom, i);
            if (atom.getPoint2d() != null) continue;
            throw new IllegalArgumentException("atom " + i + " had unset coordinates");
        }
        Integer[] foci = new Integer[container.getAtomCount()];
        int n = 0;
        for (IStereoElement element : container.stereoElements()) {
            if (!(element instanceof ITetrahedralChirality)) continue;
            ITetrahedralChirality tc = (ITetrahedralChirality)element;
            int focus = this.atomToIndex.get(tc.getChiralAtom());
            this.elements[focus] = tc;
            foci[n++] = focus;
        }
        Arrays.sort(foci, 0, n, new Comparator<Integer>(){

            @Override
            public int compare(Integer i, Integer j) {
                return -Ints.compare((int)NonplanarBonds.this.nAdjacentCentres(i), (int)NonplanarBonds.this.nAdjacentCentres(j));
            }
        });
        for (int i = 0; i < n; ++i) {
            this.label(this.elements[foci[i]]);
        }
        for (IStereoElement se : container.stereoElements()) {
            if (!(se instanceof ExtendedTetrahedral)) continue;
            this.label((ExtendedTetrahedral)se);
        }
    }

    private void label(ExtendedTetrahedral element) {
        IAtom focus = element.focus();
        IAtom[] atoms = element.peripherals();
        IBond[] bonds = new IBond[4];
        int p = this.parity(element.winding());
        List focusBonds = this.container.getConnectedBondsList(focus);
        if (focusBonds.size() != 2) {
            LoggingToolFactory.createLoggingTool(this.getClass()).warn((Object)"Non-cumulated carbon presented as the focus of extended tetrahedral stereo configuration");
            return;
        }
        IAtom[] terminals = element.findTerminalAtoms(this.container);
        IAtom left = terminals[0];
        IAtom right = terminals[1];
        bonds[0] = this.container.getBond(left, atoms[0]);
        bonds[1] = this.container.getBond(left, atoms[1]);
        bonds[2] = this.container.getBond(right, atoms[2]);
        bonds[3] = this.container.getBond(right, atoms[3]);
        int[] rank = new int[4];
        for (int i = 0; i < 4; ++i) {
            rank[i] = i;
        }
        p *= this.sortClockwise(rank, focus, atoms, 4);
        IBond.Stereo[] labels = new IBond.Stereo[4];
        for (int i = 0; i < 4; ++i) {
            int v = rank[i];
            labels[v] = (p *= -1) > 0 ? IBond.Stereo.UP : IBond.Stereo.DOWN;
        }
        int[] priority = new int[]{5, 5, 5, 5};
        int i = 0;
        for (int v : this.priority(this.atomToIndex.get(focus), atoms, 4)) {
            IBond bond = bonds[v];
            if (bond.getStereo() != IBond.Stereo.NONE || bond.getOrder() != IBond.Order.SINGLE) continue;
            priority[v] = i++;
        }
        if (priority[0] + priority[1] < priority[2] + priority[3]) {
            if (priority[0] < 5) {
                bonds[0].setAtoms(new IAtom[]{left, atoms[0]});
                bonds[0].setStereo(labels[0]);
            }
            if (priority[1] < 5) {
                bonds[1].setAtoms(new IAtom[]{left, atoms[1]});
                bonds[1].setStereo(labels[1]);
            }
        } else {
            if (priority[2] < 5) {
                bonds[2].setAtoms(new IAtom[]{right, atoms[2]});
                bonds[2].setStereo(labels[2]);
            }
            if (priority[3] < 5) {
                bonds[3].setAtoms(new IAtom[]{right, atoms[3]});
                bonds[3].setStereo(labels[3]);
            }
        }
    }

    private void label(ITetrahedralChirality element) {
        IAtom focus = element.getChiralAtom();
        IAtom[] atoms = element.getLigands();
        IBond[] bonds = new IBond[4];
        int p = this.parity(element.getStereo());
        int n = 0;
        if (p == 0) {
            return;
        }
        for (int i = 0; i < 4; ++i) {
            if (atoms[i] == focus) {
                p *= this.parity(i);
                continue;
            }
            bonds[n] = this.container.getBond(focus, atoms[i]);
            atoms[n] = atoms[i];
            ++n;
        }
        int[] rank = new int[n];
        for (int i = 0; i < n; ++i) {
            rank[i] = i;
        }
        p *= this.sortClockwise(rank, focus, atoms, n);
        int invert = -1;
        if (n == 3) {
            for (int i = 0; i < n; ++i) {
                Point2d a = atoms[rank[i]].getPoint2d();
                Point2d b = focus.getPoint2d();
                Point2d c = atoms[rank[(i + 2) % n]].getPoint2d();
                double det = (a.x - c.x) * (b.y - c.y) - (a.y - c.y) * (b.x - c.x);
                if (!(det > 0.0)) continue;
                invert = rank[(i + 1) % n];
                break;
            }
        }
        IBond.Stereo[] labels = new IBond.Stereo[n];
        for (int i = 0; i < n; ++i) {
            int v = rank[i];
            if (n == 4) {
                p *= -1;
            }
            labels[v] = invert == v ? (p > 0 ? IBond.Stereo.DOWN : IBond.Stereo.UP) : (p > 0 ? IBond.Stereo.UP : IBond.Stereo.DOWN);
        }
        for (int v : this.priority(this.atomToIndex.get(focus), atoms, n)) {
            IBond bond = bonds[v];
            if (bond.getStereo() != IBond.Stereo.NONE || bond.getOrder() != IBond.Order.SINGLE) continue;
            bond.setAtoms(new IAtom[]{focus, atoms[v]});
            bond.setStereo(labels[v]);
            return;
        }
        throw new IllegalArgumentException("could not assign non-planar (up/down) labels");
    }

    private int parity(int x) {
        return (x & 1) == 1 ? -1 : 1;
    }

    private int parity(ITetrahedralChirality.Stereo stereo) {
        switch (stereo) {
            case CLOCKWISE: {
                return -1;
            }
            case ANTI_CLOCKWISE: {
                return 1;
            }
        }
        return 0;
    }

    private int nAdjacentCentres(int i) {
        int n = 0;
        for (IAtom atom : this.elements[i].getLigands()) {
            if (this.elements[this.atomToIndex.get(atom)] == null) continue;
            ++n;
        }
        return n;
    }

    private int[] priority(int focus, IAtom[] atoms, int n) {
        int[] rank = new int[n];
        for (int i = 0; i < n; ++i) {
            rank[i] = i;
        }
        for (int j = 1; j < n; ++j) {
            int v = rank[j];
            int i = j - 1;
            while (i >= 0 && this.hasPriority(focus, this.atomToIndex.get(atoms[v]), this.atomToIndex.get(atoms[rank[i]]))) {
                rank[i + 1] = rank[i--];
            }
            rank[i + 1] = v;
        }
        return rank;
    }

    boolean hasPriority(int focus, int i, int j) {
        if (this.elements[i] == null && this.elements[j] != null) {
            return true;
        }
        if (this.elements[i] != null && this.elements[j] == null) {
            return false;
        }
        boolean iCyclic = this.ringSearch.cyclic(focus, i);
        boolean jCyclic = this.ringSearch.cyclic(focus, j);
        if (!iCyclic && jCyclic) {
            return true;
        }
        if (iCyclic && !jCyclic) {
            return false;
        }
        if (this.graph[i].length < this.graph[j].length) {
            return true;
        }
        if (this.graph[i].length > this.graph[j].length) {
            return false;
        }
        if (this.container.getAtom(i).getAtomicNumber() < this.container.getAtom(j).getAtomicNumber()) {
            return true;
        }
        if (this.container.getAtom(i).getAtomicNumber() > this.container.getAtom(j).getAtomicNumber()) {
            return false;
        }
        return false;
    }

    private int sortClockwise(int[] indices, IAtom focus, IAtom[] atoms, int n) {
        int x = 0;
        for (int j = 1; j < n; ++j) {
            int v = indices[j];
            int i = j - 1;
            while (i >= 0 && NonplanarBonds.less(v, indices[i], atoms, focus.getPoint2d())) {
                indices[i + 1] = indices[i--];
                ++x;
            }
            indices[i + 1] = v;
        }
        return this.parity(x);
    }

    static boolean less(int i, int j, IAtom[] atoms, Point2d center) {
        Point2d a = atoms[i].getPoint2d();
        Point2d b = atoms[j].getPoint2d();
        if (a.x - center.x >= 0.0 && b.x - center.x < 0.0) {
            return true;
        }
        if (a.x - center.x < 0.0 && b.x - center.x >= 0.0) {
            return false;
        }
        if (a.x - center.x == 0.0 && b.x - center.x == 0.0) {
            if (a.y - center.y >= 0.0 || b.y - center.y >= 0.0) {
                return a.y > b.y;
            }
            return b.y > a.y;
        }
        double det = (a.x - center.x) * (b.y - center.y) - (b.x - center.x) * (a.y - center.y);
        if (det < 0.0) {
            return true;
        }
        if (det > 0.0) {
            return false;
        }
        double d1 = (a.x - center.x) * (a.x - center.x) + (a.y - center.y) * (a.y - center.y);
        double d2 = (b.x - center.x) * (b.x - center.x) + (b.y - center.y) * (b.y - center.y);
        return d1 > d2;
    }
}

