% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/func_utils.R
\name{psd-utilities}
\alias{psd-utilities}
\alias{na_locf}
\alias{na_locf.matrix}
\alias{na_locf.default}
\alias{vardiff}
\alias{varddiff}
\alias{varddiff.spec}
\alias{varddiff.default}
\alias{create_poly}
\alias{dB}
\alias{decibels}
\alias{db}
\alias{vector_reshape}
\alias{colvec}
\alias{rowvec}
\alias{is.spec}
\alias{is.amt}
\alias{is.tapers}
\alias{na_mat}
\alias{zeros}
\alias{ones}
\alias{mod}
\title{Various utility functions.}
\usage{
na_locf(x)

\method{na_locf}{matrix}(x)

\method{na_locf}{default}(x)

vardiff(x, double.diff = FALSE)

varddiff(x)

\method{varddiff}{spec}(x)

\method{varddiff}{default}(x)

create_poly(x, y, dy, from.lower = FALSE)

dB(Rat, invert = FALSE, pos.only = TRUE, is.power = FALSE)

vector_reshape(x, vec.shape = c("horizontal", "vertical"))

colvec(x)

rowvec(x)

is.spec(Obj)

is.amt(Obj)

is.tapers(Obj)

na_mat(nrow, ncol = 1)

zeros(nrow)

ones(nrow)

mod(x, y)
}
\arguments{
\item{x, y}{objects; in \code{\link{create_poly}} these are the vectors used to
create a \code{\link{polygon}}-compatible sequence (\code{x} is sorted by default);
in \code{\link{mod}} these are the "numerator" and "denominator", respectively.}

\item{double.diff}{logical; should the double difference be used instead?}

\item{dy}{numeric; the distance from \code{y} to the top and bottom of
the polygonal surfaces; see \code{from.lower}}

\item{from.lower}{logical; should the bottom be \code{y} instead of \code{y+dy}, so that
\code{dy} represents the distance from the lower surface?}

\item{Rat}{numeric; the values -- ratios -- to convert to decibels (\code{dB}).}

\item{invert}{logical; assumes \code{Rat} is already in decibels, so return ratio}

\item{pos.only}{logical; if \code{invert=FALSE}, sets negative or zero values to NA}

\item{is.power}{logical; should the factor of 2 be included in the decibel calculation?}

\item{vec.shape}{choice between horizontally-long or vertically-long vector.}

\item{Obj}{An object to test for class inheritance.}

\item{nrow, ncol}{integer; the number of rows and/or columns to create}
}
\value{
\code{vector_reshape} returns a "reshaped" vector, meaning it has
had it's dimensions changes so that it has either one row 
(if \code{vec.shape=="horizontal"}), or one column (\code{"vertical"}).

\code{is.spec}, \code{is.amt}, and \code{is.tapers} return the output of \code{\link{inherits}}.

\code{na_mat} returns a matrix of dimensions \code{(nrow,ncol)} with
\code{NA} values, the representation of which is set by \code{NA_real_}

\code{mod} returns the result of a modulo division, which is 
equivalent to \code{(x) \%\% (y)}.
}
\description{
\emph{The various utility functions are:}

\code{\link{na_locf}} is meant as a simple replacement for zoo::na.locf
which carries the last observation forward; here we force both directions, meaning
the first observation is carried backwards as well.

\code{\link{vardiff}} returns the variance of the first (or second) 
difference of the series. \code{\link{varddiff}} is a convenience wrapper
to return variance for the second difference.

\code{\link{create_poly}} generates an x-y sequence compatible for use with \code{\link{polygon}}

\code{\link{dB}} returns an object converted to decibels.

\code{\link{vector_reshape}} reshapes a vector into another vector.

\code{\link{colvec}} returns the object as a vertically long vector; whereas
\code{\link{rowvec}} returns the object as a horizontally long vector.

\code{\link{is.spec}} and \code{\link{is.amt}} report whether an object has class \code{'spec'} or \code{'amt'}, as
would one returned by, for example, \code{\link{spectrum}} or \code{\link{psdcore}}.

\code{\link{is.tapers}} reports whether an object has class \code{'tapers'}, as
would one returned by, for example, \code{\link{as.tapers}}.

\code{\link{na_mat}} populates a matrix of specified dimensions 
with \code{NA} values.

\code{\link{zeros}} populate a column-wise matrix with zeros; whereas,
\code{\link{ones}} populates a column-wise matrix with ones.  \emph{Note that 
\code{n} is enforced to be at least 1 for both functions.}

\code{\link{mod}} finds the modulo division of two values
}
\details{
Decibels are defined as \eqn{10 \log{}_{10} \frac{X_1}{X_2}}, 
unless \code{is.power=TRUE} in which \eqn{\mathrm{db} X^2 \equiv 20 \log{}_{10} X^2}

\code{colvec, rowvec} are simple wrapper functions to \code{vector_reshape}.

Modulo division has higher order-of-operations ranking than other
arithmetic operations; hence, \code{x + 1 \%\% y} is equivalent to
\code{x + (1 \%\% y)} which can produce confusing results. \code{mod}
is simply a series of \code{trunc} commands which
reduces the chance for unintentionally erroneous results.
}
\note{
The performance of \code{\link{mod}} has not been tested against the 
\code{\%\%} arithmetic method -- it may or may not be slower for large
numeric vectors.
}
\examples{
\dontrun{#REX
library(psd)

##
## Various utilities
##

set.seed(1234)
X <- rnorm(1e2)

#
# Matrix and vector creation:
#
# NA matrix
nd <- 5
na_mat(nd)
na_mat(nd,nd-1)

# zeros
zeros(nd)

# and ones
ones(nd)

#
# Check for tapers object:
#
is.tapers(X)
is.tapers(as.tapers(X))

#
# Check for spec object:
#
PSD <- spectrum(X, plot=FALSE)
plot(PSD)
# return is class 'spec'
is.spec(PSD) # TRUE
# but the underlying structure is just a list
PSD <- unclass(PSD)
is.spec(PSD) # FALSE

#
# decibels
#
dB(1) # signal is equal <--> zero dB
sig <- 1e-10
all.equal(sig, dB(dB(sig), invert=TRUE))
pow <- sig**2
all.equal(pow, dB(dB(sig, is.power=TRUE), invert=TRUE, is.power=TRUE))

# 
# Variance of difference series
#
vardiff(X) # first difference
varddiff(X) # second difference
all.equal(vardiff(X, TRUE), varddiff(X))

#
# modulo division
#
x <- 1:10
mc1a <- mod(1,2)
mc2a <- mod(1+x,2)
mc1b <- 1 \%\% 2
mc2b <- 1 + x \%\% 2
mc2c <- (1 + x) \%\% 2
all.equal(mc1a, mc1b) # TRUE
all.equal(mc2a, mc2b) # "Mean absolute difference: 2"
all.equal(mc2a, mc2c) # TRUE
# on a series
modulo_floor(1:10) # defaults to 2
modulo_floor(1:10, 3)

}#REX
}
\references{
For \code{\link{mod}}: see Peter Dalgaard's explanation of 
the non-bug (#14771) I raised (instead I should've asked it on R-help): 
\url{https://bugs.r-project.org/bugzilla3/show_bug.cgi?id=14771\#c2}
}
\seealso{
\code{\link{psd-package}}, \code{\link{as.tapers}}, \code{\link{modulo_floor}}
}
\author{
A.J. Barbour
}
