% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/latent.R
\name{cl_agg}
\alias{cl_agg}
\title{Weighted averaging within clusters of parameter draws}
\usage{
cl_agg(
  draws,
  cl = seq_len(nrow(draws)),
  wdraws = rep(1, nrow(draws)),
  eps_wdraws = 0
)
}
\arguments{
\item{draws}{An \eqn{S \times P}{S x P} matrix of parameter draws, with
\eqn{P} denoting the number of parameters.}

\item{cl}{A numeric vector of length \eqn{S}, giving the cluster indices for
the draws. Draws that should be dropped (e.g., by thinning) need to have an
\code{NA} in \code{cl}.}

\item{wdraws}{A numeric vector of length \eqn{S}, giving the weights of the
draws. It doesn't matter whether these are normalized (i.e., sum to \code{1}) or
not because internally, these weights are normalized to sum to \code{1} within
each cluster. Draws that should be dropped (e.g., by thinning) can (but
must not necessarily) have an \code{NA} in \code{wdraws}.}

\item{eps_wdraws}{A positive numeric value (typically small) which will be
used to improve numerical stability: The weights of the draws within each
cluster are multiplied by \code{1 - eps_wdraws}. The default of \code{0} should be
fine for most cases; this argument only exists to help in those cases where
numerical instabilities occur (which must be detected by the user; this
function will not detect numerical instabilities itself).}
}
\value{
An \eqn{S_{\mathrm{cl}} \times P}{S_cl x P} matrix of aggregated
parameter draws.
}
\description{
This function aggregates \eqn{S} parameter draws that have been clustered
into \eqn{S_{\mathrm{cl}}}{S_cl} clusters by averaging across the draws that
belong to the same cluster. This averaging can be done in a weighted fashion.
}
\examples{
set.seed(323)
S <- 100L
P <- 3L
draws <- matrix(rnorm(S * P), nrow = S, ncol = P)
# Clustering example:
S_cl <- 10L
cl_draws <- sample.int(S_cl, size = S, replace = TRUE)
draws_cl <- cl_agg(draws, cl = cl_draws)
# Clustering example with nonconstant `wdraws`:
w_draws <- rgamma(S, shape = 4)
draws_cl <- cl_agg(draws, cl = cl_draws, wdraws = w_draws)
# Thinning example (implying constant `wdraws`):
S_th <- 50L
idxs_thin <- round(seq(1, S, length.out = S_th))
th_draws <- rep(NA, S)
th_draws[idxs_thin] <- seq_len(S_th)
draws_th <- cl_agg(draws, cl = th_draws)

}
