#' @include internal.R ConservationProblem-class.R
NULL

#' Evaluate number of planning units selected by solution
#'
#' Calculate the number of planning units selected within a solution
#' to a conservation planning problem.
#'
#' @inheritParams eval_cost_summary
#'
#' @inheritSection eval_cost_summary Solution format
#'
#' @details
#' This summary statistic is calculated as the sum of the values in
#' the solution. As a consequence, this metric can produce a
#' non-integer value (e.g., 4.3) for solutions containing proportion values
#' (e.g., generated by solving a [problem()] built using the
#' [`add_proportion_decisions()`] function).
#'
#' @return A [tibble::tibble()] object containing the number of planning
#'   units selected within a solution.
#'   It contains the following columns:
#'
#'   \describe{
#'
#'   \item{summary}{`character` description of the summary statistic.
#'     The statistic associated with the `"overall"` value
#'     in this column is calculated using the entire solution
#'     (including all management zones if there are multiple zones).
#'     If multiple management zones are present, then summary statistics
#'     are also provided for each zone separately
#'     (indicated using zone names).}
#'
#'   \item{n}{`numeric` number of selected planning units.}
#'
#'   }
#'
#' @name eval_n_summary
#'
#' @seealso
#' See [summaries] for an overview of all functions for summarizing solutions.
#'
#' @family summaries
#'
#' @examples
#' \dontrun{
#' # set seed for reproducibility
#' set.seed(500)
#'
#' # load data
#' sim_pu_raster <- get_sim_pu_raster()
#' sim_pu_polygons <- get_sim_pu_polygons()
#' sim_features <- get_sim_features()
#' sim_zones_pu_polygons <- get_sim_zones_pu_polygons()
#' sim_zones_features <- get_sim_zones_features()
#'
#' # build minimal conservation problem with raster data
#' p1 <-
#'   problem(sim_pu_raster, sim_features) %>%
#'   add_min_set_objective() %>%
#'   add_relative_targets(0.1) %>%
#'   add_binary_decisions() %>%
#'   add_default_solver(verbose = FALSE)
#'
#' # solve the problem
#' s1 <- solve(p1)
#'
#' # print solution
#' print(s1)
#'
#' # plot solution
#' plot(s1, main = "solution", axes = FALSE)
#'
#' # calculate number of selected planning units within solution
#' r1 <- eval_n_summary(p1, s1)
#' print(r1)
#'
#' # build minimal conservation problem with polygon data
#' p2 <-
#'   problem(sim_pu_polygons, sim_features, cost_column = "cost") %>%
#'   add_min_set_objective() %>%
#'   add_relative_targets(0.1) %>%
#'   add_binary_decisions() %>%
#'   add_default_solver(verbose = FALSE)
#'
#' # solve the problem
#' s2 <- solve(p2)
#'
#' # plot solution
#' plot(s2[, "solution_1"])
#'
#' # print solution
#' print(s2)
#'
#' # calculate number of selected planning units within solution
#' r2 <- eval_n_summary(p2, s2[, "solution_1"])
#' print(r2)
#'
#' # manually calculate number of selected planning units
#' r2_manual <- sum(s2$solution_1, na.rm = TRUE)
#' print(r2_manual)
#'
#' # build multi-zone conservation problem with polygon data
#' p3 <-
#'   problem(
#'     sim_zones_pu_polygons, sim_zones_features,
#'     cost_column = c("cost_1", "cost_2", "cost_3")
#'   ) %>%
#'   add_min_set_objective() %>%
#'   add_relative_targets(matrix(runif(15, 0.1, 0.2), nrow = 5, ncol = 3)) %>%
#'   add_binary_decisions() %>%
#'   add_default_solver(verbose = FALSE)
#'
#' # solve the problem
#' s3 <- solve(p3)
#'
#' # print solution
#' print(s3)
#'
#' # create new column representing the zone id that each planning unit
#' # was allocated to in the solution
#' s3$solution <- category_vector(
#'   s3[, c("solution_1_zone_1", "solution_1_zone_2", "solution_1_zone_3")]
#' )
#' s3$solution <- factor(s3$solution)
#'
#' # plot solution
#' plot(s3[, "solution"])
#'
#' # calculate number of selected planning units within solution
#' r3 <- eval_n_summary(
#'   p3, s3[, c("solution_1_zone_1", "solution_1_zone_2", "solution_1_zone_3")]
#' )
#' print(r3)
#' }
#' @export
eval_n_summary <- function(x, solution) {
  # assert arguments are valid
  assert_required(x)
  assert_required(solution)
  assert(is_conservation_problem(x))
  # convert solution to status matrix format
  solution <- planning_unit_solution_status(x, solution)
  # convert NA values in solution to 0
  solution[is.na(solution)] <- 0
  # calculate overall number of selected planning units
  total_n <- sum(solution)
  if (x$number_of_zones() > 1) {
    out <- tibble::tibble(
      summary = c("overall", zone_names(x)),
      n = c(total_n, unname(colSums(solution)))
    )
  } else {
    out <- tibble::tibble(summary = "overall", n = total_n)
  }
  # return result
  out
}
