% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eval_cost_summary.R
\name{eval_cost_summary}
\alias{eval_cost_summary}
\title{Evaluate cost of solution}
\usage{
eval_cost_summary(x, solution)
}
\arguments{
\item{x}{\code{\link[=problem]{problem()}} object.}

\item{solution}{\code{numeric}, \code{matrix}, \code{data.frame},
\code{\link[terra:rast]{terra::rast()}}, or \code{\link[sf:sf]{sf::sf()}} object.
The argument should be in the same format as the planning unit cost
data in the argument to \code{x}.
See the Solution format section for more information.}
}
\value{
A \code{\link[tibble:tibble]{tibble::tibble()}} object containing the solution cost.
It contains the following columns:

\describe{

\item{summary}{\code{character} description of the summary statistic.
The statistic associated with the \code{"overall"} value
in this column is calculated using the entire solution
(including all management zones if there are multiple zones).
If multiple management zones are present, then summary statistics
are also provided for each zone separately
(indicated using zone names).}

\item{cost}{\code{numeric} cost value.
Greater values correspond to solutions that are more costly
to implement.
Thus conservation planning exercises typically prefer solutions
with smaller values, because they are cheaper to implement
(assuming all other relevant factors, such as feature representation,
are equal).}

}
}
\description{
Calculate the total cost of a solution to a conservation planning
problem.
For example, if the planning unit cost data describe land acquisition costs
(USD), then the total cost would be net cost (USD) needed to acquire
all planning units selected within the solution.
}
\details{
This metric is equivalent to the \code{Cost} metric reported by the
\href{https://marxansolutions.org}{\emph{Marxan} software} (Ball \emph{et al.} 2009).
Specifically, the cost of a solution is defined as the sum of the cost
values, supplied when creating a \code{\link[=problem]{problem()}} object
(e.g., using the \code{cost_column} argument),
weighted by the status of each planning unit in the solution.
}
\section{Solution format}{

Broadly speaking, the argument to \code{solution} must be in the same format as
the planning unit data in the argument to \code{x}.
Further details on the correct format are listed separately
for each of the different planning unit data formats:

\describe{

\item{\code{x} has \code{numeric} planning units}{The argument to \code{solution} must be a
\code{numeric} vector with each element corresponding to a different planning
unit. It should have the same number of planning units as those
in the argument to \code{x}. Additionally, any planning units missing
cost (\code{NA}) values should also have missing (\code{NA}) values in the
argument to \code{solution}.
}

\item{\code{x} has \code{matrix} planning units}{The argument to \code{solution} must be a
\code{matrix} vector with each row corresponding to a different planning
unit, and each column correspond to a different management zone.
It should have the same number of planning units and zones
as those in the argument to \code{x}. Additionally, any planning units
missing cost (\code{NA}) values for a particular zone should also have a
missing (\code{NA}) values in the argument to \code{solution}.
}

\item{\code{x} has \code{\link[terra:rast]{terra::rast()}} planning units}{The argument to \code{solution}
be a \code{\link[terra:rast]{terra::rast()}} object where different grid cells (pixels) correspond
to different planning units and layers correspond to
a different management zones. It should have the same dimensionality
(rows, columns, layers), resolution, extent, and coordinate reference
system as the planning units in the argument to \code{x}. Additionally,
any planning units missing cost (\code{NA}) values for a particular zone
should also have missing (\code{NA})  values in the argument to \code{solution}.
}

\item{\code{x} has \code{data.frame} planning units}{The argument to \code{solution} must
be a \code{data.frame} with each column corresponding to a different zone,
each row corresponding to a different planning unit, and cell values
corresponding to the solution value. This means that if a \code{data.frame}
object containing the solution also contains additional columns, then
these columns will need to be subsetted prior to using this function
(see below for example with \code{\link[sf:sf]{sf::sf()}} data).
Additionally, any planning units missing cost
(\code{NA}) values for a particular zone should also have missing (\code{NA})
values in the argument to \code{solution}.
}

\item{\code{x} has \code{\link[sf:sf]{sf::sf()}} planning units}{The argument to \code{solution} must be
a \code{\link[sf:sf]{sf::sf()}} object with each column corresponding to a different
zone, each row corresponding to a different planning unit, and cell values
corresponding to the solution value. This means that if the
\code{\link[sf:sf]{sf::sf()}} object containing the solution also contains additional
columns, then these columns will need to be subsetted prior to using this
function (see below for example).
Additionally, the argument to \code{solution} must also have the same
coordinate reference system as the planning unit data.
Furthermore, any planning units missing cost
(\code{NA}) values for a particular zone should also have missing (\code{NA})
values in the argument to \code{solution}.
}
}
}

\examples{
\dontrun{
# set seed for reproducibility
set.seed(500)

# load data
sim_pu_raster <- get_sim_pu_raster()
sim_pu_polygons <- get_sim_pu_polygons()
sim_features <- get_sim_features()
sim_zones_pu_polygons <- get_sim_zones_pu_polygons()
sim_zones_features <- get_sim_zones_features()

# build minimal conservation problem with raster data
p1 <-
  problem(sim_pu_raster, sim_features) \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(0.1) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# solve the problem
s1 <- solve(p1)

# print solution
print(s1)

# plot solution
plot(s1, main = "solution", axes = FALSE)

# calculate cost of the solution
r1 <- eval_cost_summary(p1, s1)
print(r1)

# build minimal conservation problem with polygon data
p2 <-
  problem(sim_pu_polygons, sim_features, cost_column = "cost") \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(0.1) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# solve the problem
s2 <- solve(p2)

# plot solution
plot(s2[, "solution_1"])

# print solution
print(s2)

# calculate cost of the solution
r2 <- eval_cost_summary(p2, s2[, "solution_1"])
print(r2)

# manually calculate cost of the solution
r2_manual <- sum(s2$solution_1 * sim_pu_polygons$cost, na.rm = TRUE)
print(r2_manual)

# build multi-zone conservation problem with polygon data
p3 <-
  problem(
    sim_zones_pu_polygons, sim_zones_features,
    cost_column = c("cost_1", "cost_2", "cost_3")
  ) \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(matrix(runif(15, 0.1, 0.2), nrow = 5, ncol = 3)) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# solve the problem
s3 <- solve(p3)

# print solution
print(s3)

# create new column representing the zone id that each planning unit
# was allocated to in the solution
s3$solution <- category_vector(
  s3[, c("solution_1_zone_1", "solution_1_zone_2", "solution_1_zone_3")]
)
s3$solution <- factor(s3$solution)

# plot solution
plot(s3[, "solution"])

# calculate cost of the solution
r3 <- eval_cost_summary(
  p3, s3[, c("solution_1_zone_1", "solution_1_zone_2", "solution_1_zone_3")]
)
print(r3)
}
}
\references{
Ball IR, Possingham HP, and Watts M (2009) \emph{Marxan and relatives:
Software for spatial conservation prioritisation} in Spatial conservation
prioritisation: Quantitative methods and computational tools. Eds Moilanen
A, Wilson KA, and Possingham HP. Oxford University Press, Oxford, UK.
}
\seealso{
See \link{summaries} for an overview of all functions for summarizing solutions.

Other summaries: 
\code{\link{eval_asym_connectivity_summary}()},
\code{\link{eval_boundary_summary}()},
\code{\link{eval_connectivity_summary}()},
\code{\link{eval_feature_representation_summary}()},
\code{\link{eval_n_summary}()},
\code{\link{eval_target_coverage_summary}()}
}
\concept{summaries}
