% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bugs-funs.R
\name{fit_trend}
\alias{fit_trend}
\title{Fit a Bayesian model to estimate population size trend}
\usage{
fit_trend(
  data,
  model_opts = list(100, TRUE),
  mcmc_opts = list(ni = 50000, nt = 3, nb = 10000, nc = 2),
  path = "."
)
}
\arguments{
\item{data}{a named \code{list}. The output of \code{\link[=format_data]{format_data()}} or
\code{\link[=filter_series]{filter_series()}}.}

\item{model_opts}{a \code{list} of two vectors. The model smoothing factor
(\code{numeric}) and a \code{logical} indicating if the parameter \strong{r} must be
limited by \code{rmax} (\code{TRUE}) or not (\code{FALSE}). If this second parameter is
\code{TRUE}, the argument \code{rmax} cannot be \code{NULL} unless species are listed in
\code{popbayes} (in \code{species_info}).}

\item{mcmc_opts}{a \code{list} containing the number of iterations (\code{ni}), the
thinning factor (\code{nt}), the length of burn in (\code{nb}), i.e. the number of
iterations to discard at the beginning, and the number of chains (\code{nc}).}

\item{path}{a \code{character} string. The directory to save BUGS outputs (the
same as in \code{\link[=format_data]{format_data()}}).}
}
\value{
An n-element \code{list} (where \code{n} is the number of count series). Each
element of the list is a BUGS output as provided by JAGS (also written
in the folder \code{path}).
}
\description{
This function applies a Bayesian model to count series in order to infer
the population trend over time. This function only works on the output of
\code{\link[=format_data]{format_data()}} or \code{\link[=filter_series]{filter_series()}}.

\strong{Important:} This function uses \code{\link[R2jags:jags]{R2jags::jags()}} and the
freeware \strong{JAGS} (\url{https://mcmc-jags.sourceforge.io/}) must be
installed.

There are two types of options: model options (argument \code{model_opts}) and
MCMC options (argument \code{mcmc_opts}).

\strong{A. Model options}
\itemize{
\item a smoothing factor: the precision (the inverse of variance) of a normal
distribution centered on the current relative rate of increase r from which
the next candidate relative rate of increase (see below) is drawn.
The highest this number, the tighter the link between successive relative
rates of increase. The default \code{100} corresponds to a moderate link.
\item a logical indicating whether the population growth must remain limited by
the species demographic potential (provided by the argument \code{rmax} in
\code{\link[=format_data]{format_data()}}).
}

The relative rate of increase is the change in log population size between
two dates. The quantity actually being modeled is the relative rate of
increase per unit of time (usually one date). This quantity reflects more
directly the prevailing conditions than the population size itself, which is
the reason why it has been chosen.

When the second model option is set to \code{TRUE}, the candidate rate of
increase is compared to the maximum relative rate of increase
(obtained when using \code{\link[=format_data]{format_data()}}) and replaced by \code{rmax} if greater.

\strong{B. MCMC options}

Classical Markov chain Monte Carlo (MCMC) settings (see argument \code{mcmc_opts}
below).
}
\examples{
## Load Garamba raw dataset ----
file_path <- system.file("extdata", "garamba_survey.csv", 
                         package = "popbayes")
                         
garamba <- read.csv(file = file_path)

## Create temporary folder ----
temp_path <- tempdir()

## Format dataset ----
garamba_formatted <- popbayes::format_data(
  data              = garamba, 
  path              = temp_path,
  field_method      = "field_method",
  pref_field_method = "pref_field_method",
  conversion_A2G    = "conversion_A2G",
  rmax              = "rmax")

## Get data for Alcelaphus buselaphus at Garamba only ----
a_buselaphus <- popbayes::filter_series(garamba_formatted, 
                                        location = "Garamba",
                                        species  = "Alcelaphus buselaphus")
                                        
\donttest{
## Fit population trend (requires JAGS) ----
a_buselaphus_mod <- popbayes::fit_trend(a_buselaphus, path = temp_path)

## Check for convergence ----
popbayes::diagnostic(a_buselaphus_mod, threshold = 1.1)

## Plot estimated population trend ----
popbayes::plot_trend(series = "garamba__alcelaphus_buselaphus", 
                     path   = temp_path)

## Plot MCMC traceplot ----
R2jags::traceplot(a_buselaphus_mod[[1]], ask = TRUE)
}
}
