% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MDP.R
\name{MDP}
\alias{MDP}
\alias{is_solved_MDP}
\title{Define an MDP Problem}
\usage{
MDP(
  states,
  actions,
  transition_prob,
  reward,
  discount = 0.9,
  horizon = Inf,
  start = "uniform",
  info = NULL,
  name = NA
)

is_solved_MDP(x, stop = FALSE)
}
\arguments{
\item{states}{a character vector specifying the names of the states.}

\item{actions}{a character vector specifying the names of the available
actions.}

\item{transition_prob}{Specifies the transition probabilities between
states.}

\item{reward}{Specifies the rewards dependent on action, states and
observations.}

\item{discount}{numeric; discount rate between 0 and 1.}

\item{horizon}{numeric; Number of epochs. \code{Inf} specifies an infinite
horizon.}

\item{start}{Specifies in which state the MDP starts.}

\item{info}{A list with additional information.}

\item{name}{a string to identify the MDP problem.}

\item{x}{a \code{MDP} object.}

\item{stop}{logical; stop with an error.}
}
\value{
The function returns an object of class MDP which is list with
the model specification. \code{\link[=solve_MDP]{solve_MDP()}} reads the object and adds a list element called
\code{'solution'}.
}
\description{
Defines all the elements of a finite state-space MDP problem.
}
\details{
Markov decision processes (MDPs) are discrete-time stochastic control
process with completely observable states. We implement here
MDPs with a finite state space. similar to \link{POMDP}
models, but without the observation model. The \code{'observations'} column in
the the reward specification is always missing.

\code{\link[=make_partially_observable]{make_partially_observable()}} reformulates an MDP as a POMDP by adding an observation
model with one observation per state
that reveals the current state. This is achieved by adding identity
observation probability matrices.

More details on specifying the model components can be found in the documentation
for \link{POMDP}.
}
\examples{
# Michael's Sleepy Tiger Problem is like the POMDP Tiger problem, but
# has completely observable states because the tiger is sleeping in front
# of the door. This makes the problem an MDP.

STiger <- MDP(
  name = "Michael's Sleepy Tiger Problem",
  discount = .9,

  states = c("tiger-left" , "tiger-right"),
  actions = c("open-left", "open-right", "do-nothing"),
  start = "uniform",

  # opening a door resets the problem
  transition_prob = list(
    "open-left" =  "uniform",
    "open-right" = "uniform",
    "do-nothing" = "identity"),

  # the reward helper R_() expects: action, start.state, end.state, observation, value
  reward = rbind(
    R_("open-left",  "tiger-left",  v = -100),
    R_("open-left",  "tiger-right", v =   10),
    R_("open-right", "tiger-left",  v =   10),
    R_("open-right", "tiger-right", v = -100),
    R_("do-nothing",                v =    0)
  )
)

STiger

sol <- solve_MDP(STiger)
sol

policy(sol)
plot_value_function(sol)

# convert the MDP into a POMDP and solve
STiger_POMDP <- make_partially_observable(STiger)
sol2 <- solve_POMDP(STiger_POMDP)
sol2

policy(sol2)
plot_value_function(sol2, ylim = c(80, 120))
}
\seealso{
Other MDP: 
\code{\link{MDP2POMDP}},
\code{\link{MDP_policy_functions}},
\code{\link{accessors}},
\code{\link{actions}()},
\code{\link{add_policy}()},
\code{\link{gridworld}},
\code{\link{reachable_and_absorbing}},
\code{\link{regret}()},
\code{\link{simulate_MDP}()},
\code{\link{solve_MDP}()},
\code{\link{transition_graph}()},
\code{\link{value_function}()}

Other MDP_examples: 
\code{\link{Cliff_walking}},
\code{\link{DynaMaze}},
\code{\link{Maze}},
\code{\link{Windy_gridworld}}
}
\author{
Michael Hahsler
}
\concept{MDP}
\concept{MDP_examples}
