% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trelliPlots.R
\name{trelli_missingness_bar}
\alias{trelli_missingness_bar}
\title{Bar chart trelliscope building function for missing data}
\usage{
trelli_missingness_bar(
  trelliData,
  cognostics = c("total count", "observed count", "observed proportion"),
  proportion = TRUE,
  ggplot_params = NULL,
  interactive = FALSE,
  path = .getDownloadsFolder(),
  name = "Trelliscope",
  test_mode = FALSE,
  test_example = 1,
  single_plot = FALSE,
  ...
)
}
\arguments{
\item{trelliData}{A trelliscope data object made by as.trelliData.edata or
as.trelliData. Required.}

\item{cognostics}{A vector of cognostic options for each plot. Defaults are "total count",
"observed count", and "observed proportion". If grouping
data is included, all cognostics will be reported per group. If the 
trelliData is paneled by a biomolecule, the counts and proportion we be 
samples. If paneled by a sample or biomolecule class, the counts and proportions
will be biomolecules. If statRes data is included, "g-test p-value" may 
be included.}

\item{proportion}{A logical to determine whether plots should display counts
or proportions. Default is TRUE.}

\item{ggplot_params}{An optional vector of strings of ggplot parameters to
the backend ggplot function. For example, c("ylab('')", "xlab('')").
Default is NULL.}

\item{interactive}{A logical argument indicating whether the plots should be
interactive or not. Interactive plots are ggplots piped to ggplotly (for
now). Default is FALSE.}

\item{path}{The base directory of the trelliscope application. Default is
Downloads.}

\item{name}{The name of the display. Default is Trelliscope.}

\item{test_mode}{A logical to return a smaller trelliscope to confirm plot
and design. Default is FALSE.}

\item{test_example}{A vector of plot indices to return for test_mode. Default
is 1.}

\item{single_plot}{A TRUE/FALSE to indicate whether 1 plot (not a
trelliscope) should be returned. Default is FALSE.}

\item{...}{Additional arguments to be passed on to the trelli builder}
}
\value{
No return value, builds a trelliscope display of missingness bar charts that is stored in `path`
}
\description{
Specify a plot design and cognostics for the missing barchart
   trelliscope. Missingness is displayed per panel_by variable. Main_effects
   data is used to split samples when applicable.
}
\examples{
\dontshow{if (requireNamespace("pmartRdata", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{
if (interactive()) {
library(pmartRdata)

trelliData1 <- as.trelliData.edata(e_data = pep_edata,
                                   edata_cname = "Peptide",
                                   omics_type = "pepData")
# Transform the data
omicsData <- edata_transform(omicsData = pep_object, data_scale = "log2")

# Group the data by condition
omicsData <- group_designation(omicsData = omicsData, main_effects = c("Phenotype"))

# Apply the IMD ANOVA filter
imdanova_Filt <- imdanova_filter(omicsData = omicsData)
omicsData <- applyFilt(filter_object = imdanova_Filt, omicsData = omicsData,
                       min_nonmiss_anova = 2)

# Normalize my pepData
omicsData <- normalize_global(omicsData, "subset_fn" = "all", "norm_fn" = "median",
                             "apply_norm" = TRUE, "backtransform" = TRUE)

# Implement the IMD ANOVA method and compute all pairwise comparisons 
# (i.e. leave the `comparisons` argument NULL)
statRes <- imd_anova(omicsData = omicsData, test_method = 'combined')

# Generate the trelliData object
trelliData2 <- as.trelliData(omicsData = omicsData)
trelliData3 <- as.trelliData(statRes = statRes)
trelliData4 <- as.trelliData(omicsData = omicsData, statRes = statRes)

# Build the missingness bar plot with an edata file. Generate trelliData in as.trelliData.edata
trelli_panel_by(trelliData = trelliData1, panel = "Peptide") \%>\% 
  trelli_missingness_bar(test_mode = TRUE, test_example = 1:10, path = tempdir())
trelli_panel_by(trelliData = trelliData1, panel = "Sample") \%>\% 
  trelli_missingness_bar(test_mode = TRUE, test_example = 1:10, 
   cognostics = "observed proportion", path = tempdir())

# Build the missingness bar plot with an omicsData object. Generate trelliData in as.trelliData
trelli_panel_by(trelliData = trelliData2, panel = "Peptide") \%>\% 
  trelli_missingness_bar(test_mode = TRUE, test_example = 1:10, path = tempdir())

# Build the missingness bar plot with a statRes object. Generate trelliData in as.trelliData
trelli_panel_by(trelliData = trelliData3, panel = "Peptide") \%>\%
  trelli_missingness_bar(test_mode = TRUE, test_example = 1:10, path = tempdir(),
                         cognostics = c("observed proportion", "g-test p-value"))

# Build the missingness bar plot with an omicsData and statRes object. 
# Generate trelliData in as.trelliData.
trelli_panel_by(trelliData = trelliData4, panel = "Peptide") \%>\%
  trelli_missingness_bar(test_mode = TRUE, test_example = 1:10, path = tempdir()) 

# Or making the plot interactive 
trelli_panel_by(trelliData = trelliData2, panel = "Peptide") \%>\% 
   trelli_missingness_bar(
     test_mode = TRUE, test_example = 1:5, interactive = TRUE, path = tempdir())
   
# Or visualize only count data 
trelli_panel_by(trelliData = trelliData2, panel = "Peptide") \%>\% 
   trelli_missingness_bar(
     test_mode = TRUE, test_example = 1:5, 
     cognostics = "observed count", proportion = FALSE,
     path = tempdir()
   )

\dontshow{closeAllConnections()}
}
}
\dontshow{\}) # examplesIf}
}
\author{
David Degnan, Lisa Bramer
}
