#' @name sprinkle
#' @export sprinkle
#' 
#' @title Define Customizations to a Table
#' @description Customizations to a \code{dust} table are added by "sprinkling"
#'   with a little extra pixie dust.  Sprinkles are a collection of attributes
#'   to be applied over a subset of table cells.  They may be added to any 
#'   part of the table, or to the table as a whole.
# Parameters --------------------------------------------------------  
#' @param x A dust object
#' @param rows A numeric vector specifying the rows of the table to sprinkle.
#'   See details for more about sprinkling.
#' @param cols A numeric (or character) vector specifying the columns (or 
#'   column names) to sprinkle.  See details for more about sprinkling.
#' @param part A character string denoting which part of the table to modify.
#' @param fixed \code{logical(1)} indicating if the values in \code{rows} 
#'   and \code{cols} should be read as fixed coordinate pairs.  By default, 
#'   sprinkles are applied at the intersection of \code{rows} and \code{cols}, 
#'   meaning that the arguments do not have to share the same length.  
#'   When \code{fixed = TRUE}, they must share the same length.
#' @param recycle A \code{character} one that determines how sprinkles are 
#'   managed when the sprinkle input doesn't match the length of the region
#'   to be sprinkled.  By default, recycling is turned off.  Recycling 
#'   may be performed across rows first (left to right, top to bottom), 
#'   or down columns first (top to bottom, left to right).
#' @param ... named arguments, each of length 1, defining the customizations
#'   for the given cells.  See "Sprinkles" for a listing of these arguments.
#'   
# Details -----------------------------------------------------------
#' @details Sprinkling is done over the intersection of rows and columns 
#'   (unless \code{fixed = TRUE}.  If
#'   rows but no columns are specified, sprinkling is performed over all columns
#'   of the given given rows. The reverse is true for when columns but no rows
#'   are specified.  If neither columns nor rows are specified, the attribute 
#'   is applied over all of the cells in the table part denoted in \code{part}.
#'
#'   If at least one of \code{border}, \code{border_thickness}, \code{border_units},
#'   \code{border_style} or \code{border_color} is specified, the remaining
#'   unspecified attributes assume their default values.
#'   
#'   Other sprinkle pairings are \code{height} and \code{height_units}; 
#'   \code{width} and \code{width_units}; \code{font_size} and \code{font_size_units};
#'   \code{bg_pattern} and \code{bg_pattern_by}
#'   
#'   The sprinkles \code{bg} and \code{bg_pattern} may not be used together.
#'   
#'   A more detailed demonstration of the use of sprinkles is available in 
#'   \code{vignette("pixiedust", package = "pixiedust")}
#'   
#'   Using \code{sprinkle_table}, sprinkles may be applied to the columns of multiple tables. Table
#'   parts are required to have the same number of columns, but not necessarily the same number 
#'   of rows, which is why the \code{rows} argument is not available for the \code{sprinkle_table}.
#'   In contrast to \code{sprinkle}, the \code{part} argument in \code{sprinkle_table} will 
#'   accept multiple parts.  If any of the named parts is \code{"table"}, the sprinkle will be 
#'   applied to the columns of all of the parts.
#'   
# Sprinkles ---------------------------------------------------------  
#' @section Sprinkles:
#' The following table describes the valid sprinkles that may be defined in the 
#' \code{...} dots argument.  All sprinkles may be defined for any output type, but 
#' only sprinkles recognized by that output type will be applied when printed.  
#' A more readable format of this information is available in  
#' \code{vignette("sprinkles", package = "pixiedust")}.
#' 
#' \tabular{lll}{
#' bg  \tab           \tab  \cr
#'     \tab action    \tab Modifies the background color of a cell. \cr
#'     \tab default   \tab  \cr
#'     \tab accepts   \tab dvips color names; rgb(R,G,B); rgba(R,G,B,A); \cr
#'     \tab           \tab  #RRGGBB; #RRGGBBAA. See the "Colors" section \cr
#'     \tab           \tab  for further details. \cr
#'     \tab console   \tab Not recognized \cr
#'     \tab markdown  \tab Not recognized \cr
#'     \tab html      \tab Accepts any of the listed formats; \cr
#'     \tab           \tab recognizes transparency \cr
#'     \tab latex     \tab Accepts any of the listed formats, \cr
#'     \tab           \tab but ignores transparency \cr
#' bg_pattern \tab    \tab  \cr
#'  \tab action       \tab Generates a pattern of background colors.  \cr
#'  \tab              \tab Can be used to make striping \cr
#'  \tab              \tab by rows or by columns. \cr
#'  \tab default      \tab c("#FFFFFF", "#DDDDDD") \cr
#'  \tab accepts      \tab A vector of color names: \cr
#'  \tab              \tab dvips color names; rgb(R,G,B); rgba(R,G,B,A); \cr 
#'  \tab              \tab #RRGGBB; #RRGGBBAA \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Accepts any of the listed formats; \cr
#'  \tab              \tab recognizes transparency \cr
#'  \tab latex        \tab Accepts any of the listed formats, \cr
#'  \tab              \tab but ignores transparency \cr
#' bg_pattern_by  \tab  \tab  \cr
#'  \tab action       \tab Determines if a `bg_pattern` is patterned \cr 
#'  \tab              \tab by row or by columns. \cr
#'  \tab default      \tab "rows" \cr
#'  \tab accepts      \tab "rows", "columns", "cols" \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' bold \tab  \tab  \cr
#'  \tab action       \tab Renders text within a cell in bold. \cr
#'  \tab default      \tab FALSE \cr
#'  \tab accepts      \tab logical(1) \cr
#'  \tab console      \tab Recognized; rendered as double asterisks on either\cr
#'  \tab              \tab side of the text \cr
#'  \tab markdown     \tab Recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' border_collapse \tab  \tab  \cr
#'  \tab action       \tab Sets the `border-collapse` property in an \cr
#'  \tab              \tab HTML table.  The property sets whether the \cr
#'  \tab              \tab table borders are collapsed into a  \cr
#'  \tab              \tab single border or detached as in standard HTML. \cr
#'  \tab default      \tab TRUE \cr
#'  \tab accepts      \tab logical(1) \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Not recognized \cr
#' border \tab  \tab  \cr
#'  \tab action       \tab Sets a border on the specified side of a cell. \cr
#'  \tab default      \tab  \cr
#'  \tab accepts      \tab Any combination of "all", "bottom", "left", "top",\cr
#'  \tab              \tab "right".  Using  "all" results in all borders \cr
#'  \tab              \tab being drawn, regardless of what other values are \cr
#'  \tab              \tab passed with it. \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' border_color \tab  \tab  \cr
#'  \tab action       \tab Sets the color of the borders specified for a cell. \cr
#'  \tab default      \tab "Black" \cr
#'  \tab accepts      \tab character(1) \cr
#'  \tab              \tab dvips color names; rgb(R,G,B); rgba(R,G,B,A); \cr
#'  \tab              \tab #RRGGBB; #RRGGBBAA. See the "Colors" section \cr
#'     \tab           \tab  for further details. \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' border_style \tab  \tab  \cr
#'  \tab action       \tab Sets the border style for a specified cell \cr
#'  \tab default      \tab "solid" \cr
#'  \tab accepts      \tab character(1) \cr
#'  \tab              \tab "solid", "dashed", "dotted", "double", "groove", \cr
#'  \tab              \tab "ridge", "inset", "outset", "hidden", "none" \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Accepts any of the values listed. \cr
#'  \tab latex; hhline = FALSE \tab accepts "solid", "dashed", "dotted",  \cr
#'  \tab              \tab  "hidden", "none" \cr
#'  \tab              \tab "dotted" is silently changed to "dashed" \cr
#'  \tab              \tab "hidden" and "none" are equivalent. \cr
#'  \tab latex; hhline = TRUE \tab accepts "solid", "double", "hidden", "none" \cr
#'  \tab              \tab "hidden" and "none" are equivalent. \cr
#' border_thickness \tab  \tab  \cr
#'  \tab action       \tab Sets the thickness of the specified border \cr
#'  \tab default      \tab 1 \cr
#'  \tab accepts      \tab numeric(1) \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' border_units \tab  \tab  \cr
#'  \tab action       \tab Sets the unit of measure for the specified border \cr
#'  \tab              \tab thickness \cr
#'  \tab default      \tab "pt" \cr
#'  \tab accepts      \tab "pt", "px" \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Silently changes "px" to "pt" \cr
#' caption \tab  \tab  \cr
#'  \tab action       \tab Adds or alters the `caption` property \cr
#'  \tab default      \tab  \cr
#'  \tab accepts      \tab character(1) \cr
#'  \tab console      \tab Recognized \cr
#'  \tab markdown     \tab Recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' discrete \tab \tab \cr
#'  \tab action       \tab Adds distinct background colors based on \cr
#'  \tab              \tab discrete values in the selected region. \cr
#'  \tab              \tab May not be used concurrently with \code{bg}. \cr
#'  \tab              \tab \code{"font"} is an alias for \code{"font_color"} \cr
#'  \tab              \tab and \code{"border"} is an alias for \cr
#'  \tab              \tab all borders. \cr
#'  \tab default      \tab "bg" \cr
#'  \tab accepts      \tab "bg", "font", "font_color", "border", \cr
#'  \tab              \tab "left_border", "top_border", "right_border", \cr
#'  \tab              \tab "bottom_border" \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' discrete_color \tab \tab \cr
#'  \tab action       \tab Sets the color palette from which \code{discrete} \cr
#'  \tab              \tab selects background colors. If \code{NULL} \cr
#'  \tab              \tab colors are automatically selected using \cr
#'  \tab              \tab the \code{scales} package. \cr
#'  \tab default      \tab \code{getOption("pixie_discrete_pal", NULL)} \cr
#'  \tab accepts      \tab character \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' float \tab  \tab  \cr
#'  \tab action       \tab Sets the `float` property \cr
#'  \tab default      \tab TRUE \cr
#'  \tab accepts      \tab logical(1) \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Not recognized \cr
#'  \tab latex        \tab Recognized \cr
#' fn \tab  \tab  \cr
#'  \tab action       \tab Applies a function to the value of a cell. \cr
#'  \tab              \tab The function should be an \cr
#'  \tab              \tab expression that acts on the variable `value`.  \cr 
#'  \tab              \tab For example, \code{quote(format(value, nsmall = 3))} \cr
#'  \tab default      \tab  \cr
#'  \tab accepts      \tab call \cr
#'  \tab console      \tab Recognized \cr
#'  \tab markdown     \tab Recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' font_color \tab  \tab  \cr
#'  \tab action       \tab Sets the color of the cell text \cr
#'  \tab default      \tab Black \cr
#'  \tab accepts      \tab dvips color names; rgb(R,G,B); rgba(R,G,B,A); \cr
#'  \tab              \tab #RRGGBB; #RRGGBBAA. See the "Colors" section \cr
#'     \tab           \tab  for further details. \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized; transparency recognized \cr
#'  \tab latex        \tab Recognized; transparency ignored \cr
#' font_family \tab  \tab  \cr
#'  \tab action       \tab Sets the font for the text \cr
#'  \tab default      \tab Times New Roman \cr
#'  \tab accepts      \tab character(1) \cr
#'  \tab              \tab http://www.w3schools.com/cssref/css_websafe_fonts.asp \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Not recognized \cr
#' font_size \tab  \tab  \cr
#'  \tab action       \tab Sets the size of the font in the cell \cr
#'  \tab default      \tab  \cr
#'  \tab accepts      \tab numeric(1) \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' font_size_units \tab  \tab  \cr
#'  \tab action       \tab Determines the units in which `font_size` \cr
#'  \tab              \tab is measured \cr
#'  \tab default      \tab "px" \cr
#'  \tab accepts      \tab "px", "pt", "\%", "em" \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Only recognizes "pt" and "em".  \cr
#'  \tab              \tab All others are coerced to "pt" \cr
#' gradient \tab  \tab  \cr
#'  \tab action       \tab Adds distinct background colors based on \cr
#'  \tab              \tab progressively increasing values in the \cr
#'  \tab              \tab selected region. May not be used concurrently \cr 
#'  \tab              \tab with \code{bg}. \cr
#'  \tab              \tab \code{"font"} is an alias for \code{"font_color"} \cr
#'  \tab              \tab and \code{"border"} is an alias for \cr
#'  \tab              \tab all borders. \cr
#'  \tab default      \tab "bg" \cr
#'  \tab accepts      \tab "bg", "font", "font_color", "border", \cr
#'  \tab              \tab "left_border", "top_border", "right_border", \cr
#'  \tab              \tab "bottom_border" \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' gradient_colors \tab \tab \cr
#'  \tab action       \tab Provides the colors between which to \cr
#'  \tab              \tab shade gradients. \cr
#'  \tab default      \tab \code{getOptions("pixie_gradient_pal", NULL)} \cr
#'  \tab accepts      \tab character \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' gradient_cut \tab  \tab  \cr
#'  \tab action       \tab Determines the breaks points for the \cr
#'  \tab              \tab gradient shading. When \code{NULL}  \cr
#'  \tab              \tab equally spaced quantiles are used, the \cr
#'  \tab              \tab number of which are determined by \cr
#'  \tab              \tab \code{gradient_n}. \cr
#'  \tab default      \tab NULL \cr
#'  \tab accepts      \tab numeric \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' gradient_n \tab  \tab  \cr
#'  \tab action       \tab Determines the number of shades to use \cr
#'  \tab              \tab between the colors in \code{gradient_colors}.\cr
#'  \tab default      \tab 10 \cr
#'  \tab accepts      \tab numeric \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' gradient_na \tab  \tab  \cr
#'  \tab action       \tab Sets the color of NA values when gradients \cr
#'  \tab              \tab are shaded. \cr
#'  \tab default      \tab grey \cr
#'  \tab accepts      \tab character(1) \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' halign \tab  \tab  \cr
#'  \tab action       \tab Sets the horizontal alignment of the text in \cr
#'  \tab              \tab the cell \cr
#'  \tab default      \tab  \cr
#'  \tab accepts      \tab "left", "center", "right" \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Recognized; numeric values will auto align to the \cr
#'  \tab              \tab right if no value given. \cr
#'  \tab html         \tab Recognized.  Does not currently employ auto \cr
#'  \tab              \tab alignment of numeric values, but this may change. \cr
#'  \tab latex        \tab Recognized; numeric values will auto align to \cr
#'  \tab              \tab the right if no value given. \cr
#' height \tab  \tab  \cr
#'  \tab action       \tab Sets the height of the cell \cr
#'  \tab default      \tab  \cr
#'  \tab accepts      \tab numeric(1) \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' height_units \tab  \tab  \cr
#'  \tab action       \tab Determines the units in which `height` is measured \cr
#'  \tab default      \tab "pt" \cr
#'  \tab accepts      \tab "px", "pt", "cm", "in", "\%" \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized; "px" is coerced to "pt" \cr
#' hhline \tab  \tab  \cr
#'  \tab action       \tab Toggles the option for cell border drawing with \cr 
#'  \tab              \tab the `hhline` LaTeX package \cr
#'  \tab default      \tab FALSE \cr
#'  \tab accepts      \tab logical(1) \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Not recognized \cr
#'  \tab latex        \tab Recognized.  When `FALSE` double borders are \cr 
#'  \tab              \tab not available. \cr
#'  \tab              \tab When `TRUE`, colored and dashed borders are not \cr
#'  \tab              \tab available. This is usually the better option \cr
#'  \tab              \tab when using colored backgrounds in table cells. \cr
#' italic \tab  \tab  \cr
#'  \tab action       \tab Renders the text in the cell in italic \cr
#'  \tab default      \tab FALSE \cr
#'  \tab accepts      \tab logical(1) \cr
#'  \tab console      \tab Recognized; rendered as an underscore on either \cr
#'  \tab              \tab side of the text. \cr
#'  \tab markdown     \tab Recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' justify \tab  \tab  \cr
#'  \tab action       \tab Justifies the entire table on the page. \cr
#'  \tab default      \tab "center" \cr
#'  \tab accepts      \tab character(1) \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognizes "center", but both "left" and "right" \cr
#'  \tab              \tab are rendered as left justified.  This may change \cr
#'  \tab              \tab if a satisfactory solution is found.  \cr
#'  \tab              \tab Usually, tables are best left centered. \cr
#' longtable \tab  \tab  \cr
#'  \tab action       \tab Toggles the use of the LaTeX `longtable` style \cr
#'  \tab              \tab tables, namely allowing long tables to be broken \cr
#'  \tab              \tab into multiple sections. The table header appears \cr 
#'  \tab              \tab at the top of each section. The table interfoot \cr
#'  \tab              \tab appears at the bottom of each section, except \cr
#'  \tab              \tab for the last. \cr
#'  \tab              \tab The table foot appears at the bottom of the \cr
#'  \tab              \tab last section. \cr
#'  \tab              \tab May accept either a logical or a numerical value.  \cr
#'  \tab              \tab If numerical, each section will have the specified \cr
#'  \tab              \tab number of rows. \cr
#'  \tab default      \tab FALSE \cr
#'  \tab accepts      \tab logical(1); numeric(1) \cr
#'  \tab console      \tab Recognized; when `TRUE`, defaults to 25 rows \cr
#'  \tab              \tab per section. \cr
#'  \tab markdown     \tab Recognized; when `TRUE`, defaults to 25 rows \cr
#'  \tab              \tab per section. \cr
#'  \tab html         \tab Recognized; when `TRUE`, defaults to 25 rows \cr
#'  \tab              \tab per section. \cr
#'  \tab latex        \tab Recognized; when `TRUE`, `longtable`'s own algorithm \cr 
#'  \tab              \tab will determine the number of rows per section. \cr
#'  \tab              \tab When numeric, breaks are forced at the specified \cr
#'  \tab              \tab number of rows. \cr
#' merge \tab  \tab  \cr
#'  \tab action       \tab Merges cells in the specified range into a \cr
#'  \tab              \tab single cell. In cases where \cr
#'  \tab              \tab either `merge_rowval` or `merge_colval` is \cr
#'  \tab              \tab specified, they will only be  \cr
#'  \tab              \tab honored if `merge = TRUE`.  You must opt in to \cr
#'  \tab              \tab this action. \cr
#'  \tab default      \tab FALSE \cr
#'  \tab accepts      \tab logical(1) \cr
#'  \tab console      \tab Recognized \cr
#'  \tab markdown     \tab Recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' merge_rowval \tab  \tab  \cr
#'  \tab action       \tab Specifies the row value of the merged range to \cr
#'  \tab              \tab print in the table \cr
#'  \tab default      \tab minimum row value of the merged range \cr
#'  \tab accepts      \tab numeric(1) \cr
#'  \tab console      \tab Recognized \cr
#'  \tab markdown     \tab Recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' merge_colval \tab  \tab  \cr
#'  \tab action       \tab Specifies the column value of the merged range \cr
#'  \tab              \tab to print in the table \cr
#'  \tab default      \tab minimum col value of the merged range \cr
#'  \tab accepts      \tab numeric(1) \cr
#'  \tab console      \tab Recognized \cr
#'  \tab markdown     \tab Recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' na_string \tab  \tab  \cr
#'  \tab action       \tab Designates the character string to use in place \cr
#'  \tab              \tab of missing values \cr
#'  \tab default      \tab NA \cr
#'  \tab accepts      \tab character(1) \cr
#'  \tab console      \tab Recognized \cr
#'  \tab markdown     \tab Recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' pad \tab  \tab  \cr
#'  \tab action       \tab Designates the padding to place between cell \cr
#'  \tab              \tab text and boundaries \cr
#'  \tab              \tab Measured in pixels. \cr
#'  \tab default      \tab 0 \cr
#'  \tab accepts      \tab numeric(1) \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Not recognized \cr
#' replace \tab  \tab  \cr
#'  \tab action       \tab Replaces existing cell values with user-specified \cr
#'  \tab              \tab content. Replacement occurs moving down columns \cr
#'  \tab              \tab from left to right. \cr
#'  \tab default      \tab  \cr
#'  \tab accepts      \tab character vector of the same length as the number \cr
#'  \tab              \tab of cells being replaced. \cr
#'  \tab console      \tab Recognized \cr
#'  \tab markdown     \tab Recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' rotate_degree \tab  \tab  \cr
#'  \tab action       \tab Rotates text in cells by the designated angle \cr
#'  \tab              \tab in degrees \cr
#'  \tab default      \tab  \cr
#'  \tab accepts      \tab numeric(1) \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' round \tab  \tab  \cr
#'  \tab action       \tab Applies the `round` function to values in the \cr 
#'  \tab              \tab cell.  Skips any character values it encounters. \cr
#'  \tab default      \tab \code{getOption("digits")} \cr
#'  \tab accepts      \tab numeric(1) \cr
#'  \tab console      \tab Recognized \cr
#'  \tab markdown     \tab Recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' sanitize \tab  \tab  \cr
#'  \tab action	      \tab Sanitizes character values that may cause \cr
#'  \tab              \tab difficulties for the rendered format.  \cr
#'  \tab default	    \tab FALSE      \cr
#'  \tab accepts	    \tab logical(1) \cr
#'  \tab console	    \tab Not recognized \cr
#'  \tab markdown	    \tab Not recognized \cr
#'  \tab html         \tab Not recognized \cr
#'  \tab latex	      \tab Recognized.  Sanitization is performed using \cr
#'  \tab              \tab \code{\link[Hmisc]{latexTranslate}} \cr
#' sanitize_args	\tab \tab \cr
#'  \tab action	      \tab Passes additional arguments to \code{\link[Hmisc]{latexTranslate}} \cr
#'  \tab default	    \tab \code{list()} \cr
#'  \tab accepts	    \tab list.  See documentation for \code{\link[Hmisc]{latexTranslate}} \cr
#'  \tab              \tab for details \cr
#'  \tab console	    \tab Not recognized \cr
#'  \tab markdown	    \tab Not recognized \cr
#'  \tab html	        \tab Not recognized \cr
#'  \tab latex	      \tab Recognized \cr
#' tabcolsep \tab  \tab  \cr
#'  \tab action       \tab Modifies the LaTeX `tabcolsep` parameter of tables \cr
#'  \tab              \tab This is similar to `pad` for HTML tables, but only  \cr
#'  \tab              \tab affects the space between columns. Measured in "pt" \cr
#'  \tab default      \tab 6 \cr
#'  \tab accepts      \tab numeric(1) \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Not recognized \cr
#'  \tab latex        \tab Recognized \cr
#' valign \tab  \tab  \cr
#'  \tab action       \tab Designates the vertical alignment of a cell. \cr
#'  \tab default      \tab  \cr
#'  \tab accepts      \tab "top", "middle", "bottom" \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' width \tab  \tab  \cr
#'  \tab action       \tab Sets the width of the cell \cr
#'  \tab default      \tab  \cr
#'  \tab accepts      \tab numeric(1) \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized \cr
#' width_units \tab  \tab  \cr
#'  \tab action       \tab Determines the units in which `width` is measured \cr
#'  \tab default      \tab "pt" \cr
#'  \tab accepts      \tab "px", "pt", "cm", "in", "\%" \cr
#'  \tab console      \tab Not recognized \cr
#'  \tab markdown     \tab Not recognized \cr
#'  \tab html         \tab Recognized \cr
#'  \tab latex        \tab Recognized; "px" is coerced to "pt" \cr
#' }
#' 
# Longtable details -------------------------------------------------
#' @section Longtable:
#' The \code{longtable} feature is named for the LaTeX package used to break very large 
#' tables into multiple pages.  
#' 
#' When using the \code{longtable=TRUE} option, the default number of rows per table is 25 for 
#' console, HTML, and markdown output.  For LaTeX output, the number of rows is determined by 
#' the LaTeX \code{longtable} package's algorithm. The number of rows per table only considers 
#' the content in the body of the table.  Consideration for the number of rows in the head and 
#' foot are the responsibility of the user.
#'   
#' Whenever a table is broken into multiple parts, each part retains the table head.  If any 
#' \code{interfoot} is provided, it is appended to the bottom of each section, with the 
#' exception of the last section.  The last section has the \code{foot} appended.
#'
# Colors ------------------------------------------------------------
#' @section Colors:
#' Colors may be declared as any of the color names in \code{colors()}, 
#' as rgb character strings such as \code{"rgb(rrr,ggg,bbb)"} or as 
#' hexadecimal character strings such as \code{"#rrggbb"}.  
#' 
#' Transparency is also recognized by HTML output, and may be indicated 
#' in the rgba format \code{"rgba(rrr,ggg,bbb,aa)"}, where \code{aa} is a 
#' number between 0 and 1, inclusive.  Alternative, transparency may be 
#' given as \code{"#rrggbbAA"}, where \code{AA} is a hexadecimal 
#' representation of transparency with "00" being completely transparent 
#' and "FF" being completely opaque.
#' 
#' LaTeX output does not recognize transparency and will quietly drop the 
#' transparency parameter.
#'
#' All colors are internally translated into rgb format and are case insensitive.
#' 
# LaTeX Packages ----------------------------------------------------
#' @section Required LaTeX Packages:
#' 
#' If you will be using the LaTeX output, some sprinkles will require you 
#' to include additional LaTeX packages in your document preamble.  In 
#' \code{.Rnw} files, additional packages can be included with the 
#' \code{\\usepackage\{[package]\}} syntax.  In markdown, additional packages
#' are included using \code{header-includes:} in the YAML front matter with 
#' a line of the format \code{\\usepackage\{[package]\}} for each package to 
#' be used.  Sprinkles that require additional packages, and the LaTeX packages
#' required, are listed below:
#' 
#' \tabular{ll}{
#'   Sprinkle \tab LaTeX Package(s) \cr
#'   \code{font_color} \tab \code{\\usepackage[dvipsnames]\{xcolor\}} \cr
#'   \code{bg, bg_pattern} \tab \code{\\usepackage[dvipsnames,table]\{xcolor\}} \cr
#'   \code{border_style} \tab \code{\\usepackage\{arydshln\}} \cr
#'       \tab  \code{\\usepackage\{amssymb\}} \cr
#'       \tab  \code{\\usepackage\{hhline\}} \cr
#'       (with vertical dashed lines) \tab \\usepackage\{graphicx\} \cr
#'       \tab \code{\\makeatletter} \cr
#'       \tab \code{\\newcommand*\\vdashline\{\\rotatebox[origin=c]\{90\}\{\$\\dabar@@\\dabar@@\\dabar@@\$\}\}} \cr
#'       \tab \code{\\makeatother} \cr
#'   \code{longtable} \tab \code{\\usepackage\{longtable\}} \cr
#'       \tab (Must be loaded before \code{arydshln}) \cr
#'   \code{merge} \tab \code{\\usepackage\{multirow\}} \cr
#'   \code{captions} for non floats \tab \code{\\usepackage\{caption\}} 
#' }
#' 
#' Note that \code{hhline} is used to make horizontal lines when 
#' \code{options(pixiedust_latex_hhline = TRUE)} (the package default is \code{FALSE}), 
#' otherwise the \code{cline} command is used.  
#' 
#' Use of \code{cline} permits colored borders and dashed borders, but 
#' borders around cells with background colors are sometimes (often) lost.
#' 
#' Use of \code{hhline} preserves borders around cells with background colors 
#' and permits double borders, but colored and dashed borders are not available.
#' 
#' In order to ensure all features are available, the recommended code block (accounting for 
#' the proper order to load packages) is:
#' 
#' \code{header-includes:} \cr
#' \code{ - \\usepackage\{amssymb\}} \cr
#' \code{ - \\usepackage\{arydshln\}} \cr
#' \code{ - \\usepackage\{caption\}}  \cr
#' \code{ - \\usepackage\{graphicx\}} \cr
#' \code{ - \\usepackage\{hhline\}} \cr
#' \code{ - \\usepackage\{longtable\}} \cr
#' \code{ - \\usepackage\{multirow\}} \cr
#' \code{ - \\usepackage[dvipsnames,table]\{xcolor\}} \cr
#' \code{ - \\makeatletter} \cr
#' \code{ - \\newcommand*\\vdashline\{\\rotatebox[origin=c]\{90\}\{\$\\dabar@@\\dabar@@\\dabar@@\$\}\}} \cr
#' \code{ - \\makeatother}
#' 
# Remaining Documentation -------------------------------------------
#' @seealso 
#' \code{\link{sprinkle_colnames}} for changing column names in a table.
#' 
#' @source 
#' Altering the number of rows in a LaTeX longtable \cr
#' http://tex.stackexchange.com/questions/19710/how-can-i-set-the-maximum-number-of-rows-in-a-page-for-longtable
#' 
#' Vertical dashed cell borders in LaTeX table \cr
#' http://www.latex-community.org/forum/viewtopic.php?f=45&t=3149
#' 
#' Colored Cell border \cr
#' http://tex.stackexchange.com/questions/40666/how-to-change-line-color-in-tabular
#' 
#' @author Benjamin Nutter
#' 
#' @examples 
#' x <- dust(lm(mpg ~ qsec + factor(am), data = mtcars))
#' x %>% sprinkle(cols = 2:4, round = 3) %>% 
#'   sprinkle(cols = 5, fn = quote(pvalString(value))) %>% 
#'   sprinkle(rows = 2, bold = TRUE)
#'

#' @rdname sprinkle
#' @export

# sprinkle method ---------------------------------------------------
sprinkle <- function(x, rows = NULL, cols = NULL, ...,
                     part = c("body", "head", "foot", "interfoot", "table"))
{
  UseMethod("sprinkle")
}

# sprinkle.default --------------------------------------------------
#' @rdname sprinkle
#' @export

sprinkle.default <- function(x, rows = NULL, cols = NULL, ...,
                             part = c("body", "head", "foot", "interfoot", "table"),
                             fixed = FALSE, 
                             recycle = c("none", "rows", "cols", "columns"))
{
  
  sprinkles <- list(...)
  
# Argument validations ----------------------------------------------
  coll <- checkmate::makeAssertCollection()

  checkmate::assertClass(x = x,
                         classes = "dust",
                         add = coll)

  if (!length(sprinkles)){
    coll$push("At least one sprinkle must be declared in ...")
  }
  else
  {
    sprinkle_match <- 
      unlist(sprinkle_groups)[pmatch(names(sprinkles), 
                                     unlist(sprinkle_groups))]
    
    unmatched_sprinkle <- 
      names(sprinkles)[which(is.na(sprinkle_match))]
    
    names(sprinkles) <- sprinkle_match
    
    if (length(unmatched_sprinkle))
    {
      coll$push(
        sprintf("The following arguments could not be matched to an existing sprinkle (check spelling and partial matching): %s",
                paste(unmatched_sprinkle, collapse = ", "))
      )
      
      sprinkles <- sprinkles[!is.na(names(sprinkles))]
    }
  }
  
  if (!checkmate::test_named(sprinkles)){
    coll$push("Arguments to ... must be named")
  }
  
  not_sprinkles <- 
    sprinkles[!names(sprinkles) %in% unlist(sprinkle_groups)]
  
  if (length(not_sprinkles)){
    coll$push(sprintf("The following are not valid sprinkles: %s",
                      paste0(names(not_sprinkles), collapse = ", ")))
  }

  indices <- index_to_sprinkle(x = x, 
                               rows = rows, 
                               cols = cols, 
                               fixed = fixed,
                               part = part,
                               recycle = recycle,
                               coll = coll)
  
  recycle <- recycle[1]

  for (i in seq_along(sprinkle_groups))
  {
    if (any(sprinkle_groups[[i]] %in% names(sprinkles)))
    {
      sprinkle_arg <- c(sprinkles[sprinkle_groups[[i]]], "recycle" = recycle)
      sprinkle_arg <- sprinkle_arg[names(sprinkle_arg) %in% sprinkle_groups[[i]]]
      sprinkle_arg <- sprinkle_arg[!vapply(sprinkle_arg, is.null, logical(1))]
     
      if (!"fn" %in% names(sprinkle_arg))
      {
        args_list <- 
          if (names(sprinkle_groups)[i] == "replace")
          {
            list(indices = indices, 
                 coll = coll)
          }
          else
          {
            list(coll = coll)
          }

        do.call(what = sprintf("sprinkle_%s_index_assert",
                               names(sprinkle_groups)[[i]]),
                args = c(sprinkle_arg,
                         args_list))
      }
    }
  }
  
  if ("fn" %in% names(sprinkles))
  {
    sprinkle_fn_index_assert(fn = sprinkles$fn, 
                             coll = coll)
  }

  #* Return any errors found.
  checkmate::reportAssertions(coll)

# Functional Code ---------------------------------------------------

  part <- part[1]
  
  for (i in seq_along(sprinkle_groups))
  {
    if (any(sprinkle_groups[[i]] %in% names(sprinkles)))
    {
      sprinkle_arg <- sprinkles[sprinkle_groups[[i]]]
      sprinkle_arg <- sprinkle_arg[!vapply(sprinkle_arg, is.null, logical(1))]

      if (!"fn" %in% names(sprinkle_arg))
      {
        x <- do.call(what = sprintf("sprinkle_%s_index",
                                    names(sprinkle_groups)[[i]]),
                     args = c(sprinkle_arg,
                              list(x = x,
                                   indices = indices,
                                   part = part)))
      }
    }
  }

  if ("fn" %in% names(sprinkles))
  {
    x <- sprinkle_fn_index(x = x,
                           indices = indices,
                           fn = sprinkles$fn,
                           part = part)
  }
  
  x
}


# sprinkle.dust_list ------------------------------------------------
#' @rdname sprinkle
#' @export

sprinkle.dust_list <- function(x, rows = NULL, cols = NULL, ...,
                               part = c("body", "head", "foot", "interfoot", "table"))
{
  structure(
    lapply(X = x,
           FUN = sprinkle,
           rows = rows,
           cols = cols,
           part = part,
           ...),
    class = "dust_list"
  )
}

# Unexported --------------------------------------------------------

sprinkle_groups <-  
  list(
    align = c("halign", "valign", "recycle"),
    bg = c("bg", "recycle"),
    bg_pattern = c("bg_pattern", "bg_pattern_by"),
    bookdown = "bookdown",
    border = c("border", "border_color", "border_style", 
               "border_thickness", "border_units", "recycle"),
    border_collapse = "border_collapse",
    caption = "caption", 
    caption_number = "caption_number",
    discrete = c("discrete", "discrete_colors"),
    fixed_header = c("fixed_header", "include_fixed_header_css",
                     "fixed_header_class_name", 
                     "scroll_body_height", "scroll_body_height_units",
                     "scroll_body_background_color",
                     "fixed_header_height", "fixed_header_height_units",
                     "fixed_header_text_height", "fixed_header_text_height_units",
                     "fixed_header_background_color"),
    float = "float",
    fn = c("fn"),
    font = c("bold", "italic", "font_size", "font_size_units",
             "font_color", "font_family", "recycle"),
    gradient = c("gradient", "gradient_colors", "gradient_cut",
                 "gradient_n", "gradient_na"),
    height = c("height", "height_units", "recycle"),
    hhline = "hhline",
    justify = "justify",
    label = "label",
    longtable = "longtable",
    merge = c("merge", "merge_rowval", "merge_colval"),
    na_string = c("na_string", "recycle"),
    pad = c("pad", "recycle"),
    replace = "replace",
    rotate_degree = c("rotate_degree"),#, "recycle"),
    round = c("round"),#, "recycle"),
    sanitize = c("sanitize", "sanitize_args"),
    tabcolsep = "tabcolsep",
    width = c("width", "width_units", "recycle")
  )
