#' @name sprinkle_replace
#' @title Replace Contents of Selected Cells
#' 
#' @description At times it may be necessary to replace the contents of a 
#' cell with user-supplied values.
#'   
#' @param x An object of class \code{dust}
#' @param rows Either a numeric vector of rows in the tabular object to be 
#'   modified or an object of class \code{call}.  When a \code{call}, 
#'   generated by \code{quote(expression)}, the expression resolves to 
#'   a logical vector the same length as the number of rows in the table.
#'   Sprinkles are applied to where the expression resolves to \code{TRUE}.
#' @param cols Either a numeric vector of columns in the tabular object to
#'   be modified, or a character vector of column names. A mixture of 
#'   character and numeric indices is permissible.
#' @param replace \code{character} A character vector giving the desired
#'   content for the selected cells.
#' @param part A character string denoting which part of the table to modify.
#' @param fixed \code{logical(1)} indicating if the values in \code{rows} 
#'   and \code{cols} should be read as fixed coordinate pairs.  By default, 
#'   sprinkles are applied at the intersection of \code{rows} and \code{cols}, 
#'   meaning that the arguments do not have to share the same length.  
#'   When \code{fixed = TRUE}, they must share the same length.
#' @param recycle A \code{character} one that determines how sprinkles are 
#'   managed when the sprinkle input doesn't match the length of the region
#'   to be sprinkled.  By default, recycling is turned off.  Recycling 
#'   may be performed across rows first (left to right, top to bottom), 
#'   or down columns first (top to bottom, left to right).
#' @param ... Additional arguments to pass to other methods. Currently ignored.
#' 
#' @section Functional Requirements:
#' \enumerate{
#'  \item Correctly reassigns the appropriate elements \code{replace} column
#'    in the table part.
#'  \item Casts an error if \code{x} is not a \code{dust} object.
#'  \item Casts an error if \code{replace} is not a vector
#'  \item Casts an warning if the number of indices to replace is not a 
#'    multiple of \code{replace} 
#'  \item Casts an error if \code{length(replace)} is greater than the 
#'    number of cells to replace.
#'  \item Casts an error if \code{part} is not one of \code{"body"}, 
#'    \code{"head"}, \code{"foot"}, or \code{"interfoot"}
#'  \item Casts an error if \code{fixed} is not a \code{logical(1)}
#'  \item Casts an error if \code{recycle} is not one of \code{"none"},
#'    \code{"rows"}, or \code{"cols"}
#' }
#' 
#' The functional behavior of the \code{fixed} and \code{recycle} arguments 
#' is not tested for this function. It is tested and validated in the
#' tests for \code{\link{index_to_sprinkle}}.
#' 
#' @seealso \code{\link{sprinkle}}, 
#'   \code{\link{index_to_sprinkle}}
#'
#' @export

sprinkle_replace <- function(x, rows = NULL, cols = NULL,
                               replace, 
                               part = c("body", "head", "foot", "interfoot"),
                               fixed = FALSE, 
                               recycle = c("none", "rows", "cols", "columns"),
                               ...)
{
  UseMethod("sprinkle_replace")
}

#' @rdname sprinkle_replace
#' @export

sprinkle_replace.default <- function(x, rows = NULL, cols = NULL,
                                       replace,
                                       part = c("body", "head", "foot", "interfoot"),
                                       fixed = FALSE, 
                                       recycle = c("none", "rows", "cols", "columns"),
                                       ...)
{
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = x,
                          classes = "dust",
                          add = coll)
  
  indices <- index_to_sprinkle(x = x, 
                               rows = rows, 
                               cols = cols, 
                               fixed = fixed,
                               part = part,
                               recycle = recycle,
                               coll = coll)
  
  sprinkle_replace_index_assert(replace = replace, 
                                indices = indices,
                                coll = coll)
  
  checkmate::reportAssertions(coll)
  
  sprinkle_replace_index(x = x, 
                         indices = indices, 
                         replace = replace, 
                         part = part)
}

#' @rdname sprinkle_replace
#' @export

sprinkle_replace.dust_list <- function(x, rows = NULL, cols = NULL,
                                         replace,
                                         part = c("body", "head", "foot", "interfoot"),
                                         fixed = FALSE, 
                                         recycle = c("none", "rows", "cols", "columns"),
                                         ...)
{
  structure(
    lapply(X = x,
           FUN = sprinkle_replace.default,
           rows = rows,
           cols = cols,
           replace = replace,
           part = part,
           fixed = fixed,
           recycle = recycle,
           ...),
    class = "dust_list"
  )
}

# Unexported Utility ------------------------------------------------

# These functions are to be used inside of the general `sprinkle` call
# When used inside `sprinkle`, the indices are already determined, 
# the only the `replace` argument needs to be validated. 
# The assert function is kept separate so it may be called earlier
# without attempting to perform the assignment.

sprinkle_replace_index_assert <- function(replace, indices, coll)
{
  checkmate::assert_atomic_vector(x = replace,
                                  max.len = sum(indices),
                                  add = coll,
                                  .var.name = "replace")
}

sprinkle_replace_index <- function(x, indices, replace, part)
{
  if (length(replace) != 1 & (sum(indices) %% length(replace)) != 0)
  {
    warning("The number of cells to edit is not a multiple of ",
            "length(replace). Values may not recycle as expected.")
    
    replace <- rep(replace, length.out = sum(indices))
  }
  
  # At this point, part should have passed the assertions in 
  # index_to_sprinkle. The first element is expected to be valid.
  
  part <- part[1]
  
  x[[part]][["replace"]][indices] <- replace
  
  x
}