#' @name sprinkle_align
#' @title Sprinkle Appearance of NA's
#' 
#' @description The alignment refers to the positioning of the text within
#'   a cell.  Alignment may be given relative to the left, center, or right
#'   of a cell, and the top, middle, or bottom of the cell.
#'   
#' @param x An object of class \code{dust}
#' @param rows Either a numeric vector of rows in the tabular object to be 
#'   modified or an object of class \code{call}.  When a \code{call}, 
#'   generated by \code{quote(expression)}, the expression resolves to 
#'   a logical vector the same length as the number of rows in the table.
#'   Sprinkles are applied to where the expression resolves to \code{TRUE}.
#' @param cols Either a numeric vector of columns in the tabular object to
#'   be modified, or a character vector of column names. A mixture of 
#'   character and numeric indices is permissible.
#' @param halign \code{character} One of \code{"left"}, \code{"center"}, 
#'   or \code{"right"}. Defaults to \code{NULL}, for no change to the 
#'   current value.
#' @param valign \code{character} One of \code{"top"}, \code{"middle"}, 
#'   or \code{"bottom"}. Defaults to \code{NULL}, for no change to the 
#'   current value.
#' @param part A character string denoting which part of the table to modify.
#' @param fixed \code{logical(1)} indicating if the values in \code{rows} 
#'   and \code{cols} should be read as fixed coordinate pairs.  By default, 
#'   sprinkles are applied at the intersection of \code{rows} and \code{cols}, 
#'   meaning that the arguments do not have to share the same length.  
#'   When \code{fixed = TRUE}, they must share the same length.
#' @param recycle A \code{character} one that determines how sprinkles are 
#'   managed when the sprinkle input doesn't match the length of the region
#'   to be sprinkled.  By default, recycling is turned off.  Recycling 
#'   may be performed across rows first (left to right, top to bottom), 
#'   or down columns first (top to bottom, left to right).
#' @param ... Additional arguments to pass to other methods. Currently ignored.
#' 
#' @section Functional Requirements:
#' \enumerate{
#'  \item Correctly reassigns the appropriate elements of \code{halign} 
#'    and \code{valign} columns in the table part.
#'  \item Casts an error if \code{x} is not a \code{dust} object.
#'  \item Casts an error if \code{bg} is not a \code{character(1)}
#'  \item Casts an error if \code{part} is not one of \code{"body"}, 
#'    \code{"head"}, \code{"foot"}, or \code{"interfoot"}
#'  \item Casts an error if \code{fixed} is not a \code{logical(1)}
#'  \item Casts an error if \code{recycle} is not one of \code{"none"},
#'    \code{"rows"}, or \code{"cols"}
#' }
#' 
#' The functional behavior of the \code{fixed} and \code{recycle} arguments 
#' is not tested for this function. It is tested and validated in the
#' tests for \code{\link{index_to_sprinkle}}.
#' 
#' @seealso \code{\link{sprinkle}}, 
#'   \code{\link{index_to_sprinkle}}
#'
#' @export

sprinkle_align <- function(x, rows = NULL, cols = NULL, 
                           halign = NULL, valign = NULL,
                           part = c("body", "head", "foot", "interfoot"),
                           fixed = FALSE, 
                           recycle = c("none", "rows", "cols", "columns"),
                           ...)
{
  UseMethod("sprinkle_align")
}

#' @rdname sprinkle_align
#' @export

sprinkle_align.default <- function(x, rows = NULL, cols = NULL, 
                                   halign = NULL, valign = NULL,
                                   part = c("body", "head", "foot", "interfoot"),
                                   fixed = FALSE, 
                                   recycle = c("none", "rows", "cols", "columns"),
                                   ...)
{
  coll <- checkmate::makeAssertCollection()
  
  if (!is.null(halign))
  {
    halign <- checkmate::matchArg(x = halign,
                                  choices = c("left", "center", "right"),
                                  add = coll)
  }
  
  if (!is.null(valign))
  {
    valign <- checkmate::matchArg(x = valign,
                                  choices = c("top", "middle", "bottom"),
                                  add = coll)
  }
  
  indices <- index_to_sprinkle(x = x, 
                               rows = rows, 
                               cols = cols, 
                               fixed = fixed,
                               part = part,
                               recycle = recycle,
                               coll = coll)
  
  checkmate::reportAssertions(coll)
  
  if (is.null(halign) & is.null(valign))
  {
    return(x)
  }
  
  # At this point, part should have passed the assertions in 
  # index_to_sprinkle. The first element is expected to be valid.
  
  part <- part[1]
  
  if (!is.null(halign))
  {
    x[[part]][["halign"]][indices] <- halign
  }
  
  if (!is.null(valign))
  {
    x[[part]][["valign"]][indices] <- valign
  }
  
  x
}

#' @rdname sprinkle_align
#' @export

sprinkle_align.dust_list <- function(x, rows = NULL, cols = NULL,
                                     halign = NULL, valign = NULL, 
                                     part = c("body", "head", "foot", "interfoot"),
                                     fixed = FALSE, 
                                     recycle = c("none", "rows", "cols", "columns"),
                                     ...)
{
  structure(
    lapply(X = x,
           FUN = sprinkle_align.default,
           rows = rows,
           cols = cols,
           halign = halign,
           valign = valign,
           part = part,
           fixed = fixed,
           recycle = recycle,
           ...),
    class = "dust_list"
  )
}

# Unexported Utility ------------------------------------------------

# These functions are to be used inside of the general `sprinkle` call
# When used inside `sprinkle`, the indices are already determined, 
# the only the `halign` and `valign` arguments needs to be validated. 
# The assert function is kept separate so it may be called earlier
# without attempting to perform the assignment.

sprinkle_align_index_assert <- function(halign, valign, coll)
{
  if (!is.null(halign))
  {
    halign <- checkmate::matchArg(x = halign,
                                  choices = c("left", "center", "right"),
                                  add = coll,
                                  .var.name = "halign")
  }
  
  if (!is.null(valign))
  {
    valign <- checkmate::matchArg(x = valign,
                                  choices = c("top", "middle", "bottom"),
                                  add = coll,
                                  .var.name = "valign")
  }
}

sprinkle_align_index <- function(x, indices, halign, valign, part)
{
  if (!is.null(halign))
  {
    x[[part]][["halign"]][indices] <- halign
  }
  
  if (!is.null(valign))
  {
    x[[part]][["valign"]][indices] <- valign
  }
  
  x
}