% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/polynomial.R
\name{pdPolynomial}
\alias{pdPolynomial}
\title{Generate intrinsic HPD polynomial curves}
\usage{
pdPolynomial(p0, v0, delta.t = 0.01, steps = 100)
}
\arguments{
\item{p0}{a \eqn{(d, d)}-dimensional HPD matrix specifying the starting point of the polynomial curve.}

\item{v0}{a \eqn{(d, d, k)}-dimensional array corresponding to a sequence of covariant derivatives of
order zero up to order \eqn{k - 1} at the starting point \code{p0}.}

\item{delta.t}{a numeric value determining the incrementing step size in the numerical integration procedure.
A smaller step size results in a higher resolution and therefore a more accurate approximation of the polynomial curve,
defaults to \code{delta.t = 0.01}.}

\item{steps}{number of incrementing steps in the numerical integration procedure, defaults to \code{steps = 100}.}
}
\value{
A \code{(d, d, length(steps))}-dimensional array containing the approximated intrinsic polynomial
curve of degree \eqn{k} passing through \code{p0} with the given covariant derivatives up to order
\eqn{k - 1} and vanishing higher order covariant derivatives.
}
\description{
\code{pdPolynomial()} generates intrinsic polynomial curves on the manifold of HPD matrices
equipped with the Riemannian metric according to the numerical integration procedure described in (Hinkle et al., 2014).
Given an initial starting point \code{p0} (i.e. HPD matrix) on the Riemannian manifold and the covariant
derivatives up to order \eqn{k - 1} at \code{p0}, \code{pdPolynomial()} approximates the uniquely existing
intrinsic polynomial curve of degree \eqn{k} passing through \code{p0} with the given covariant derivatives up
to order \eqn{k - 1} and vanishing higher order covariant derivatives.
}
\examples{
## First-order polynomial
p0 <- diag(3) ## HPD starting point
v0 <- array(H.coeff(rnorm(9), inverse = TRUE), dim = c(3, 3, 1)) ## zero-th order cov. derivative
P.poly <- pdPolynomial(p0, v0)

## First-order polynomials coincide with geodesic curves
geo <- function(A, B, t) Expm(A, t * Logm(A, B))
P.geo <- sapply(seq(0, 1, length = 100), function(t) geo(p0, P.poly[, , 100], t),
                simplify = "array")
all.equal(P.poly, P.geo)

}
\references{
Hinkle, J., Fletcher, P. and Joshi, S. (2014). Intrinsic polynomials for regression on Riemannian manifolds.
\emph{Journal of Mathematical Imaging and Vision} 50, 32-52.
}
\seealso{
\code{\link{pdNeville}}
}
