% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generate.R
\name{generateItem}
\alias{generateItem}
\title{Generate paired comparison data for one or more items given
absolute latent scores}
\usage{
generateItem(df, theta, th = 0.5, name, ..., scale = 1, alpha = 1)
}
\arguments{
\item{df}{a data frame with pairs of vertices given in columns \code{pa1} and \code{pa2}, and item response data in other columns}

\item{theta}{a vector or matrix of absolute latent scores. See details below.}

\item{th}{a vector of thresholds}

\item{name}{a vector of item names}

\item{...}{Not used.  Forces remaining arguments to be specified by name.}

\item{scale}{a vector of scaling constants}

\item{alpha}{a vector of item discriminations}
}
\value{
The given data.frame \code{df} with additional columns for each item.
}
\description{
To add a single item, \code{theta} should be a vector of latent
scores. To add multiple items at a time, \code{theta} should be a
matrix with one item in each column. Item names can be given as
the colnames of \code{theta}.

The interpretation of \code{theta} depends on the context where the
data were generated. For example, in chess, \code{theta} represents
unobserved chess skill that is partially revealed by match
outcomes.

The graph can be regarded as undirected, but data are generated
relative to the order of vertices within each row. Vertices do not
commute. For example, a \code{-1} for vertices \sQuote{a} and
\sQuote{b} is the same as \code{1} for vertices \sQuote{b} and
\sQuote{a}.
}
\section{Response model}{


The paired comparison item response model has thresholds and a
scale parameter similar to the partial credit model (Masters,
1982). The model is cumbersome to describe in traditional
mathematical notation, but the R code is fairly straightforward,

\preformatted{
softmax <- function(y) exp(y) / sum(exp(y))

cmp_probs <- function(scale, alpha, pa1, pa2, thRaw) {
  th <- cumsum(thRaw)
  diff <- scale * (pa2 - pa1)
  unsummed <- c(0, diff + rev(th), diff - th, use.names = FALSE)
  softmax(cumsum(alpha * unsummed))
}
}

The function \code{cmp_probs} takes a \code{scale} constant,
\code{alpha} discrimination, the latent scores for two objects
\code{pa1} and \code{pa2}, and a vector of thresholds
\code{thRaw}. The thresholds are parameterized as the difference
from the previous threshold. For example, thresholds c(0.5, 0.5)
are not at the same location but are at locations c(0.5,
1.0). Thresholds are symmetric. If there is one threshold then the
model admits three possible response outcomes (e.g. \emph{win}, \emph{tie}, and
\emph{lose}). Responses are always stored centered with zero representing
a tie. Therefore, it is necessary to add one plus the number of
thresholds to response data to index into the vector returned by
\code{cmp_probs}. For example, if our response data is (-1, 0, 1)
and has one threshold then we would add 2 (1 + 1 threshold) to
obtain the indices (1, 2, 3).

Use \code{\link{itemModelExplorer}} to explore the item model. In
this \pkg{shiny} app, the \emph{discrimination} parameter does what
is customary in item response models. However, it is not difficult
to show that discrimination is a function of thresholds and
scale. That is, discrimination is not an independent parameter.  In
paired comparison models, discrimination and measurement error are
confounded.
}

\examples{
df <- roundRobinGraph(letters[1:5], 40)
df <- generateItem(df)
}
\references{
Masters, G. N. (1982). A Rasch model for partial credit scoring.
Psychometrika, 47, 149–174. doi: 10.1007/BF02296272
}
\seealso{
Other item generators: 
\code{\link{generateCovItems}()},
\code{\link{generateFactorItems}()},
\code{\link{generateSingleFactorItems}()}
}
\concept{item generators}
