#' Plot spectra
#'
#' Plots reflectance spectra in different arrangements.
#'
#' @param x (required) a data frame, possibly an object of class \code{rspec},
#' with a column with wavelength data, named 'wl', and the remaining column containing
#' spectra to plot.
#' @param select specification of which spectra to plot. Can be a numeric vector or
#' factor (e.g., \code{sex=='male'})
#' @param type what type of plot should be drawn. Possibilities are:
#' \itemize{
#'  \item \code{overlay} (default) for plotting multiple spectra in a single panel with
#' a common y-axis.
#'  \item \code{stack} for plotting multiple spectra in a vertical arrangement.
#'  \item \code{heatmap} for plotting reflectance values by wavelength and a third variable
#'        (\code{varying}).
#' }
#' @param varying a numeric vector giving values for y-axis in \code{heatplot}.
#' @param n number of bins with which to interpolate colors and \code{varying} for the
#' heatplot.
#' @param labels.stack a vector of labels for the stacked spectra when using \code{type = stack}.
#' Defaults to the numeric column ID's.
#' @param ... additional arguments passed to plot (or image for \code{'heatmap'}).
#'
#' @export
#'
#' @examples
#' \dontrun{
#' 
#' data(teal)
#' plot(teal, type = "overlay")
#' plot(teal, type = "stack")
#' plot(teal, type = "heatmap")
#' }
#' 
#' @author Chad Eliason \email{cme16@@zips.uakron.edu}
#' @author Thomas White \email{thomas.white026@@gmail.com}
#'
#' @seealso \code{\link{spec2rgb}}, \code{\link{image}}, \code{\link{plot}}

# TODO: add argument for padding region between x in stack plot

plot.rspec <- function(x, select = NULL, type = c("overlay", "stack", "heatmap"),
                       varying = NULL, n = 100, labels.stack = NULL, ...) {
  type <- match.arg(type)

  # make wavelength vector
  wl_index <- which(names(x) == "wl")
  if (length(wl_index) > 0) {
    haswl <- TRUE
    wl <- x[, wl_index]
  } else {
    haswl <- FALSE
    wl <- seq_len(nrow(x))
    warning("No wavelengths provided; using arbitrary index values")
  }

  # subset based on indexing vector
  if (is.logical(select)) {
    select <- which(select == "TRUE")
  }
  if (is.null(select) & haswl == TRUE) {
    select <- (seq_len(ncol(x)))[-wl_index]
  }
  if (is.null(select) & haswl == FALSE) {
    select <- seq_len(ncol(x))
  }

  x <- as.data.frame(x[select]) # CE: removed comma before select

  arg <- list(...)

  # Set defaults
  if (is.null(arg$xlab)) {
    arg$xlab <- "Wavelength (nm)"
  }
  if (is.null(arg$xlim)) {
    arg$xlim <- range(wl, na.rm = TRUE)
  }

  # heat plot
  if (type == "heatmap") {
    if (is.null(arg$xlab)) {
      arg$xlab <- "Wavelength (nm)"
    }
    if (is.null(arg$ylab)) {
      arg$ylab <- "Index"
    }
    if (is.null(varying)) {
      varying <- seq_len(ncol(x))
      print("No varying vector supplied; using arbitrary values")
    }
    if (is.null(arg$ylim)) {
      arg$ylim <- range(varying, na.rm = TRUE)
    }
    if (is.null(arg$col)) {
      arg$col <- viridisLite::cividis(n)
    } else {
      jc <- colorRampPalette(arg$col)
      arg$col <- jc(n)
    }

    if (is.null(arg$lty)) {
      arg$lty <- 1
    }

    Index <- approx(varying, n = n)$y

    dat <- vapply(seq_len(nrow(x)), function(z) {
      approx(
        x = varying,
        y = x[z, ],
        n = n
      )$y
    }, numeric(length(Index)))

    arg$x <- wl
    arg$y <- Index
    arg$z <- t(dat)

    do.call(image, arg)
  }

  # coloring for overlay plot & others
  if (length(arg$col) < ncol(x)) {
    arg$col <- rep(arg$col, ncol(x))
    arg$col <- arg$col[seq_len(ncol(x))]
  }

  if (any(names(arg$col) %in% names(x))) {
    arg$col <- arg$col[select - 1]
  }

  # line types for overlay plot & others
  if (length(arg$lty) < ncol(x)) {
    arg$lty <- rep(arg$lty, ncol(x))
    arg$lty <- arg$lty[seq_len(ncol(x))]
  }

  if (any(names(arg$lty) %in% names(x))) {
    arg$col <- arg$lty[select - 1]
  }


  # overlay different spec curves
  if (type == "overlay") {
    if (is.null(arg$ylim)) {
      arg$ylim <- range(x, na.rm = TRUE)
    }
    if (is.null(arg$ylab)) {
      arg$ylab <- "Reflectance (%)"
    }
    arg$type <- "l"
    arg$x <- wl
    arg$y <- x[, 1]
    col <- arg$col
    arg$col <- col[1]
    lty <- arg$lty
    arg$lty <- lty[1]


    do.call(plot, arg)

    if (ncol(x) > 1) {
      for (i in 2:ncol(x)) {
        arg$col <- col[i]
        arg$lty <- lty[i]
        arg$y <- x[, i]
        do.call(lines, arg)
      }
    }
  }

  # Stack curves along y-axis
  if (type == "stack") {
    arg$type <- "l"
    if (is.null(arg$ylab)) {
      arg$ylab <- "Cumulative reflectance (arb. units)"
    }

    x2 <- as.data.frame(x[, c(rev(seq_len(ncol(x))))])
    if (length(select) == 1) {
      y <- max(x2)
    } else {
      y <- apply(x2, 2, max)
    }
    ym <- cumsum(y)
    ymins <- c(0, ym[-length(ym)])

    arg$x <- wl
    arg$y <- x2[, 1]
    if (is.null(arg$ylim)) {
      arg$ylim <- c(0, sum(y))
    }

    col <- rev(arg$col)
    arg$col <- col[1]
    do.call(plot, arg)
    if (ncol(x2) > 1) {
      for (i in 2:ncol(x2)) {
        arg$y <- x2[, i] + ymins[i]
        arg$col <- col[i]
        do.call(lines, arg)
      }
    }

    yend <- tail(x2, 1)
    yloc <- ymins + yend

    if (is.null(labels.stack)) {
      labels.stack <- rev(select)
    }

    axis(side = 4, at = yloc, labels = labels.stack, las = 1)
  }
}
